﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <nn/nn_Result.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/util/util_FormatString.h>
#include "../DevMenu_Config.h"
#include "../DevMenu_RootSurface.h"

namespace devmenu { namespace system { namespace download {

class ProgressViewUnit
{
public:
    ProgressViewUnit(
        nn::ns::ApplicationView view,
        glv::space_t yPos,
        glv::space_t width,
        glv::space_t height,
        glv::space_t idPos,
        glv::space_t statePos,
        glv::space_t downloadedPos,
        glv::space_t sliderPos,
        glv::space_t sliderLength,
        glv::space_t buttonPos,
        glv::space_t buttonLength,
        std::function< void( const nn::ncm::ApplicationId& id )> callback
    ) NN_NOEXCEPT;

    /**
    * @brief 描画オブジェクトを破棄する
    */
    ~ProgressViewUnit();

    /**
    * @brief アプリケーションのダウンロード要求をセットする
    */
    void SetDownloadRequired() NN_NOEXCEPT
    {
        m_DownloadButtonCallback( m_View.id );
        m_IsDownloadRequired = true;
    };

    /**
    * @brief ダウンロード要求を判定する
    */
    bool IsDownloadRequired() const NN_NOEXCEPT { return m_IsDownloadRequired; };

    /**
    * @brief 描画オブジェクトを更新する
    */
    void UpdateProgress(const nn::ns::ApplicationView& view) NN_NOEXCEPT;

    // getter
    nn::ns::ApplicationView GetView() const NN_NOEXCEPT { return m_View; };
    glv::Label* GetIdLabel() const NN_NOEXCEPT { return m_IdLabel; };
    glv::Label* GetStateLabel() const NN_NOEXCEPT { return m_StateLabel.get(); };
    glv::Label* GetProgressLabel() const NN_NOEXCEPT { return m_ProgressLabel.get(); };
    glv::Slider* GetProgressSlider() const NN_NOEXCEPT { return m_ProgressSlider; };
    glv::Button* GetButton() const NN_NOEXCEPT { return m_Button; };
    glv::Table* GetApplicationTable() const NN_NOEXCEPT { return m_ApplicationTable.get(); };

private:
    static const int PosSetSize = 6;

    /**
    * @brief 描画オブジェクトの消去
    */
    template<class T> void DeleteViewObject(T** pViewObject) NN_NOEXCEPT;

    /**
    * @brief ダウンロード状況を判定する
    */
    const char* DecisionDownloadState(nn::ns::ApplicationView view);

    //!< 描画オブジェクトの位置セット
    struct PositionSet
    {
        glv::space_t y;
        glv::space_t id;
        glv::space_t state;
        glv::space_t progressLabel;
        glv::space_t progressSlider;
        glv::space_t button;
    };

    nn::ns::ApplicationView m_View; //!< 描画する ApplicationView

    //!< ページ生成時に描画するオブジェクト
    glv::Label* m_IdLabel;          //!< アプリケーションの Id を表示するラベル
    glv::Slider* m_ProgressSlider;  //!< ダウンロード進捗を表示するためのタスクバー
    glv::Button* m_Button;          //!< アプリケーションを削除するボタン
    glv::Label* m_ButtonLabel;      //!< 削除ボタンのラベル

    //!< 毎フレーム描画するオブジェクト
    std::unique_ptr<glv::Label> m_StateLabel;       //!< ダウンロード状態を表示するためのラベル
    std::unique_ptr<glv::Label> m_ProgressLabel;    //!< ダウンロード容量を表示するラベル
    std::unique_ptr<glv::Table> m_ApplicationTable; //!< スクロールコンテナの1行に表示する情報

    bool m_IsDownloadRequired;      //!< ダウンロード要求のフラグ
    PositionSet m_Position;         //!< 各オブジェクトの位置

    // Downloadボタン押下時のコールバック関数
    std::function< void( const nn::ncm::ApplicationId& id ) > m_DownloadButtonCallback; //!< Downloadボタンを押下された時に実行するコールバック関数
};

}}} // ~namespace devmenu::system::download, ~namespace devmenu::system, ~namespace devmenu
