﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <atomic>
#include <thread>
#include <vector>
#include <nn/nifm.h>
#include <nn/nn_Abort.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>

#include "DevMenu_Config.h"
#include "DevMenu_Common.h"
#include "DevMenu_RootSurface.h"
#undef GetCurrentTime

#define GT GLV_TEXT_API_WIDE_STRING

namespace devmenu { namespace system { namespace update {

template<typename T>
glv::WideString ToWideString(T data) NN_NOEXCEPT
{
    glv::WideString ws;
    return glv::CopyWithConvertWideString(ws, std::to_string(data).c_str());
}

glv::WideString ToWideString(std::string str) NN_NOEXCEPT
{
    glv::WideString ws;
    return glv::CopyWithConvertWideString(ws, str.c_str());
}

const char* ToString(nn::ns::LatestSystemUpdate latest) NN_NOEXCEPT
{
    switch (latest)
    {
    case nn::ns::LatestSystemUpdate::Downloaded: return "Downloaded";
    case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
    case nn::ns::LatestSystemUpdate::UpToDate: return "UpToDate";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

std::string ToMilliSecondsString(nn::TimeSpan time) NN_NOEXCEPT
{
    return std::string(" (") + std::to_string(time.GetMilliSeconds()) + " ms)";
}


std::string ToString(nn::Result result) NN_NOEXCEPT
{
    char buffer[sizeof("Failure 0x01234567")];
    nn::util::SNPrintf(buffer, sizeof(buffer), "Failure 0x%08x", result.GetInnerValueForDebug());
    return buffer;
}

const char* GetLoadingString() NN_NOEXCEPT
{
    nn::TimeSpan current = nn::os::GetSystemTick().ToTimeSpan();

    switch (current.GetSeconds() % 3)
    {
    case 0: return ".  ";
    case 1: return ".. ";
    case 2: return "...";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

class TitleLabel : public glv::Group
{
public:
    explicit TitleLabel(glv::space_t w, const glv::WideString& titleText, const glv::WideString& valueText) NN_NOEXCEPT
    {
        auto titleLabel = new glv::Label(titleText, glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
        auto valueLabel = new glv::Label(valueText, glv::Label::Spec(glv::Place::TR, 0, 0, CommonValue::InitialFontSize));
        set(glv::Rect(w, titleLabel->h));
        *this << titleLabel << valueLabel;

        m_ValueLabel = valueLabel;
    }

    void UpdateValue(const glv::WideString& value) NN_NOEXCEPT
    {
        m_ValueLabel->remove();
        delete m_ValueLabel;

        auto label = new glv::Label(value, glv::Label::Spec(glv::Place::TR, 0, 0, CommonValue::InitialFontSize));
        *this << label;

        m_ValueLabel = label;
    }

private:
    glv::Label* m_ValueLabel;
};

class BackgroundStateIndicator : public glv::Table
{
public:
    explicit BackgroundStateIndicator(glv::space_t w) NN_NOEXCEPT : m_Timer(nn::os::EventClearMode_ManualClear)
    {
        auto label = new TitleLabel(w, GT("Background network update state"), GT("None"));

        *this << label;
        arrange().fit(false);

        m_StateLabel = label;

        m_Timer.StartPeriodic(nn::TimeSpan::FromSeconds(0), nn::TimeSpan::FromSeconds(1));
    }

    void Update() NN_NOEXCEPT
    {
        if (m_Timer.TryWait())
        {
            m_Timer.Clear();
            m_StateLabel->UpdateValue(GetStateString(nn::ns::GetBackgroundNetworkUpdateState()));
        }
    }

private:
    glv::WideString GetStateString(nn::ns::BackgroundNetworkUpdateState state) NN_NOEXCEPT
    {
        switch (state)
        {
        case nn::ns::BackgroundNetworkUpdateState::None: return GT("None");
        case nn::ns::BackgroundNetworkUpdateState::InProgress: return GT("InProgress");
        case nn::ns::BackgroundNetworkUpdateState::Ready: return GT("Ready");
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::os::TimerEvent m_Timer;
    TitleLabel* m_StateLabel;
};


class RequestAndResultPanel : public glv::Group
{
public:
    explicit RequestAndResultPanel(glv::space_t w, const glv::WideString& title) NN_NOEXCEPT
        : m_Title(title), m_HasNewRequest()
    {
        m_pResultLabel = new glv::Label(GT(""), glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize));
        m_pSpacer = new Spacer(0.0f, 0.0f);
        m_pButton = new Button(GT("Request ") + title, [&]{ RequestOrCancel(); }, glv::Rect(w / 2, m_pResultLabel->h + 10.0f));
        m_pTable = new glv::Table("<x>");

        set(glv::Rect(w, m_pButton->h));

        *m_pTable << m_pButton << m_pSpacer << m_pResultLabel;
        m_pTable->arrange().fit();
        *this << m_pTable;
    }

    bool Update() NN_NOEXCEPT
    {
        bool requiresUpdateAll = false;
        if (m_HasNewRequest)
        {
            if (!IsOnRequest())
            {
                JoinRequest();

                UpdateResult(ToWideString(GetResultString()));
                m_pButton->UpdateLabelText(GT("Request ") + m_Title);
                m_HasNewRequest = false;
                requiresUpdateAll = true;
            }
            else
            {
                auto label = std::string("Wating") + GetLoadingString();
                UpdateResult(ToWideString(label));
                m_pButton->UpdateLabelText(GT("Cancel ") + m_Title);
            }
        }

        return requiresUpdateAll;
    }

    void ClearResult() NN_NOEXCEPT
    {
        UpdateResult(GT(""));
    }

private:
    void RequestOrCancel() NN_NOEXCEPT
    {
        if (!IsOnRequest())
        {
            m_HasNewRequest = true;
            Request();
        }
        else if(m_HasNewRequest)
        {
            Cancel();
        }
    }

    void UpdateResult(const glv::WideString& result) NN_NOEXCEPT
    {
        m_pResultLabel->setValue(result);
        const auto spacerWidth = w - m_pButton->w - m_pResultLabel->w;
        NN_ASSERT(spacerWidth > 0.0f);
        m_pSpacer->set(glv::Rect(spacerWidth, 0.0f));

        m_pSpacer->remove();
        m_pResultLabel->remove();
        *m_pTable << m_pSpacer << m_pResultLabel;
        m_pTable->arrange().fit(false);
    }

    virtual bool IsOnRequest() NN_NOEXCEPT = 0;
    virtual void Request() NN_NOEXCEPT = 0;
    virtual void Cancel() NN_NOEXCEPT = 0;
    virtual void JoinRequest() NN_NOEXCEPT = 0;
    virtual std::string GetResultString() NN_NOEXCEPT = 0;

    glv::Table*      m_pTable;
    Button*          m_pButton;
    Spacer*          m_pSpacer;
    glv::Label*      m_pResultLabel;
    glv::WideString  m_Title;

    bool m_HasNewRequest;
};

class ConnectNetworkPanel : public RequestAndResultPanel
{
public:
    explicit ConnectNetworkPanel(glv::space_t w) NN_NOEXCEPT : RequestAndResultPanel(w, GT("connect network"))
    {
        m_IsOnRequest = false;
    }

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsOnRequest;
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        m_IsOnRequest = true;

        m_Thread = std::thread([this] {
            nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            nn::nifm::SubmitNetworkRequestAndWait();
            m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            m_LastResult = nn::nifm::IsNetworkAvailable();
            m_IsOnRequest = false;
        });
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::nifm::CancelNetworkRequest();
    }

    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Thread.join();
    }

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        std::string label = m_LastResult ? "Success" : "Failure";
        return label + ToMilliSecondsString(m_LastTime);
    }

    std::thread m_Thread;
    std::atomic_bool m_IsOnRequest;
    bool m_LastResult;
    nn::TimeSpan m_LastTime;
};

class CheckLatestUpdatePanel : public RequestAndResultPanel
{
public:
    CheckLatestUpdatePanel(nn::ns::SystemUpdateControl* control, glv::space_t w) NN_NOEXCEPT
        : RequestAndResultPanel(w, GT("check latest update")), m_Control(control) {}

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Latest && !m_Latest->TryWait();
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Latest.emplace();
        auto result = m_Control->RequestCheckLatestUpdate(&(*m_Latest));
        if (result.IsFailure())
        {
            m_Latest = nn::util::nullopt;
            m_LastResult = result;
            m_LastTime = 0;
        }
        else
        {
            m_Thread = std::thread([this] {
                nn::ns::LatestSystemUpdate latest;
                nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
                m_LastResult = m_Latest->Get(&latest);
                m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
                m_LastLatest = latest;
            });
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Latest)
        {
            m_Latest->Cancel();
        }
    }

    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Latest)
        {
            m_Thread.join();
            m_Latest = nn::util::nullopt;
        }
    }

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_LastResult.IsSuccess())
        {
            return ToString(m_LastLatest) + ToMilliSecondsString(m_LastTime);
        }
        else
        {
            return ToString(m_LastResult) + ToMilliSecondsString(m_LastTime);
        }
    }

    nn::ns::SystemUpdateControl* m_Control;
    std::thread m_Thread;
    nn::util::optional<nn::ns::AsyncLatestSystemUpdate> m_Latest;
    nn::Result m_LastResult;
    nn::ns::LatestSystemUpdate m_LastLatest;
    nn::TimeSpan m_LastTime;
};

class DownloadLatestUpdatePanel : public RequestAndResultPanel
{
public:
    DownloadLatestUpdatePanel(nn::ns::SystemUpdateControl* control, glv::space_t w) NN_NOEXCEPT
        : RequestAndResultPanel(w, GT("download latest update")), m_Control(control) {}

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Async && !m_Async->TryWait();
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Async.emplace();
        auto result = m_Control->RequestDownloadLatestUpdate(&(*m_Async));
        if (result.IsFailure())
        {
            m_Async = nn::util::nullopt;
            m_LastResult = result;
            m_LastTime = 0;
        }
        else
        {
            m_Thread = std::thread([this] {
                nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
                m_LastResult = m_Async->Get();
                m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            });
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Async)
        {
            m_Async->Cancel();
        }
    }

    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Async)
        {
            m_Thread.join();
            m_Async = nn::util::nullopt;
        }
    }

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_LastResult.IsSuccess())
        {
            return std::string("Success") + ToMilliSecondsString(m_LastTime);
        }
        else
        {
            return ToString(m_LastResult) + ToMilliSecondsString(m_LastTime);
        }
    }

    nn::ns::SystemUpdateControl* m_Control;
    std::thread m_Thread;
    nn::util::optional<nn::ns::AsyncResult> m_Async;
    nn::Result m_LastResult;
    nn::TimeSpan m_LastTime;
};

class ApplyDownloadedPanel : public RequestAndResultPanel
{
public:
    ApplyDownloadedPanel(nn::ns::SystemUpdateControl* control, glv::space_t w) NN_NOEXCEPT : RequestAndResultPanel(w, GT("apply downloaded")), m_Control(control), m_HasApplied() {}

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Control->HasDownloaded())
        {
            nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            m_Control->ApplyDownloadedUpdate();
            m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            m_HasApplied = true;
        }
        else
        {
            m_HasApplied = false;
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE {}
    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE {}

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_HasApplied ? "Success" + ToMilliSecondsString(m_LastTime) : "Not downloaded";
    }

    nn::ns::SystemUpdateControl* m_Control;
    bool m_HasApplied;
    nn::TimeSpan m_LastTime;
};

class SetupCardUpdatePanel : public RequestAndResultPanel
{
public:
    explicit SetupCardUpdatePanel(nn::ns::SystemUpdateControl* control, glv::space_t w) NN_NOEXCEPT
        : RequestAndResultPanel(w, GT("setup card update")), m_Control(control), m_CardUpdateBuffer() {}

    void ClearCardUpdateBuffer()
    {
        if (m_CardUpdateBuffer != nullptr)
        {
            devmenu::MemoryAllocator::FreeAlignedMemory(m_CardUpdateBuffer);
            m_CardUpdateBuffer = nullptr;
        }
        ClearResult();
    }

    bool IsSetupCardUpdate()
    {
        return m_CardUpdateBuffer != nullptr;
    }

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_CardUpdateBuffer == nullptr)
        {
            const size_t buffSize = 256 * nn::os::MemoryPageSize; // 1MB
            const size_t buffAlignment = 4096;
            auto buff = devmenu::MemoryAllocator::AllocAlignedMemory(buffAlignment, buffSize);

            nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            m_LastResult = m_Control->SetupCardUpdate(buff, buffSize);
            m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            if ( m_LastResult.IsSuccess() )
            {
                m_CardUpdateBuffer = buff;
            }
            else
            {
                devmenu::MemoryAllocator::FreeAlignedMemory(buff);
            }
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE {}
    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE {}

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_LastResult.IsSuccess())
        {
            return std::string("Success") + ToMilliSecondsString(m_LastTime);
        }
        else
        {
            return ToString(m_LastResult) + ToMilliSecondsString(m_LastTime);
        }
    }

    nn::ns::SystemUpdateControl* m_Control;
    void* m_CardUpdateBuffer;
    nn::Result m_LastResult;
    nn::TimeSpan m_LastTime;
};

class PrepareCardUpdatePanel : public RequestAndResultPanel
{
public:
    PrepareCardUpdatePanel(nn::ns::SystemUpdateControl* control, SetupCardUpdatePanel* panel, glv::space_t w) NN_NOEXCEPT
        : RequestAndResultPanel(w, GT("prepare card update")), m_Control(control), m_SetupCardUpdatePanel(panel) {}

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Async && !m_Async->TryWait();
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_SetupCardUpdatePanel->IsSetupCardUpdate())
        {
            m_Async.emplace();
            auto result = m_Control->RequestPrepareCardUpdate(&(*m_Async));
            if (result.IsFailure())
            {
                m_Async = nn::util::nullopt;
                m_LastResult = result;
                m_LastTime = 0;
            }
            else
            {
                m_Thread = std::thread([this] {
                    nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
                    m_LastResult = m_Async->Get();
                    m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
                });
            }
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Async)
        {
            m_Async->Cancel();
        }
    }

    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Async)
        {
            m_Thread.join();
            m_Async = nn::util::nullopt;
        }
    }

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_SetupCardUpdatePanel->IsSetupCardUpdate())
        {
            if (m_LastResult.IsSuccess())
            {
                return std::string("Success") + ToMilliSecondsString(m_LastTime);
            }
            else
            {
                return ToString(m_LastResult) + ToMilliSecondsString(m_LastTime);
            }
        }
        else
        {
            return "Not setup";
        }
    }

    nn::ns::SystemUpdateControl* m_Control;
    SetupCardUpdatePanel* m_SetupCardUpdatePanel;
    std::thread m_Thread;
    nn::util::optional<nn::ns::AsyncResult> m_Async;
    nn::Result m_LastResult;
    nn::TimeSpan m_LastTime;
};

class ApplyCardUpdatePanel : public RequestAndResultPanel
{
public:
    ApplyCardUpdatePanel(nn::ns::SystemUpdateControl* control, SetupCardUpdatePanel* panel, glv::space_t w) NN_NOEXCEPT
        : RequestAndResultPanel(w, GT("apply card update")), m_Control(control), m_SetupCardUpdatePanel(panel), m_HasApplied() {}

private:
    virtual bool IsOnRequest() NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    virtual void Request() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_SetupCardUpdatePanel->IsSetupCardUpdate() &&  m_Control->HasPreparedCardUpdate())
        {
            nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            m_Control->ApplyCardUpdate();
            m_LastTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            m_HasApplied = true;
        }
        else
        {
            m_HasApplied = false;
        }
    }

    virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE {}
    virtual void JoinRequest() NN_NOEXCEPT NN_OVERRIDE {}

    virtual std::string GetResultString() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_HasApplied ? "Success" + ToMilliSecondsString(m_LastTime) : "Not prepared";
    }

    nn::ns::SystemUpdateControl* m_Control;
    SetupCardUpdatePanel* m_SetupCardUpdatePanel;
    bool m_HasApplied;
    nn::TimeSpan m_LastTime;
};

class SystemUpdateControlPanel : public glv::Table
{
public:
    explicit SystemUpdateControlPanel(glv::space_t w) NN_NOEXCEPT : m_IsOccupied(), m_Timer(nn::os::EventClearMode_ManualClear)
    {
        auto pTitleLabel = new glv::Label(GT("System update control"), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
        m_pOccupyButton = new Button(GT("Occupy"), [&]{ ToggleOccupy(); }, glv::Rect(w, pTitleLabel->h + 10.0f));
        m_pControlTable = new glv::Table();

        float tablePadding = 8;
        auto contentWidth = w - tablePadding;

        m_pHasDownloadedLabel = new TitleLabel(contentWidth, GT("Has downloaded"), GT("False"));
        m_pDownloadProgressLabel = new TitleLabel(contentWidth, GT("Download progress"), GT("0 / 0"));
        m_pConnectNetworkPanel = new ConnectNetworkPanel(contentWidth);
        m_pCheckLatestPanel = new CheckLatestUpdatePanel(&m_Control, contentWidth);
        m_pDownloadLatestPanel = new DownloadLatestUpdatePanel(&m_Control, contentWidth);
        m_pApplyDownloadPanel = new ApplyDownloadedPanel(&m_Control, contentWidth);

        m_pCheckCardUpdateLabel = new TitleLabel(contentWidth, GT("Check card upate"), GT("False"));
        m_pSetupCardUpdatePanel = new SetupCardUpdatePanel(&m_Control, contentWidth);
        m_pPrepareCardUpdatePanel = new PrepareCardUpdatePanel(&m_Control, m_pSetupCardUpdatePanel, contentWidth);
        m_pApplyCardUpdatePanel = new ApplyCardUpdatePanel(&m_Control, m_pSetupCardUpdatePanel, contentWidth);

        *m_pControlTable << m_pHasDownloadedLabel << m_pDownloadProgressLabel << m_pConnectNetworkPanel << m_pCheckLatestPanel << m_pDownloadLatestPanel << m_pApplyDownloadPanel;
        *m_pControlTable << m_pCheckCardUpdateLabel << m_pSetupCardUpdatePanel << m_pPrepareCardUpdatePanel << m_pApplyCardUpdatePanel;

        m_pControlTable->disable(glv::Property::Visible);
        m_pControlTable->padding(tablePadding).arrange().fit(false);

        *this << pTitleLabel << m_pOccupyButton << m_pControlTable;
        padding(tablePadding).arrange().fit(false);

        m_Timer.StartPeriodic(nn::TimeSpan::FromSeconds(0), nn::TimeSpan::FromSeconds(1));
    }

    void Update() NN_NOEXCEPT
    {
        auto requiresAllUpdate = false;
        requiresAllUpdate |= m_pConnectNetworkPanel->Update();
        requiresAllUpdate |= m_pCheckLatestPanel->Update();
        requiresAllUpdate |= m_pDownloadLatestPanel->Update();
        requiresAllUpdate |= m_pApplyDownloadPanel->Update();
        requiresAllUpdate |= m_pSetupCardUpdatePanel->Update();
        requiresAllUpdate |= m_pPrepareCardUpdatePanel->Update();
        requiresAllUpdate |= m_pApplyCardUpdatePanel->Update();

        if (requiresAllUpdate || m_Timer.TryWait())
        {
            {
                glv::WideString label = m_IsOccupied ? GT("Relieve") : GT("Occupy");
                m_pOccupyButton->UpdateLabelText(label);
            }

            if (m_IsOccupied)
            {
                {
                    glv::WideString label = m_Control.HasDownloaded() ? GT("True") : GT("False");
                    m_pHasDownloadedLabel->UpdateValue(label);
                }

                {
                    auto progress = m_Control.GetDownloadProgress();
                    std::string label = std::to_string(progress.loaded) + " / " + std::to_string(progress.total);
                    m_pDownloadProgressLabel->UpdateValue(ToWideString(label));
                }

                {
                    nn::os::SystemEvent event;
                    nn::ns::GetGameCardUpdateDetectionEvent(&event);
                    glv::WideString label = event.TryWait() ? GT("True") : GT("False");
                    m_pCheckCardUpdateLabel->UpdateValue(label);
                }

                m_pControlTable->enable(glv::Property::Visible);
            }
            else
            {
                m_pControlTable->disable(glv::Property::Visible);
            }

            m_Timer.Clear();
        }
    }

private:
    bool m_IsOccupied;
    nn::ns::SystemUpdateControl m_Control;
    Button*     m_pOccupyButton;
    TitleLabel* m_pHasDownloadedLabel;
    TitleLabel* m_pDownloadProgressLabel;
    TitleLabel* m_pCheckCardUpdateLabel;
    glv::Table* m_pControlTable;

    ConnectNetworkPanel* m_pConnectNetworkPanel;
    CheckLatestUpdatePanel* m_pCheckLatestPanel;
    DownloadLatestUpdatePanel* m_pDownloadLatestPanel;
    ApplyDownloadedPanel* m_pApplyDownloadPanel;

    SetupCardUpdatePanel* m_pSetupCardUpdatePanel;
    PrepareCardUpdatePanel* m_pPrepareCardUpdatePanel;
    ApplyCardUpdatePanel* m_pApplyCardUpdatePanel;

    nn::os::TimerEvent m_Timer;

    void ToggleOccupy() NN_NOEXCEPT
    {
        if (!m_IsOccupied)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Control.Occupy());
            m_IsOccupied = true;
        }
        else
        {
            m_Control.Relieve();
            m_IsOccupied = false;
            m_pSetupCardUpdatePanel->ClearCardUpdateBuffer();
            m_pPrepareCardUpdatePanel->ClearResult();
            m_pApplyCardUpdatePanel->ClearResult();
        }

        m_Timer.Signal();
    }
};

class SystemUpdatePage : public devmenu::Page
{
public:
    SystemUpdatePage(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT : devmenu::Page(pageId, pageCaption, rect)
    {
        nn::ns::Initialize();

        auto bgStateIndicator = new BackgroundStateIndicator(1000);
        auto controlPanel = new SystemUpdateControlPanel(1000);
        /*
        auto language = new LanguageSetting(1000);
        auto dateTime = new DateTimeSetting(1000);
        auto brightness = new BrightnessSetting(1000);
        */

        auto table = new glv::Table("<", 3, 10);
        *table << bgStateIndicator << controlPanel;
        table->arrange().fit(false);

        *this << table;
        // Attach a Click handler to detect if the B button was pressed
        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        m_Indicator = bgStateIndicator;
        m_ControlPanel = controlPanel;
    }

    ~SystemUpdatePage() NN_NOEXCEPT
    {
        nn::ns::Finalize();
    }

    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);

        m_Indicator->Update();
        m_ControlPanel->Update();
    }

private:
    BackgroundStateIndicator* m_Indicator;
    SystemUpdateControlPanel* m_ControlPanel;
};

devmenu::PageCreatorImpl< SystemUpdatePage > g_SystemUpdatePageCreator( DevMenuPageId_SystemUpdate, "Sys Update" );

}}} // ~namespace devmenu::system::update, ~namespace devmenu::system, ~namespace devmenu
