﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iomanip>
#include <algorithm>
#include <sstream>
#include <vector>
#include <atomic>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/audioctrl.h>
#include <nn/audio.h>
#include <nn/audio/audio_AudioRendererApi.private.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/os/os_Event.h>
#include <nn/mem.h>
#include <nn/nn_SdkLog.h>
#include "DevMenu_Config.h"
#include "DevMenu_SoundUtil.h"
#include "DevMenu_Sound.h"
#include "DevMenu_Notification.h"

namespace devmenu {

namespace {

const int ChannelCountMax = 6;
const char* SoundDataPaths[ChannelCountMax] = {
    "Contents:/sound/One.adpcm",
    "Contents:/sound/Two.adpcm",
    "Contents:/sound/Three.adpcm",
    "Contents:/sound/Four.adpcm",
    "Contents:/sound/Five.adpcm",
    "Contents:/sound/Six.adpcm"
};

class SoundController : public NotificationMessageReceiver
{
private:
    NN_ALIGNAS(4096) char m_WorkBuffer[256 * 1024]; // AudioOut・AudioRenderer・AudioRendererConfig のワークバッファの合計
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char m_WaveBufferPoolMemory[400 * 1024]; // 6 チャンネル分のデータの合計

    nn::mem::StandardAllocator m_Allocator;
    nn::mem::StandardAllocator m_WaveBufferAllocator;

    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[16 * 1024];
    nn::os::ThreadType m_Thread;
    std::atomic<bool> m_IsInitialized;
    std::atomic<bool> m_IsExitRequested;
    std::atomic<int> m_PlayRequestChannelCount;
    nn::os::Mutex m_Mutex;

    void* m_SoundData[ChannelCountMax];
    size_t m_SoundDataSize[ChannelCountMax];
    nn::audio::VoiceType m_Voices[ChannelCountMax];
    nn::audio::WaveBuffer m_WaveBuffers[ChannelCountMax];
    nn::audio::AdpcmHeaderInfo* m_AdpcmHeader[ChannelCountMax];
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::AudioRendererHandle m_RendererHandle;
    nn::os::SystemEvent m_RendererSystemEvent;
    nn::audio::AudioOut m_OutHandle;
    nn::os::SystemEvent m_OutSystemEvent;
    nn::os::Event       m_RunningEvent;

    std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
    {
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS(result.IsSuccess());

        int64_t size;
        uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

        result = nn::fs::GetFileSize(&size, handle);
        NN_ABORT_UNLESS(result.IsSuccess());

        *adpcmData = m_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

        result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
        NN_ABORT_UNLESS(result.IsSuccess());
        result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::fs::CloseFile(handle);

        nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

        return static_cast<std::size_t>(size) - sizeof(adpcmheader);
    }

    std::size_t ReadWavFile(WavFormat* format, void** data, const char* filename)
    {
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS(result.IsSuccess());

        int64_t size;

        result = nn::fs::GetFileSize(&size, handle);
        NN_ABORT_UNLESS(result.IsSuccess());

        *data = m_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(*data);

        // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
        const std::size_t WavHeaderDataSize = 1024;

        result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
        NN_ABORT_UNLESS(result.IsSuccess());

        WavResult wavResult = ParseWavFormat(format, *data, WavHeaderDataSize);
        NN_ABORT_UNLESS_EQUAL(wavResult, WavResult_Success);
        NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

        result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::fs::CloseFile(handle);

        return static_cast<std::size_t>(format->dataSize);
    }

    template<typename T>
    size_t ConvertBlockInterleavedToInterleved(T* pDestBuffer, size_t destBufferSize, const T* pSrcBuffer, size_t srcBufferSize, int channelCount, int blockSampleCount) NN_NOEXCEPT
    {
        const auto copySize = std::min(destBufferSize, srcBufferSize);

        for(int channel = 0; channel < channelCount; ++channel)
        {
            const auto sampleCount = srcBufferSize / sizeof(T) / channelCount;
            for(size_t index = 0; index < sampleCount; ++index)
            {
                const auto blockIndex = channelCount * (index / blockSampleCount) + channel;
                auto blockBuffer = &pSrcBuffer[blockSampleCount * blockIndex];
                pDestBuffer[channelCount * index + channel] = blockBuffer[index % blockSampleCount];
            }
        }

        return copySize;
    }

    void ThreadFunc(void* args)
    {
        NN_UNUSED(args);
        NN_ASSERT(m_IsInitialized);

        // AudioOut のパラメータを指定します。
        nn::audio::AudioOutParameter outParameter;
        nn::audio::InitializeAudioOutParameter(&outParameter);
        outParameter.sampleRate = 48000;
        outParameter.channelCount = 6;
        // AudioOut をオープンします。
        if(!nn::audio::OpenDefaultAudioOut(&m_OutHandle, &m_OutSystemEvent, outParameter).IsSuccess())
        {
            DEVMENU_LOG("Failed to OpenDefaultAudioOut()\n");
            return;
        }

        // AudioOut 用バッファを確保します。
        const auto dataSize = sizeof(int16_t) * outParameter.channelCount * outParameter.sampleRate / 60; // 60 FPS
        const auto bufferCount = 2;
        const auto bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
        nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
        for(int i = 0; i < bufferCount; ++i)
        {
            auto buffer = reinterpret_cast<int16_t*>(m_Allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment));
            memset(buffer, 0, bufferSize);
            NN_ASSERT_NOT_NULL(buffer);

            // バッファを追加します。
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], buffer, bufferSize, dataSize);
            nn::audio::AppendAudioOutBuffer(&m_OutHandle, &audioOutBuffer[i]);
        }

        // レンダラのパラメータを指定します。
        nn::audio::AudioRendererParameter rendererParameter;
        nn::audio::InitializeAudioRendererParameter(&rendererParameter);
        rendererParameter.sampleRate = 48000;
        rendererParameter.sampleCount = 48000 / 200;
        rendererParameter.mixBufferCount = 6;
        rendererParameter.voiceCount = 6; // Surround (6 ch)
        rendererParameter.subMixCount = 0;
        rendererParameter.sinkCount = 1;
        rendererParameter.effectCount = 0;
        rendererParameter.performanceFrameCount = 0;
        rendererParameter.executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
        rendererParameter.renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;

        // ミックスバッファとオーディオバスの関係を定義します。
        const int channelCount = 6;
        int8_t mainBus[channelCount];
        mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
        mainBus[nn::audio::ChannelMapping_FrontRight] = 1;
        mainBus[nn::audio::ChannelMapping_FrontCenter] = 2;
        mainBus[nn::audio::ChannelMapping_LowFrequency] = 3;
        mainBus[nn::audio::ChannelMapping_RearLeft] = 4;
        mainBus[nn::audio::ChannelMapping_RearRight] = 5;

        // パラメータがシステムでサポートされているかどうかを確認します。
        NN_ABORT_UNLESS(
            nn::audio::IsValidAudioRendererParameter(rendererParameter),
            "Invalid AudioRendererrendererrendererParameter specified."
        );

        // レンダラのワークバッファを準備します。
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(rendererParameter);
        void* workBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer);

        // レンダラを初期化します。
        NN_ABORT_UNLESS(
            nn::audio::OpenAudioRenderer(&m_RendererHandle, &m_RendererSystemEvent, rendererParameter, workBuffer, workBufferSize).IsSuccess(),
            "Failed to open AudioRenderer"
        );

        // AudioRendererConfig を初期化します。
        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(rendererParameter);
        void* configBuffer = m_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(configBuffer);
        nn::audio::InitializeAudioRendererConfig(&m_Config, rendererParameter, configBuffer, configBufferSize);

        NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, channelCount));

        // レンダラの出力先を用意します。
        // CircularBufferSink のバッファを確保します。
        const int circularBufferSinkFrameCount = 6;
        size_t circularBufferSinkBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&rendererParameter, channelCount, circularBufferSinkFrameCount, nn::audio::SampleFormat_PcmInt16);
        void* circularBufferSinkBuffer = m_Allocator.Allocate(nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity), nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(circularBufferSinkBuffer);

        // CircularBufferSink のメモリープールを確保します。
        nn::audio::MemoryPoolType circularBufferMemoryPool;
        NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&m_Config, &circularBufferMemoryPool, circularBufferSinkBuffer, nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity)));
        NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&circularBufferMemoryPool));

        // CircularBufferSink を追加します。
        nn::audio::CircularBufferSinkType circularBufferSink;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&m_Config, &circularBufferSink, &m_FinalMix, mainBus, channelCount, circularBufferSinkBuffer, circularBufferSinkBufferSize, nn::audio::SampleFormat_PcmInt16));
        // 設定したパラメータをレンダラに反映させます。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_Config));

        // レンダリング結果を格納する一時バッファを確保します。
        const auto bufferingFrameCount = 3;
        size_t resultDataSize = rendererParameter.sampleCount * channelCount * sizeof(int16_t) * bufferingFrameCount;
        int16_t* pResultData = reinterpret_cast<int16_t*>(m_Allocator.Allocate(resultDataSize, nn::audio::AudioOutBuffer::AddressAlignment));
        NN_ASSERT_NOT_NULL(pResultData);

        // レンダリングを開始します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_RendererHandle));

        // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
        nn::audio::MemoryPoolType waveBufferMemoryPool;
        NN_ABORT_UNLESS(AcquireMemoryPool(&m_Config, &waveBufferMemoryPool, m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory)));
        NN_ABORT_UNLESS(RequestAttachMemoryPool(&waveBufferMemoryPool));

        for(int i = 0; i < ChannelCountMax; ++i)
        {
            // データを読み込みます
            m_AdpcmHeader[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(m_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
            m_SoundDataSize[i] = ReadAdpcmFile(m_AdpcmHeader[i], &m_SoundData[i], SoundDataPaths[i]);

            // Voice を用意します。
            nn::audio::AcquireVoiceSlot(&m_Config, &m_Voices[i], 32000, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, m_AdpcmHeader[i], sizeof(nn::audio::AdpcmParameter));
            nn::audio::SetVoiceDestination(&m_Config, &m_Voices[i], &m_FinalMix);
            nn::audio::SetVoiceMixVolume(&m_Voices[i], &m_FinalMix, 1.0f, 0, i);

            // WaveBuffer にデータを用意します。
            m_WaveBuffers[i].buffer = m_SoundData[i];
            m_WaveBuffers[i].size = m_SoundDataSize[i];
            m_WaveBuffers[i].startSampleOffset = 0;
            m_WaveBuffers[i].endSampleOffset = m_AdpcmHeader[i]->sampleCount;
            m_WaveBuffers[i].loop = false;
            m_WaveBuffers[i].isEndOfStream = true;
            m_WaveBuffers[i].pContext = nullptr;
            m_WaveBuffers[i].contextSize = 0;
        }

        // AudioOut を開始します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioOut(&m_OutHandle));

        // メインスレッドから終了指示が来るまで音を再生し続けます。
        m_PlayRequestChannelCount = 0;
        for(;;)
        {
            // サスペンド中はここで待ちます。
            m_RunningEvent.Wait();

            // 終了要求が来ている時はループを抜けます。
            if(m_IsExitRequested)
            {
                break;
            }

            nn::audio::AudioOutBuffer *pReleasedBuffer = nn::audio::GetReleasedAudioOutBuffer(&m_OutHandle);

            // バッファが取り出せなかった場合はバッファが一個以上消費されるまで待機した後に、再試行します。
            if(pReleasedBuffer == nullptr)
            {
                m_OutSystemEvent.Wait();
                continue;
            }

            // CircularBufferSink からデータを取得します。
            memset(pResultData, 0, resultDataSize);
            size_t totalWrittenSize = 0;
            while(totalWrittenSize < resultDataSize)
            {
                const auto remainSize = resultDataSize - totalWrittenSize;
                const auto writtenSize = nn::audio::ReadCircularBufferSink(&circularBufferSink, pResultData + (totalWrittenSize / sizeof(int16_t)), remainSize);
                totalWrittenSize += writtenSize;

                // オーディオレンダラのレンダリング処理を行います。
                nn::audio::ExecuteAudioRendererRendering(m_RendererHandle);

                // オーディオレンダラの更新処理を行います。
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_Config));
                m_RendererSystemEvent.Wait();
            }
            NN_ASSERT(totalWrittenSize == resultDataSize);

             // AudioOut にバッファを供給します。
            const auto releasedBufferBufferSize = nn::audio::GetAudioOutBufferBufferSize(pReleasedBuffer);
            NN_ASSERT(releasedBufferBufferSize > resultDataSize);

            int16_t* pInterleavedBuffer = reinterpret_cast<int16_t*>(nn::audio::GetAudioOutBufferDataPointer(pReleasedBuffer));

            // ブロックインターリーブフォーマットのデータをインタリーブフォーマットに変換します。
            const auto writtenSize = ConvertBlockInterleavedToInterleved(pInterleavedBuffer, releasedBufferBufferSize, pResultData, resultDataSize, channelCount, rendererParameter.sampleCount);
            NN_ASSERT(writtenSize == resultDataSize);

            // AudioOut にバッファを追加します。
            nn::audio::SetAudioOutBufferInfo(pReleasedBuffer, pInterleavedBuffer, releasedBufferBufferSize, writtenSize);
            nn::audio::AppendAudioOutBuffer(&m_OutHandle, pReleasedBuffer);

            // 再生が完了した WaveBuffer を取り出します。
            for(int i = 0; i < ChannelCountMax; ++i)
            {
                while(nn::audio::GetReleasedWaveBuffer(&m_Voices[i]) != nullptr);
            }

            const auto playRequestChannelCount = m_PlayRequestChannelCount.exchange(0);
            if(playRequestChannelCount > 0)
            {
                // Voice を Stop 状態に遷移させます。
                for(int i = 0; i < ChannelCountMax; ++i)
                {
                    nn::audio::SetVoicePlayState(&m_Voices[i], nn::audio::VoiceType::PlayState_Stop);
                }

                // オーディオレンダラのレンダリング処理を行います。
                nn::audio::ExecuteAudioRendererRendering(m_RendererHandle);

                // Stop 状態を反映するために、オーディオレンダラの更新処理を行います。
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_Config));
                m_RendererSystemEvent.Wait();

                // 要求されたチャンネル分の Voice に WaveBuffer を登録し、Play 状態に遷移させます。
                for(int i = 0; i < playRequestChannelCount; ++i)
                {
                    nn::audio::AppendWaveBuffer(&m_Voices[i], &m_WaveBuffers[i]);
                    nn::audio::SetVoicePlayState(&m_Voices[i], nn::audio::VoiceType::PlayState_Play);
                }
            }
        }

        // Voice を解放します。
        for(int i = 0; i < ChannelCountMax; ++i)
        {
            nn::audio::ReleaseVoiceSlot(&m_Config, &m_Voices[i]);
        }

        // レンダリングを終了します。
        nn::audio::StopAudioRenderer(m_RendererHandle);
        nn::audio::CloseAudioRenderer(m_RendererHandle);

        // AudioOut を終了します。
        nn::audio::StopAudioOut(&m_OutHandle);
        nn::audio::CloseAudioOut(&m_OutHandle);

        // 確保したメモリを解放します。
        for(size_t i = 0; i < NN_ARRAY_SIZE(audioOutBuffer); ++i)
        {
            auto p = nn::audio::GetAudioOutBufferDataPointer(&audioOutBuffer[i]);
            m_Allocator.Free(p);
        }
        for(int i = 0; i < ChannelCountMax; ++i)
        {
            if (m_SoundData[i])
            {
                m_WaveBufferAllocator.Free(m_SoundData[i]);
                m_SoundData[i] = nullptr;
            }
        }
        if (configBuffer)
        {
            m_Allocator.Free(configBuffer);
            configBuffer = nullptr;
        }
        if (workBuffer)
        {
            m_Allocator.Free(workBuffer);
            workBuffer = nullptr;
        }
        if (circularBufferSinkBuffer)
        {
            m_Allocator.Free(circularBufferSinkBuffer);
            circularBufferSinkBuffer = nullptr;
        }
        if (pResultData)
        {
            m_Allocator.Free(pResultData);
            pResultData = nullptr;
        }
    } // NOLINT(readability/fn_size)

    /**
     * @brief       AEメッセージ通知受信コールバック。
     *
     * @param[in]   message 受信した通知メッセージ。
     *
     * @details     起動時の Foreground イベントは呼び出されません。
     *              本コールバックは通知メッセージ受信スレッド上で呼び出されます。( DevMenu_Notification.cpp )
     *              通知メッセージ受信スレッド上での処理は以下の制約があります。
     *              - 受信スレッドを長時間停止させる処理を行わない。
     *                  停止保証値はありません、遅延が発生した場合システム状態との不和などの影響が発生します。
     *              - 高負荷な処理を行わない。( 他スレッドとの同期、演算処理など。)
     *                  他スレッドへの通知用フラグ( イベントオブジェクト )などの制御に留めてください。
     *              - スタックを大量に消費する処理を行わない。 ( 受信スレッドのスタックは 8 KiB です。 )
     */
    virtual void OnNotificationMessageReceived( NotificationMessageReceiver::Message message ) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( m_IsInitialized )
        {
            switch ( message )
            {
            case NotificationMessageReceiver::Message_ChangeIntoBackground:
                Suspend();
                break;
            case NotificationMessageReceiver::Message_ChangeIntoForeground:
                Resume();
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

public:
    SoundController() NN_NOEXCEPT
        : m_Mutex(false),
          m_RunningEvent(nn::os::EventClearMode_ManualClear)
    {

    }

    void Initialize() NN_NOEXCEPT
    {
        m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
        m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

        m_IsInitialized = true;
        m_IsExitRequested = false;
        m_RunningEvent.Signal();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadEntry, this, m_ThreadStack, sizeof(m_ThreadStack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&m_Thread);
    }

    void Finalize() NN_NOEXCEPT
    {
        m_IsExitRequested = true;
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);

        m_IsInitialized = false;
        m_Allocator.Finalize();
        m_WaveBufferAllocator.Finalize();
    }

    void Suspend() NN_NOEXCEPT
    {
        // シグナル状態を解除してサウンドスレッドをサスペンド
        m_RunningEvent.Clear();
    }

    void Resume() NN_NOEXCEPT
    {
        // シグナル状態にしてサウンドスレッドをレジューム
        m_RunningEvent.Signal();
    }

    bool IsInitialzied() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    static void ThreadEntry(void *args) NN_NOEXCEPT
    {
        reinterpret_cast<SoundController*>(args)->ThreadFunc(args);
    }

    void Playback(int channelCount) NN_NOEXCEPT
    {
        NN_ASSERT(channelCount <= 6);

        m_PlayRequestChannelCount = channelCount;
    }
};

SoundController g_SoundController;

} // ~namespace devmenu::<anonymous>

void InitializeSound() NN_NOEXCEPT
{
    g_SoundController.Initialize();
}

void FinalizeSound() NN_NOEXCEPT
{
    g_SoundController.Finalize();
}

void SuspendSound() NN_NOEXCEPT
{
    g_SoundController.Suspend();
}

void ResumeSound() NN_NOEXCEPT
{
    g_SoundController.Resume();
}

void PlaybackTestSound(int channelCount) NN_NOEXCEPT
{
    g_SoundController.Playback(channelCount);
}

bool EnableNotificationMessageReceivingOnSound( bool permission ) NN_NOEXCEPT
{
    return g_SoundController.EnableNotificationMessageReceiving( permission );
}

} // ~namespace devmenu
