﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 *  @file
 *  @brief DebugタブのMiiデバッグ機能用ヘッダ
 *  @details
 */

#include <cstdlib>
#include <nn/fs.h>
#include <glv_CustomVerticalListView.h>
#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_DebugSettingsCommon.h"
#include "DevMenu_RootSurface.h"

namespace devmenu { namespace debugsettings {

struct MiiImportFilePropertyType
{
    static const int64_t MaxDirectoryNameLength = 300;  //!< SD カード内ファイルパス最大文字数（終端文字を含む）
    /**
     * @brief SD カード内のファイル用プロパティデータを構築します。
     */
    void Prepare( const nn::fs::DirectoryEntry& dirEntry, const char* pFilePath ) NN_NOEXCEPT;

    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];
};

/**
 * @brief Miiデータベースインポートデータ選択リスト
 * @details
 */
class MiiImportFileListView : public glv::CustomVerticalListView< MiiImportFilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef CustomVerticalListView< MiiImportFilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit MiiImportFileListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT;
protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE;

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE;
private:
};

/**
 * @brief Miiデータベースインポートデータ選択用シーン
 * @details
 */
class MiiImportFileScene : public glv::Group
{
public:
    void Open(Page* pParentPage) NN_NOEXCEPT;
    void Close() NN_NOEXCEPT;
    glv::View* GetFirstFocusTargetView() NN_NOEXCEPT;

    MiiImportFileScene(
        const glv::Rect& rect,
        std::function<void()> openCallback,
        std::function<void()> closeCallback,
        std::function<void(const char* pPath)> importCallback) NN_NOEXCEPT;

    nn::Result MakeImportFileList() NN_NOEXCEPT;
    int GetListCount() const NN_NOEXCEPT;
    void ImportMiiDatabase(const MiiImportFilePropertyType& property) NN_NOEXCEPT;

private:
    devmenu::Button m_BackButton;
    std::function<void()> m_OpenCallback;
    std::function<void()> m_CloseCallback;
    std::function<void(const char* pPath)> m_ImportCallback;

    std::vector< MiiImportFilePropertyType > m_FileList;
    MiiImportFileListView* m_pImportFileView;
};

/**
 * @brief Miiデバッグ機能のUIテーブル
 */
class MiiSetting : public SettingsItemTableView
{
public:
    enum ExecuteType {
        ExecuteType_ImportMiiDatabase,
        ExecuteType_ExportMiiDatabase,
        ExecuteType_FormatMiiDatabase,
        ExecuteType_ExportCharInfo,
        ExecuteType_End,
    };
    explicit MiiSetting (glv::space_t width,RootSurfaceContext* pRootSurface, std::function< void() > openImportScene) NN_NOEXCEPT;
    void Execute(ExecuteType type) NN_NOEXCEPT;

    /// MiiImportFileSceneからの復帰でフォーカスを合わせる
    devmenu::Button* GetImportButton() NN_NOEXCEPT { return m_pImportDatabaseButton; }

    /// 実際のMiiインポート処理
    void ImportDatabase(const char* pPath) NN_NOEXCEPT
    {
        auto result = ImportDatabaseImpl(pPath);
        ViewMessage(result,ExecuteType_ImportMiiDatabase);
    }
private:
    RootSurfaceContext* m_pRootSurface;
    std::function< void() > m_OpenImportCallback;
    devmenu::Button* m_pImportDatabaseButton;
    devmenu::Button* m_pExportDatabaseButton;
    devmenu::Button* m_pFormatDatabaseButton;
    devmenu::Button* m_pExportCharInfoButton;

    void ExecuteImpl(ExecuteType type) NN_NOEXCEPT;

    nn::Result ImportDatabaseImpl(const char* pFilePath) NN_NOEXCEPT;
    nn::Result ExportMiiDatabase() NN_NOEXCEPT;
    nn::Result FormatMiiDatabase() NN_NOEXCEPT;
    nn::Result ExportCharInfo() NN_NOEXCEPT;

    void ViewMessage(nn::Result result,ExecuteType type) NN_NOEXCEPT;

    void UpdateButtonFocusAndColor( bool isValid ) NN_NOEXCEPT;
};

/**
 * @brief Mii画像デバッグ機能のUIテーブル
 */
class MiiImgSetting : public SettingsItemTableView
{
public:
    explicit MiiImgSetting (glv::space_t width) NN_NOEXCEPT;
};

}} // ~namespace devmenu::debugsettings, ~namespace devmenu
