﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <vector>

#include <nn/ncm/ncm_ContentMetaKey.h>

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <glv/utils/glv_MessageThread.h>

#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

namespace devmenu { namespace application {

class ApplicationCatalogPage;
class CatalogScene;
class ContentsListView;

class DeleteContentsScene : public Scene
{
class CloseButton;

public:
    DeleteContentsScene( ApplicationCatalogPage* pParentPage, Scene* pDetailScene = nullptr, Scene* pCatalogScene = nullptr ) NN_NOEXCEPT;
    void SetApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;
    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE {}
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    void CloseScene( Scene* pDstScene ) NN_NOEXCEPT;
    void ResetRequiredVersion() NN_NOEXCEPT;
    void DeleteContent() NN_NOEXCEPT;
    void DeleteAllContents() NN_NOEXCEPT;
    CloseButton* GetCloseButton() NN_NOEXCEPT
    {
        return m_pCloseButton;
    }

private:
    class CloseButton : public devmenu::Button
    {
    public:
        CloseButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor) NN_NOEXCEPT;
        void CloseEventCallback() NN_NOEXCEPT;

    private:
        DeleteContentsScene* m_pParent;
    };

    class ResetRequiredVersionButton : public devmenu::Button
    {
    public:
        ResetRequiredVersionButton(DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor) NN_NOEXCEPT;

    private:
        void ResetRequiredVersionEventCallback() NN_NOEXCEPT;
        DeleteContentsScene* m_pParent;
    };

    class DeleteButton : public devmenu::Button
    {
    public:
        DeleteButton(DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor) NN_NOEXCEPT;

    private:
        void DeleteEventCallback() NN_NOEXCEPT;
        DeleteContentsScene* m_pParent;
    };

    class DeleteAllButton : public devmenu::Button
    {
    public:
        DeleteAllButton(DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor) NN_NOEXCEPT;

    private:
        void DeleteAllEventCallback() NN_NOEXCEPT;
        DeleteContentsScene* m_pParent;
    };

    class ShowContentsHashButton : public devmenu::Button
    {
    public:
        ShowContentsHashButton(DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor) NN_NOEXCEPT;

    private:
        void ShowContentsHashEventCallback() NN_NOEXCEPT;
        DeleteContentsScene* m_pParent;
    };

    class DescriptionLabel
    {
    public:
        DescriptionLabel() NN_NOEXCEPT;
        void SetApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;
        glv::Label* GetLabel() NN_NOEXCEPT {return &m_Label;}
        void SetStyle( glv::Style* style ) NN_NOEXCEPT { m_Label.style( style ); }
    private:
        glv::Label m_Label;
    };

    class ContentsTileView : public glv::View
    {
    public:
        ContentsTileView( glv::space_t width, glv::space_t height ) NN_NOEXCEPT;

        void UpdateContentTileViewValues( const nn::ncm::StorageContentMetaKey& storageContent,
            const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

        static const char* GetStorageIdString( nn::ncm::StorageId storageId ) NN_NOEXCEPT;

        virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;

    private:
        Table                           m_ContentInformationTable;
        glv::Label*                     m_pContentTarget;
        glv::Label*                     m_pContentVersion;
        glv::Label*                     m_pContentStorageId;
    };

    class ContentTileContainer : public glv::View
    {
    public:
        ContentTileContainer(
            ContentsTileView* pContentTile,
            glv::space_t width, glv::space_t height,
            const nn::ncm::StorageContentMetaKey& content,
            const nn::ncm::ApplicationId& applicationId,
            DeleteContentsScene* pParent ) NN_NOEXCEPT;
        ~ContentTileContainer() NN_NOEXCEPT;

        virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;
        virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;
    private:
        nn::ncm::StorageContentMetaKey  m_StorageContent;
        nn::ncm::ApplicationId          m_ApplicationId;
        ContentsTileView*               m_pContentTile;
        DeleteContentsScene*            m_pParent;
    };

private:
    static void OnButtonClickNotification( const glv::Notification& notification ) NN_NOEXCEPT;
    void LoadContents() NN_NOEXCEPT;

    void ShowContentsHash() NN_NOEXCEPT;
    nn::Result GetApplicationRomIdHash( nn::ncm::Hash* pOutHash ) NN_NOEXCEPT;
    void AddApplicationRomIdHash( MessageView* pView ) NN_NOEXCEPT;

private:
    std::function< void() >                     m_Callback;
    nn::ncm::ApplicationId                      m_ApplicationId;
    std::vector<nn::ncm::StorageContentMetaKey> m_ContentMetaKey;

    CloseButton*                                m_pCloseButton;
    ResetRequiredVersionButton*                 m_pResetRequiredVersionButton;
    DeleteButton*                               m_pDeleteButton;
    DeleteAllButton*                            m_pDeleteAllButton;
    ShowContentsHashButton*                     m_pShowContentsHashButton;
    DescriptionLabel*                           m_pDescriptionLabel;

    ApplicationCatalogPage*                     m_pParentPage;
    Scene*                                      m_pDetailScene;
    Scene*                                      m_pCatalogScene;
    glv::View*                                  m_pContainer;
    ScrollableBoxView*                          m_pScrollContainer;
    glv::Table*                                 m_pHeaderTable;
    nn::ncm::StorageContentMetaKey*             m_pSelectedContent;
    std::vector< DeleteContentsScene::ContentsTileView* > m_ContentsTilesList;

    bool                                        m_ExistsApplication; // Application 本体が存在するか否か

    static const int                            MaxCountOfVisibleContentsTiles = 6;
};

}} // ~namespace devmenu::application, ~namespace devmenu
