﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <iomanip>
#include <mutex>
#include <sstream>
#include <vector>

#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/pctl/pctl_ApiForRecovery.h>
#include <nn/pctl/pctl_ApiSystemReadOnly.h>

#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"
#include "Common/DevMenu_CommonDropDown.h"

namespace devmenu { namespace cssettings {

typedef enum ParentalControlMode
{
    ParentalControlMode_AllEnable,    //!< 全て有効
    ParentalControlMode_AllDisable,   //!< 全て無効
} DeviceStyle;

const ParentalControlMode ParentalControlModeTable[] =
{
    ParentalControlMode_AllEnable,
    ParentalControlMode_AllDisable,
};

const char* GetParentalControlModeLabel(const ParentalControlMode& mode) NN_NOEXCEPT
{
    switch(mode)
    {
    case ParentalControlMode_AllEnable:
        return "Enabled";
    case ParentalControlMode_AllDisable:
        return "Disabled";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

class ParentalControlRestrictedSetting : public glv::Table
{
public:
    explicit ParentalControlRestrictedSetting(glv::space_t width) NN_NOEXCEPT
    {
        auto pLlabel = new glv::Label(GLV_TEXT_API_STRING_UTF16("ペアコンの制限状態"), glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize));
        m_pState = new glv::Label(GLV_TEXT_API_WIDE_STRING(""), glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize));
        m_pState->anchor(glv::Place::TR).pos(-m_pState->width(), 0);
        m_pState->enable(glv::Property::KeepWithinParent);

        auto pGroup = new glv::Group(glv::Rect(width, m_pState->height()));

        *pGroup << pLlabel << m_pState;

        *this << pGroup;
        arrange().fit(false);
    }

    void SetRestrictionStatus() NN_NOEXCEPT
    {
        if (nn::pctl::IsRestrictionEnabled())
        {
            m_pState->setValue(glv::WideString(GLV_TEXT_API_WIDE_STRING("On")));
        }
        else
        {
            m_pState->setValue(glv::WideString(GLV_TEXT_API_WIDE_STRING("Off")));
        }
    }

private:
    glv::Label* m_pState;
};

class ParentalControlCurrentlyModeDropDown : public devmenu::DropDownBase
{
public:
    ParentalControlCurrentlyModeDropDown(const glv::Rect& rect, float textSize, std::function< void() > stateChangeCallback) NN_NOEXCEPT
        : devmenu::DropDownBase(rect, textSize)
        , m_StateChangeCallback(stateChangeCallback)
    {
        for (int i = 0; i < sizeof(ParentalControlModeTable) / sizeof(ParentalControlModeTable[0]); i++)
        {
            addItem(GetParentalControlModeLabel(ParentalControlModeTable[i]));
        }

        attach([](const glv::Notification& notification)->void { notification.receiver<ParentalControlCurrentlyModeDropDown>()->UpdateMode(); }, glv::Update::Action, this);
    }

    void UpdateMode() NN_NOEXCEPT
    {
        const auto outputMode = ParentalControlModeTable[mSelectedItem];
        switch (outputMode)
        {
        case ParentalControlMode_AllEnable:
            // 現在の状態を Enable にすることはできない
            break;
        case ParentalControlMode_AllDisable:
            nn::pctl::DisableAllFeatures();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        m_StateChangeCallback();
    }

    void SetMode() NN_NOEXCEPT
    {
        bool isNextBootEnable;
        bool isCurrentDisable = nn::pctl::IsAllFeaturesDisabled(&isNextBootEnable);
        DEVMENU_LOG("%d = nn:: pctl:: IsAllFeaturesDisabled(%d)\n", isCurrentDisable, isNextBootEnable);
        auto currentMode = (isCurrentDisable == true) ? ParentalControlMode_AllDisable : ParentalControlMode_AllEnable;

        setValue(GetParentalControlModeLabel(currentMode));
    }
private:
    std::function< void() > m_StateChangeCallback;
};

class ParentalControlOnNextBootModeDropDown : public devmenu::DropDownBase
{
public:
    ParentalControlOnNextBootModeDropDown(const glv::Rect& rect, float textSize, std::function< void() > stateChangeCallback) NN_NOEXCEPT
        : devmenu::DropDownBase(rect, textSize)
        , m_StateChangeCallback(stateChangeCallback)
    {
        for (int i = 0; i < sizeof(ParentalControlModeTable) / sizeof(ParentalControlModeTable[0]); i++)
        {
            addItem(GetParentalControlModeLabel(ParentalControlModeTable[i]));
        }

        attach([](const glv::Notification& notification)->void { notification.receiver<ParentalControlOnNextBootModeDropDown>()->UpdateMode(); }, glv::Update::Action, this);
    }

    void UpdateMode() NN_NOEXCEPT
    {
        const auto outputMode = ParentalControlModeTable[mSelectedItem];
        switch (outputMode)
        {
        case ParentalControlMode_AllEnable:
            nn::pctl::PostEnableAllFeatures();
            break;
        case ParentalControlMode_AllDisable:
            nn::pctl::DisableAllFeatures();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        m_StateChangeCallback();
    }

    void SetMode() NN_NOEXCEPT
    {
        bool isNextBootEnable;
        bool isCurrentDisable = nn::pctl::IsAllFeaturesDisabled(&isNextBootEnable);
        DEVMENU_LOG("%d = nn:: pctl:: IsAllFeaturesDisabled(%d)\n", isCurrentDisable, isNextBootEnable);
        auto currentMode = (isNextBootEnable == false) ? ParentalControlMode_AllDisable : ParentalControlMode_AllEnable;

        if (isNextBootEnable == false && isCurrentDisable == false)
        {
            // 返り値と引数(out)がfalseの場合は、次回起動時のペアコンは有効として扱う
            currentMode = ParentalControlMode_AllEnable;
        }

        setValue(GetParentalControlModeLabel(currentMode));
    }
private:
    std::function< void() > m_StateChangeCallback;
};

class ParentalControlModeSetting : public glv::Table
{
public:
    explicit ParentalControlModeSetting(glv::space_t width, const glv::WideString& text, devmenu::DropDownBase* pDropDown) NN_NOEXCEPT
    {
        auto label = new glv::Label(text, glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize));
        pDropDown->anchor(glv::Place::TR).pos(-pDropDown->width(), 0.0f);
        pDropDown->enable(glv::Property::KeepWithinParent);

        auto group = new glv::Group(glv::Rect(width, pDropDown->h));
        *group << label << pDropDown;

        *this << group;
        arrange();
        fit(false);

        m_pDropDown = pDropDown;
    }

    glv::View* GetBeginFocus() NN_NOEXCEPT
    {
        return m_pDropDown;
    }

private:
    glv::View* m_pDropDown;
};

class CustomerSupportSettingsPage : public devmenu::Page
{
public:
    CustomerSupportSettingsPage(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT
        : devmenu::Page(pageId, pageCaption, rect),
        m_pFirst( nullptr ),
        m_pMainTable( nullptr)
    {
    }

    ~CustomerSupportSettingsPage() NN_NOEXCEPT
    {
    }

    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        const glv::space_t rightPadding = 24.f;
        const glv::space_t panelWidth = width() - rightPadding;
        m_RestrictedSetting = new ParentalControlRestrictedSetting(panelWidth);
        m_CurrentlyModeDropDown = new ParentalControlCurrentlyModeDropDown(glv::Rect(320.0f, 35.0f), CommonValue::InitialFontSize, [&]() { ChabgeDisplayStatus(); });
        m_OnNextBootModeDropDown = new ParentalControlOnNextBootModeDropDown(glv::Rect(320.0f, 35.0f), CommonValue::InitialFontSize, [&]() { ChabgeDisplayStatus(); });
        auto pCurrentlyModeSetting = new ParentalControlModeSetting(
            panelWidth,
            GLV_TEXT_API_STRING_UTF16("ペアコン機能の切替"),
            m_CurrentlyModeDropDown);
        auto pOnNextBootModeSetting = new ParentalControlModeSetting(
            panelWidth,
            GLV_TEXT_API_STRING_UTF16("再起動後のペアコン機能"),
            m_OnNextBootModeDropDown);

        auto table = new glv::Table( "<", 3.0f, 10.0f );
        *table << m_RestrictedSetting << pCurrentlyModeSetting << pOnNextBootModeSetting;
        table->arrange().fit( false );

        *this << table;

        m_pFirst = pCurrentlyModeSetting->GetBeginFocus();

        ChabgeDisplayStatus();

        m_pMainTable = table;
    }

    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFirst;
    }

    void ChabgeDisplayStatus() NN_NOEXCEPT
    {
        m_RestrictedSetting->SetRestrictionStatus();
        m_CurrentlyModeDropDown->SetMode();
        m_OnNextBootModeDropDown->SetMode();
    }

private:
    glv::View* m_pFirst;
    glv::Table* m_pMainTable;
    ParentalControlRestrictedSetting* m_RestrictedSetting;
    ParentalControlCurrentlyModeDropDown* m_CurrentlyModeDropDown;
    ParentalControlOnNextBootModeDropDown* m_OnNextBootModeDropDown;
};

devmenu::PageCreatorImpl< CustomerSupportSettingsPage > g_CustomerSupportSettingsPageCreator( DevMenuPageId_CustomerSupport, "Cs" );

}} // ~namespace devmenu::cssettings, ~namespace devmenu
