﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstdio>
#include <functional>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_Config.h"
#include "DevMenu_Common.h"
#include "DevMenu_RootSurface.h"
#include "Common/DevMenu_CommonDropDown.h"
#include "Launcher/DevMenu_LauncherLibraryAppletApis.h"
#include "SystemProgram/DevMenu_ProgramIdNameMap.h"

#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_ApplicationCommand.h"

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#include "DevMenuCommand_ShopCommand.h"
#include "DevMenuCommand_TicketCommand.h"
#include "DevMenuCommand_ElicenseCommand.h"
#include "DevMenuCommand_DynamicRightsCommand.h"
#endif

#include "DevMenuCommand_SystemUpdateCommand.h"
#include "DevMenuCommand_SystemProgramCommand.h"
#include "DevMenuCommand_ServiceDiscoveryCommand.h"
#include "DevMenuCommand_FileSystemCommand.h"
#include "DevMenuCommand_ErrorCommand.h"
#include "DevMenuCommand_PlayDataCommand.h"

using namespace nn;

// ログ出力のコールバック用
namespace devmenu {

namespace
{
    LogCallback g_RegisteredLogCallback;
    VLogCallback g_RegisteredVLogCallback;
}

void RegisterLogCallback(LogCallback f) NN_NOEXCEPT
{
    g_RegisteredLogCallback = f;
}
LogCallback GetLogCallback() NN_NOEXCEPT
{
    return g_RegisteredLogCallback;
}

void RegisterVLogCallback(VLogCallback f) NN_NOEXCEPT
{
    g_RegisteredVLogCallback = f;
}
VLogCallback GetVLogCallback() NN_NOEXCEPT
{
    return g_RegisteredVLogCallback;
}


namespace system { namespace command {

const int LabelCount = 28;
const int MaxLines = 8000;
const int DisplaySubCommandsPerCommand = 20;

class LineLog
{
public:
    LineLog() NN_NOEXCEPT
    {
        m_data[0] = '\0';
    }
    void SetData(const char* data) NN_NOEXCEPT
    {
        nn::util::Strlcpy(m_data, data, sizeof(m_data));
    }
    void ConcatData(const char* data) NN_NOEXCEPT
    {
        auto length = nn::util::Strnlen(m_data, sizeof(m_data));
        nn::util::Strlcpy(&(m_data[length]), data, sizeof(m_data) - length);
    }
    const char* GetString() NN_NOEXCEPT
    {
        return m_data;
    }
    glv::WideString GetWideString() NN_NOEXCEPT
    {
        glv::WideString ws;
        return glv::CopyWithConvertWideString(ws, m_data);
    }
private:
    char m_data[500];
};

class FullLog
{
public:
    FullLog() NN_NOEXCEPT : m_LogHeadIndex(0), m_DisplayingIndex(0)
    {
    }

    void AppendLine(const char* s, bool willContinue) NN_NOEXCEPT
    {
        m_Lines[m_LogHeadIndex].ConcatData(s);
        if(willContinue == false)
        {
            if(m_LogHeadIndex == m_DisplayingIndex)
            {
                Scroll(1);
            }
            m_LogHeadIndex++;
            m_Lines[m_LogHeadIndex].SetData("");
        }
        if(m_LogHeadIndex >= MaxLines)
        {
            m_LogHeadIndex = 0;
        }
        if(m_LogHeadIndex % 100 == 0)
        {
            // 100 行先 - 200 行先をクリアしておく
            for(int i = 100; i < 200; ++i)
            {
                int index = (m_LogHeadIndex + i) % MaxLines;
                m_Lines[index].SetData("");
            }
        }
    }
    void ParseNewLineAndRegister(char* string) NN_NOEXCEPT
    {
        char* p1 = string;
        char* p2 = string;

        while(*p2 != '\0')
        {
            if(*p2 == '\n')
            {
                *p2 = '\0';
                AppendLine(p1, false);
                p1 = (p2 + 1);
            }
            p2++;
        }
        AppendLine(p1, true);
    }
    void Scroll(int n) NN_NOEXCEPT
    {
        m_DisplayingIndex = (m_DisplayingIndex + n + MaxLines) % MaxLines;
    }
    void ScrollToHead() NN_NOEXCEPT
    {
        m_DisplayingIndex = m_LogHeadIndex;
    }
    // DisplayingIndex がさしているログを 0 とし、
    // それより古いものを 1, 2, 3, .. と指定する
    glv::WideString GetLog(int n) NN_NOEXCEPT
    {
        int index = (m_DisplayingIndex - n + MaxLines) % MaxLines;
        return m_Lines[index].GetWideString();
    }
    // LogHeadIndex  がさしているログを 0 とし、
    // それより古いものを 1, 2, 3, .. と指定する
    const char* GetPlainTextForAnalayze(int n) NN_NOEXCEPT
    {
        int index = (m_LogHeadIndex - n + MaxLines) % MaxLines;
        return m_Lines[index].GetString();
    }
    int GetCurrentHead() NN_NOEXCEPT
    {
        return m_LogHeadIndex;
    }
private:
    LineLog m_Lines[MaxLines];
    int m_LogHeadIndex;
    int m_DisplayingIndex;
};

FullLog s_FullLog;
struct AvailableCommand
{
    char commandName[64];
    Result (*function)(bool*, const Option&);;
    std::vector<std::string> subcommands;
};
AvailableCommand g_Commands[] = {
#if !defined(NN_DEVMENULOTCHECK_DOWNLOADER)
    {"application", ApplicationCommand},
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    {"shop", ShopCommand},
    {"ticket", TicketCommand},
    {"elicense", ElicenseCommand},
    {"dynamicrights", DynamicRightsCommand},
#endif
    {"systemupdate", SystemUpdateCommand},
    {"systemprogram", SystemProgramCommand},
    {"servicediscovery", ServiceDiscoveryCommand},
    {"filesystem", FileSystemCommand},
    {"error", ErrorCommand},
    {"playdata", PlayDataCommand },
#endif
};

struct SplittedCommand
{
    AvailableCommand* command;
    int splitIndex;
    int subCommandOffset;
};

// コマンド生成・実行するクラス
class CommandGenerator
{
public:
    CommandGenerator() NN_NOEXCEPT
    {
    }
    // command 自体の中身を書き換えて、argv 形式に変換する
    int split(char** argv, int maxSplit, char* command) NN_NOEXCEPT
    {
        char* p1 = command;
        char* p2 = command;
        int count = 0;

        while(*p2 != '\0')
        {
            if(*p2 == ' ' && p1 == p2)
            {
                p1++;
            }
            else if (*p2 == ' ')
            {
                *p2 = '\0';
                argv[count] = p1;
                count++;
                if(count >= maxSplit)
                {
                    return count;
                }
                p1 = p2 + 1;
            }
            p2++;
        }
        argv[count] = p1;
        count++;
        return count;
    }
    nn::Result Execute(bool* outValue, const char* command) NN_NOEXCEPT
    {

        static char buffer[1 << 10];
        snprintf(buffer, sizeof(buffer), "%s", command);

        const int maxSplit = 30;
        char* argv[maxSplit];
        char dummyArg0[] = "DevMenuCommandSystem";
        argv[0] = dummyArg0;
        int argc = split(&(argv[1]), maxSplit - 1, buffer);
        argc++;

        // 先頭はメインコマンド
        Option opt(argc, argv);
        for(auto& c : g_Commands)
        {
            if(nn::util::Strncmp(argv[1], c.commandName, sizeof(c.commandName)) == 0)
            {
                return c.function(outValue, opt);
            }
        }
        *outValue = false;
        return nn::ResultSuccess();
    }
    std::vector<SplittedCommand> ParseHelpAndGetCommands() NN_NOEXCEPT
    {
        for (auto& c : g_Commands)
        {
            static char buffer[1024];
            snprintf(buffer, sizeof(buffer), "%s --help", c.commandName);
            int pre = s_FullLog.GetCurrentHead();
            bool flag;
            Execute(&flag, buffer);
            int post = s_FullLog.GetCurrentHead();

            {
                std::string s("--help");
                c.subcommands.push_back(s);
            }

            for (int i = (post - pre); i >= 0; --i)
            {
                snprintf(buffer, sizeof(buffer), "%s", s_FullLog.GetPlainTextForAnalayze(i));
                const int maxSplit = 30;
                char* argv[maxSplit];
                int argc = split(argv, maxSplit, buffer);
                for (int j = 0; j < argc; j++)
                {
                    if (nn::util::Strncmp(argv[j], c.commandName, sizeof(c.commandName)) == 0 && j + 1 < argc)
                    {
                        std::string s(argv[j + 1]);
                        c.subcommands.push_back(s);
                        break;
                    }
                }
            }
        }
        std::vector<SplittedCommand> commands;
        for (auto& c : g_Commands)
        {
            int index = 0;
            for (int i = 0; i < static_cast<int>(c.subcommands.size()); i += DisplaySubCommandsPerCommand)
            {
                SplittedCommand split = {&c, index, i};
                commands.push_back(split);
                index++;
            }
        }
        return commands;
    }
private:
};
CommandGenerator s_CommandGenerator;
bool s_CommandChanged = false;


class MyButton
    : public glv::Button
{
private:
    glv::Label m_Label;
    std::function<void()> m_Callback;

public:
    MyButton(const glv::WideString& label, std::function<void()> callback, const glv::Rect& r, bool center = true, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT
        : glv::Button(r, true)
        , m_Label(label, glv::Label::Spec(center ? glv::Place::CC : glv::Place::CL, 0.0f, 0.0f, 15.f))
        , m_Callback(callback)
    {
        this->anchor(anchor);
        *this << m_Label;
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);
        attach([](const glv::Notification& n)->void { n.receiver< MyButton >()->m_Callback(); }, glv::Update::Clicked, this);
    }
    void SetLabel(const char* string) NN_NOEXCEPT
    {
        glv::WideString ws;
        m_Label.setValue(glv::CopyWithConvertWideString(ws, string));
    }
};

class SubCommandDropDown : public devmenu::DropDownBase
{
public:
    SubCommandDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT : devmenu::DropDownBase( rect, textSize )
    {
        mItemList.font().size(textSize);
        attach( []( const glv::Notification& n )->void { n.receiver< SubCommandDropDown >()->UpdateValue(); }, glv::Update::Action, this );
    }
    void RegisterCommands(const SplittedCommand& command) NN_NOEXCEPT
    {
        // 要素を一度消したい
        mItems.clear();
        for (int i = 0; i < DisplaySubCommandsPerCommand && i + command.subCommandOffset < static_cast<int>(command.command->subcommands.size()); ++i)
        {
            auto& c = command.command->subcommands[i + command.subCommandOffset];
            addItem( c.c_str() );
        }
    }
    void UpdateValue() NN_NOEXCEPT
    {
        s_CommandChanged = true;
    }
};


class MainCommandDropDown : public devmenu::DropDownBase
{
public:
    MainCommandDropDown( const glv::Rect& rect, float textSize, SubCommandDropDown* subCommand ) NN_NOEXCEPT : devmenu::DropDownBase( rect, textSize )
    {
        mItemList.font().size(textSize);
        m_SubCommand = subCommand;
        attach( []( const glv::Notification& n )->void { n.receiver< MainCommandDropDown >()->UpdateValue(); }, glv::Update::Action, this );
    }
    void RegisterCommands(const std::vector<SplittedCommand>& commands) NN_NOEXCEPT
    {
        m_Commands = &commands;
        for(auto& c : *m_Commands)
        {
            char buffer[64];
            snprintf(buffer, sizeof(buffer), "%s (%d)", c.command->commandName, c.splitIndex + 1);
            addItem(buffer);
        }
        m_SubCommand->RegisterCommands((*m_Commands)[0]);
    }
    void UpdateValue() NN_NOEXCEPT
    {
        m_SubCommand->RegisterCommands((*m_Commands)[mSelectedItem]);
        s_CommandChanged = true;
    }
private:
    SubCommandDropDown* m_SubCommand;
    const std::vector<SplittedCommand>* m_Commands;
};

class CommandPage;
CommandPage* s_Command;

class CommandPage : public Page
{
public:
    CommandPage(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT : Page(pageId, pageCaption, rect), m_Scroll(0)
    {
        m_Operate = new glv::Table("xxxxxxx", 10, 1);
        m_Up = new MyButton(GLV_TEXT_API_WIDE_STRING("Up"), [&] {Scroll(-5);}, glv::Rect(50, 30));
        m_Down = new MyButton(GLV_TEXT_API_WIDE_STRING("Down"), [&] {Scroll(5); }, glv::Rect(70, 30));
        m_ToHead = new MyButton(GLV_TEXT_API_WIDE_STRING("Head"), [&] {ScrollToHead(); }, glv::Rect(70, 30));
        m_SubCommand = new SubCommandDropDown(glv::Rect(140, 30), 15.f);
        m_MainCommand = new MainCommandDropDown(glv::Rect(140, 30), 15.f, m_SubCommand);
        m_Command = new MyButton(GLV_TEXT_API_WIDE_STRING(""), [&] {EditCommand(); }, glv::Rect(400, 30), false);
        m_Exec = new MyButton(GLV_TEXT_API_WIDE_STRING("Exec"), [&] {ExecuteCommand(); }, glv::Rect(70, 30));
        *m_Operate << m_Up << m_Down << m_ToHead << m_MainCommand << m_SubCommand << m_Command << m_Exec;
        m_Operate->arrange();

        glv::WideString blank = GLV_TEXT_API_WIDE_STRING("");
        m_Table = new glv::Table("<", 3, 1);
        *m_Table << m_Operate;
        for (int i = 0; i < LabelCount; ++i)
        {
            m_Labels[i] = new glv::Label(blank, glv::Label::Spec(glv::Place::CL, 0, 0, 15.f));
            *m_Table << m_Labels[i];
        }
        m_Table->arrange().fit(false);
        *this << m_Table;
        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        Refresh();
        s_Command = this;
        RegisterLogCallback(Listen);
        RegisterVLogCallback(ListenV);
        m_Commands = s_CommandGenerator.ParseHelpAndGetCommands();
        m_MainCommand->RegisterCommands(m_Commands);
        OnCommandChanged();
    }
    void Scroll(int n) NN_NOEXCEPT
    {
        s_FullLog.Scroll(n);
        Refresh();
    }
    void ScrollToHead() NN_NOEXCEPT
    {
        s_FullLog.ScrollToHead();
        Refresh();
    }
    void ExecuteCommand() NN_NOEXCEPT
    {
        Listen("> %s\n", m_CurrentCommand);
        bool isSuccess;
        auto result = s_CommandGenerator.Execute(&isSuccess, m_CurrentCommand);
        Listen("%s, (%08x)\n", (result.IsSuccess() && isSuccess) ? "[SUCCESS]" : "[FAILED]", result.GetInnerValueForDebug());
        Listen("\n");
    }

    void Refresh() NN_NOEXCEPT
    {
        for (int i = 0; i < LabelCount; ++i)
        {
            m_Labels[i]->setValue(s_FullLog.GetLog(LabelCount - i - 1));
        }
    }
    static void Listen(const char* format, ...) NN_NOEXCEPT
    {
        va_list arglist;
        va_start( arglist, format );
        ListenV(format, arglist);
        va_end( arglist );
    }
    static void ListenV(const char* format, std::va_list argList) NN_NOEXCEPT
    {
        static char buffer[16 << 10];
        vsnprintf(buffer, sizeof(buffer), format, argList);
        s_FullLog.ParseNewLineAndRegister(buffer);
        s_Command->Refresh();
    }
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        // 1sec で、最大値倒して 60 行ずれるようにする
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        m_Scroll -= (dpad.GetStickPositionRight().y * 2);

        if( true == dpad.HasAnyEvent() && dpad.IsButtonDown<nn::hid::DebugPadButton::Y>())
        {
            ScrollToHead();
        }

        if( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad(0);
            m_Scroll -= (bpad.GetStickPositionRight().y * 2);
            if( true == bpad.HasAnyEvent() && bpad.IsButtonDown<glv::BasicPadEventType::Button::Y>())
            {
                ScrollToHead();
            }
        }
        while(m_Scroll < -1.0)
        {
            Scroll(-1);
            m_Scroll += 1.0;
        }
        while(m_Scroll > 1.0)
        {
            Scroll(1);
            m_Scroll -= 1.0;
        }
        if(s_CommandChanged)
        {
            OnCommandChanged();
            s_CommandChanged = false;
        }
    }
    void OnCommandChanged() NN_NOEXCEPT
    {
        auto& command = m_Commands[m_MainCommand->selectedItem()];
        auto& mainCommand = *(command.command);
        auto& subcommand = mainCommand.subcommands[command.subCommandOffset + m_SubCommand->selectedItem()];
        snprintf(m_CurrentCommand, sizeof(m_CurrentCommand), "%s %s", mainCommand.commandName, subcommand.c_str());

        m_Command->SetLabel(m_CurrentCommand);
    }
    void EditCommand() NN_NOEXCEPT
    {
        // 現時点の swkbd ライブラリでは、swkbd を呼び出せない
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        launcher::KeyboardConfig keyboardConfig;
        keyboardConfig.preset = launcher::KeyboardConfig::Preset::Preset_Default;
        keyboardConfig.keyboardMode = launcher::KeyboardConfig::KeyboardMode::KeyboardMode_Ascii;
        keyboardConfig.isPredictionEnabled = false;
        keyboardConfig.isUseUtf8 = true;
        keyboardConfig.isUseNewLine = true;
        keyboardConfig.textMaxLength = 0;
        keyboardConfig.textMinLength = 0;
        keyboardConfig.headerTextUtf8 = nullptr;
        keyboardConfig.guideTextUtf8 = nullptr;

        auto result = launcher::LaunchSoftwareKeyboardAndGetString(
            m_CurrentCommand, sizeof(m_CurrentCommand), m_CurrentCommand, keyboardConfig);
        if (result.IsSuccess())
        {
            m_Command->SetLabel(m_CurrentCommand);
        }
#endif
    }

private:
    float m_Scroll;
    glv::Label* m_Labels[LabelCount];
    glv::Table* m_Table;
    glv::Table* m_Operate;
    MyButton* m_Up;
    MyButton* m_Down;
    MyButton* m_ToHead;
    MyButton* m_Command;
    glv::Button* m_Exec;
    MainCommandDropDown* m_MainCommand;
    SubCommandDropDown* m_SubCommand;
    char m_CurrentCommand[1024];
    std::vector<SplittedCommand> m_Commands;
};

devmenu::PageCreatorImpl< CommandPage > g_CommandCreator( DevMenuPageId_Command, "Command" );

}}} // ~namespace devmenu::system::command, ~namespace devmenu::system, ~namespace devmenu
