﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_GameCard.h> // Show game card image hash
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/settings/system/settings_Region.h>

#include "DevMenu_ApplicationCatalog.h"
#include "DevMenu_ApplicationInstaller.h"
#if !defined( NN_DEVMENULOTCHECK )
    #include "DevMenu_ApplicationWriter.h"
#endif
#include "DevMenu_DeleteContents.h"
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    #include "DevMenu_Exhibition.h"
#endif
#include "Accounts/DevMenu_AccountsSelector.h"
#include "Applications/DevMenu_ApplicationsCommon.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "Launcher/DevMenu_Launcher.h"
#include "Launcher/DevMenu_LauncherLibraryAppletApis.h"

#if defined( NN_DEVMENULOTCHECK )
    #include "QrCodeEncoder/DevMenu_CreateQrCode.h"
    #include <nn/ec/system/ec_TicketSystemApi.h>
    #include <nn/fs/fs_RightsId.h>
#endif

namespace devmenu { namespace application {

namespace {

    // 参考: util_CharacterEncoding.cpp

    // UTF-8 の 1 バイト目から 1 コードポイントを何バイトで表現するかを
    // 引くためのテーブル
    const int8_t Utf8NbytesInnerTable[ 257 ] =
    {
        -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x00
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x10
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x20
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x30
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x40
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x50
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x60
        1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x70
        0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0x80
        0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0x90
        0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0xA0
        0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0xB0
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,     // 0xC0
        2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,     // 0xD0
        3,  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,     // 0xE0
        4,  4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 7, 8,     // 0xF0
    };
    const char* const Utf8NbytesTable = reinterpret_cast< const char* >( &Utf8NbytesInnerTable[ 1 ] );

    bool IsInitialByte( const unsigned char* ptr, int size ) NN_NOEXCEPT
    {
        uint32_t character;

        switch( size )
        {
        case 1:
            // 区別がつかない
            break;
        case 2:
            if ( ( *ptr & 0x1E ) != 0 )
            {
                if ( ( *( ptr + 1 ) & 0xC0 ) == 0x80 )
                {
                    // 先頭文字
                    return true;
                }
                // エラー: 上位 2 bit は 10 である必要がある
            }
            // エラー: 冗長表現はエラー
            break;
        case 3:
            if ( ( *( ptr + 1 ) & 0xC0 ) == 0x80 && ( *( ptr + 2 ) & 0xC0 ) == 0x80 )
            {
                character = ( ( *ptr & 0x0F ) << 12 ) | ( ( *( ptr + 1 ) & 0x3F ) << 6 ) | ( *( ptr + 2 ) & 0x3F );
                // U+800 未満と U+D800-U+DFFF(サロゲートコード)を弾いている
                if ( character & 0xF800 && ( character & 0xF800 ) != 0xD800 )
                {
                    // 先頭文字
                    return true;
                }
                // エラー: 冗長表現やサロゲートコードはエラー
            }
            // エラー: 上位 2 bit は 10 である必要がある
            break;
        case 4:
            if ( ( *( ptr + 1 ) & 0xC0 ) == 0x80 && ( *( ptr + 2 ) & 0xC0 ) == 0x80 && ( *( ptr + 3 ) & 0xC0 ) == 0x80 )
            {
                character = ( ( *ptr & 0x07 ) << 18 ) | ( ( *( ptr + 1 ) & 0x3F ) << 12 ) | ( ( *( ptr + 2 ) & 0x3F ) << 6 ) | ( *( ptr + 3 ) & 0x3F );
                if ( character >= 0x10000 && character < 0x110000 )
                {
                    // 先頭文字
                    return true;
                }
                // エラー: 0x110000 以降はエラー
                // エラー: 冗長表現はエラー
            }
            // エラー: 上位 2 bit は 10 である必要がある
        default:
            break;
        }
        return false;
    }

    bool CutUtf8String( char* pOut, const char* pSrc, size_t size ) NN_NOEXCEPT
    {
        const unsigned char* ptr = nullptr;
        uint32_t character = 0;

        if ( size > strlen( pSrc ) )
        {
            nn::util::Strlcpy( pOut, pSrc, size );
            return false;
        }

        for ( auto i = 4; i >= 1; ) // 最大 1 文字 4 バイトで考慮すればいい前提を置く
        {
            ptr = reinterpret_cast< const unsigned char* >( pSrc + size - i );
            character = *ptr;
            const auto bytes = static_cast< int >( Utf8NbytesTable[ character ] );

            if ( i == 1 )
            {
                nn::util::Strlcpy( pOut, pSrc, size ); // 最後尾のバイトを NULL 終端
                return true;
            }

            if ( !IsInitialByte( ptr, bytes ) )  // 文字の先頭バイトでなければ次へ
            {
                i--;
                continue;
            }

            if ( i <= bytes )
            {
                nn::util::Strlcpy( pOut, pSrc, size - i + 1 ); // NULL 終端を考慮
                return true;
            }

            i -= bytes;
        }
        NN_ABORT( "Must not come here." );
    }

    bool IsTaskSuspended( const nn::ns::ApplicationView& view ) NN_NOEXCEPT
    {
        const auto downloadState = view.progress.state;
        const auto applyState = view.applyProgress.state;

        if ( downloadState == nn::ns::ApplicationDownloadState::Suspended || downloadState == nn::ns::ApplicationDownloadState::NotEnoughSpace
            || applyState == nn::ns::ApplicationApplyDeltaState::Suspended || applyState == nn::ns::ApplicationApplyDeltaState::NotEnoughSpace )
        {
            return true;
        }

        return false;
    }

    const nn::Result GetGameCardImageHash( char* outHash ) NN_NOEXCEPT
    {
        nn::fs::GameCardHandle handle;

        NN_RESULT_DO( nn::fs::GetGameCardHandle( &handle ) );
        NN_RESULT_DO( nn::fs::GetGameCardImageHash( outHash, nn::fs::GameCardImageHashSize, handle ) );
        NN_RESULT_SUCCESS;
    }

}

ApplicationListModel::ApplicationListModel() NN_NOEXCEPT :
    m_UpdateMutex( false ),
    m_RefreshedCount( 0LL ),
    m_IsCanceled( false )
{
}

const nn::Result ApplicationListModel::GetApplicationView( nn::ns::ApplicationView* pOutView, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    const auto result = nn::ns::GetApplicationView( pOutView, &applicationId, 1 );

    if ( result.IsFailure() )
    {
        DEVMENU_LOG( "GetApplicationView 0x%016llx failed as 0x%08x\n", applicationId.value, result.GetInnerValueForDebug() );
        return result;
    }

    NN_RESULT_SUCCESS;
}

const nn::Result ApplicationListModel::GetApplicationControlProperty( nn::util::optional<ControlProperty>* pOutProperty, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    NN_RESULT_DO( GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, applicationId ) );
    nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
    auto& applicationControlProperty = accessor.GetProperty();
    ControlProperty property = {};
    const size_t maxDisplayedNameLength = sizeof( property.name );
    property.isNameCut = CutUtf8String( property.name, applicationControlProperty.GetDefaultTitle().name, maxDisplayedNameLength );
    nn::util::Strlcpy( property.displayVersion, applicationControlProperty.displayVersion, sizeof( property.displayVersion ) );
    *pOutProperty = property;

    NN_RESULT_SUCCESS;
}

ApplicationListModel::ExitReason ApplicationListModel::Refresh() NN_NOEXCEPT
{
    std::vector< nn::ns::ApplicationRecord > recordList( 2048 );
    const auto count = nn::ns::ListApplicationRecord( recordList.data(), static_cast< int >( recordList.size() ), 0 );
    recordList.reserve( static_cast< size_t >( count ) );

    std::vector< Item > itemList( static_cast< size_t >( count ) );

    struct PropertyRequiredItemInfo
    {
        int index;
        nn::ncm::ApplicationId id;
    };
    std::vector< PropertyRequiredItemInfo > propertyRequiredItemInfoList;

    for ( size_t i = 0; i < itemList.size(); ++i )
    {
        if ( m_IsCanceled )
        {
            break;
        }

        nn::ns::ApplicationView view;
        if ( GetApplicationView( &view, recordList[ i ].id ).IsFailure() )
        {
            // 外部でアンインストールが連続で行われた時などで失敗する場合がある
            // 次回 Refresh 時にリストが修正される
            return ExitReason_Normal;
        }
        itemList[ i ].view = view;

        auto previousItem = Find( view.id );
        if ( previousItem && previousItem->property && previousItem->view.version == view.version )
        {
            // プロパティ取得済みなら引き継ぎ
            itemList[ i ].property = *previousItem->property;
        }
        else
        {
            propertyRequiredItemInfoList.push_back( { static_cast< int >( i ), itemList[ i ].view.id } );
        }
    }

    // コンテンツの存在を画面に反映させるために、ApplicationView 取得完了時点で ItemList に反映させる
    {
        std::lock_guard< nn::os::Mutex > guard( m_UpdateMutex );
        m_ItemList = std::move( itemList );
        m_RefreshedCount++;
    }

    for ( const auto& info : propertyRequiredItemInfoList )
    {
        if ( m_IsCanceled )
        {
            // property 取得完了前に Cancel された場合、N/A 表示になる
            return ExitReason_Canceled;
        }

        nn::util::optional< ControlProperty > property;

        if ( GetApplicationControlProperty( &property, info.id ).IsSuccess() && nn::util::nullopt != property )
        {
            std::lock_guard< nn::os::Mutex > guard( m_UpdateMutex );
            m_ItemList[ info.index ].property = *property;
        }
    }
    return ExitReason_Normal;
}

ApplicationListModel::ExitReason ApplicationListModel::UpdateIfNecessary() NN_NOEXCEPT
{
    for ( auto& item : m_ItemList )
    {
        if ( m_IsCanceled )
        {
            return ExitReason_Canceled;
        }

        const bool isAddOnContentOnly = item.view.HasAddOnContentRecord() && !item.view.HasPatchRecord() && !item.view.HasMainRecord();

        // 追加コンテンツのみだと property が常に nullopt になるので対象外にする
        if ( ( !item.property && !isAddOnContentOnly ) || item.view.IsDownloading() || item.view.IsApplyingDelta() )
        {
            UpdateOne( &item );
        }
    }
    return ExitReason_Normal;
}

void ApplicationListModel::UpdateOne( Item* pItem ) NN_NOEXCEPT
{
    nn::ns::ApplicationView view;

    if ( GetApplicationView( &view, pItem->view.id ).IsSuccess() )
    {
        std::lock_guard< nn::os::Mutex > guard( m_UpdateMutex );
        if ( pItem->view.version != view.version )
        {
            pItem->requiresUpdateControl = true;
        }
        pItem->view = view;
    }

    if ( !pItem->property || pItem->requiresUpdateControl )
    {
        std::lock_guard< nn::os::Mutex > guard( m_UpdateMutex );
        pItem->requiresUpdateControl = false;
        GetApplicationControlProperty( &pItem->property, view.id );
    }
}

int64_t ApplicationListModel::GetRefreshedCount() const NN_NOEXCEPT
{
    return m_RefreshedCount;
}

nn::util::optional< ApplicationListModel::Item > ApplicationListModel::Find( const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT
{
    for ( const auto& item : m_ItemList )
    {
        if ( item.view.id == applicationId )
        {
            return item;
        }
    }

    return nn::util::nullopt;
}

const std::vector< ApplicationListModel::Item >& ApplicationListModel::Get() NN_NOEXCEPT
{
    std::lock_guard< nn::os::Mutex > guard( m_UpdateMutex );
    return m_ItemList;
}

void ApplicationListModel::ResetCancel() NN_NOEXCEPT
{
    m_IsCanceled = false;
}

void ApplicationListModel::Cancel() NN_NOEXCEPT
{
    m_IsCanceled = true;
}

/*********************************
 * class CatalogScene
 *********************************/

const glv::space_t CatalogScene::HeaderRegion    = 40.0f;
const glv::space_t CatalogScene::FooterRegion    = 90.0f;
const glv::space_t CatalogScene::ListMarginLeft  = 16.0f;
const glv::space_t CatalogScene::ListMarginRight = 16.0f;

/**
 * @brief コンストラクタ
 */
CatalogScene::CatalogScene( const AbstractOperators& op, ApplicationCatalogPage* pParentPage, glv::Rect rect, InstallScene* pInstallScene ) NN_NOEXCEPT
    : Scene( pParentPage, rect, true )
    , m_Op( op )
    , m_pParentPage( pParentPage )
    , m_pInstallScene( pInstallScene )
#if !defined( NN_DEVMENULOTCHECK )
    , m_pWriteScene( nullptr )
#endif
    , m_ModelRefreshedCount()
    , m_StopEvent( nn::os::EventClearMode_ManualClear )
    , m_IsUpdateThreadRunning( false )
    , m_IsUpdateOfApplicationListEnabled( false )
    , m_IsRefreshOfApplicationListRequired( true )
    , m_pInstallSceneButton( nullptr )
#if !defined( NN_DEVMENULOTCHECK )
    , m_pWriteSceneButton( nullptr )
#endif
    , m_pContainer( nullptr )
    , m_pLabelNoItem( nullptr )
    , m_Footer( this, glv::Rect( width() - ( CatalogScene::ListMarginLeft + CatalogScene::ListMarginRight ), CatalogScene::FooterRegion ) )
    , m_ListUserActionCallback( nullptr )
    , m_IsNotifiedByTouch( true )
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;

    // ヘッダ
    auto pHeader = new glv::Group( glv::Rect( width() , CatalogScene::HeaderRegion ) );
    {
        // インストールシーン遷移ボタン
        const glv::space_t posButtonTop = 8.0f;
        m_pInstallSceneButton = new devmenu::Button( "Install from SD", [&] { SwitchToInstallScene(); }, CommonValue::InitialFontSize, 24.0f );
        m_pInstallSceneButton->anchor( glv::Place::TL ).pos( CatalogScene::ListMarginLeft, posButtonTop );
        m_pInstallSceneButton->enable( focusableProperty );
        *pHeader << m_pInstallSceneButton;

#if !defined( NN_DEVMENULOTCHECK )
        // 書き込みシーン遷移ボタン
        m_pWriteSceneButton = new devmenu::Button( "Write to Game Card", [&] { SwitchToWriteScene(); }, CommonValue::InitialFontSize, 24.0f );
        m_pWriteSceneButton->anchor( glv::Place::TL ).pos( m_pInstallSceneButton->right() + 16.0f, posButtonTop );
        m_pWriteSceneButton->enable( focusableProperty );
        *pHeader << m_pWriteSceneButton;
#endif

#if defined( NN_DEVMENULOTCHECK )
        // チケットインストールボタン
        auto pInstallTicketButton = new devmenu::Button( "Install Ticket", [&] { DisplayInstallTicketDialog(); }, CommonValue::InitialFontSize, 24.0f );
        pInstallTicketButton->anchor( glv::Place::TL ).pos( m_pInstallSceneButton->right() + CatalogScene::ListMarginRight, posButtonTop );
        pInstallTicketButton->enable( focusableProperty );
        *pHeader << pInstallTicketButton;

        // ゲームカードハッシュ表示ボタン
        auto pGameCardHashInfoButton = new devmenu::Button( "GC Image Hash", [&] { ShowGameCardImageHash(); }, CommonValue::InitialFontSize, 24.0f );
        pGameCardHashInfoButton->anchor( glv::Place::TL ).pos( pInstallTicketButton->right() + CatalogScene::ListMarginRight, posButtonTop );
        pGameCardHashInfoButton->enable( focusableProperty );
        *pHeader << pGameCardHashInfoButton;
#endif
        *this << pHeader;
    }

    // Adding a scrollable container
    m_pContainer = new glv::View( glv::Rect( ListMarginLeft, pHeader->height() + 30.f, width() - ListMarginLeft, height() - ( pHeader->height() + FooterRegion + 60.f ) ) );
    m_pContainer->disable( glv::Property::DrawBack | glv::Property::FocusHighlight | glv::Property::DrawBorder );

    *this << m_pContainer;
    m_pScrollContainer = new ScrollableBoxView( "<", glv::Rect( 0.0f, 0.0f, m_pContainer->width(), m_pContainer->height() ) );
    const auto innerRect = m_pScrollContainer->GetInnerRect();

    // リストビュー  -  Application list view
    *m_pContainer << *m_pScrollContainer;

    // Initialize the vector list that will hold app information for displaying application tiles.
    // These tiles get re used by different items, and their information gets updated to match the visible item.
    for ( int i = 0; i < MaxNumberOfVisibleApplicationTiles; ++i )
    {
        m_ApplicationTilesList.push_back( new ApplicationTileView( innerRect.width(), 80.0f ) );
    }

    // アイテムなしメッセージ
    auto pLabelNoItem = new glv::Label( "No applications are installed.", glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, 32.0f ) );
    {
        *m_pContainer << pLabelNoItem;
        m_pLabelNoItem = pLabelNoItem;
    }

    // フッタ
    m_Footer.IncreaseLaunchButtonSize( 10.0f, 0.0f );
    *this << m_Footer;

    this->SetFirstFocusTargetView( m_pInstallSceneButton );

    GetFixedSizeFirmwareDebugSettingsItemValue( &m_IsUpdateOfApplicationListEnabled, "devmenu", "enable_application_update" );

    // ae メッセージの通知を有効にする
    this->EnableNotificationMessageReceiving();
}

void CatalogScene::SwitchToInstallScene() NN_NOEXCEPT
{
    const auto result = m_pInstallScene->MakeNspList();

    if ( result.IsFailure() )
    {
        auto pView = new MessageView( true );
        std::string errorMessage;

        if ( nn::fs::ResultPortSdCardNoDevice::Includes( result ) )
        {
            errorMessage = "SD card is not inserted.";
        }
        else
        {
            // 基本的に nn::fs::ResultSdCardAccessFailed を想定している
            errorMessage = "Failed to access to the SD card.";
        }

        devmenu::CreateErrorMessageView( pView, errorMessage, result );
        StartModal( pView, true );
        return;
    }

    m_pParentPage->SwitchScene( this, m_pInstallScene, false );
}

void CatalogScene::SwitchToWriteScene() NN_NOEXCEPT
{
#if !defined( NN_DEVMENULOTCHECK )
    m_pParentPage->SwitchScene( this, m_pWriteScene, false );
#endif
}

void CatalogScene::ShowGameCardImageHash() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    char gameCardHash[ nn::fs::GameCardImageHashSize ];

    if ( false == nn::fs::IsGameCardInserted() )
    {
        pView->AddMessage( "[Attached Game Card Image Hash]\n   Game card is not inserted." );
    }
    else
    {
        auto result = GetGameCardImageHash( gameCardHash );
        if ( result.IsFailure() )
        {
            char errorValueStr[ 64 ];
            nn::util::SNPrintf( errorValueStr, sizeof( errorValueStr ), "(result: 0x%08x)", result.GetInnerValueForDebug() );
            const std::string errorMessage = "[Attached Game Card Image Hash]\n  Failed to get hash value. " + std::string( errorValueStr );
            DEVMENU_LOG( "%s\n", errorMessage.c_str() );
            pView->AddMessage( errorMessage );
        }
        else
        {
            const std::string titleStr( "[Attached Game Card Image Hash]\n  " );
            std::string displayHashStr, qrCodeHashStr;
            GetHashString( &displayHashStr, &qrCodeHashStr, gameCardHash, sizeof( gameCardHash ) );
            pView->AddMessage( titleStr + displayHashStr );

#if defined( NN_DEVMENULOTCHECK )

            mw::qre::ImageInfo imageInfo = { 0, 0, nullptr, mw::qre::Rect() };
            uint32_t imageSize = 0;
            const auto hashSize = qrCodeHashStr.size();
            std::unique_ptr< uint8_t[] > hashData( new uint8_t[ hashSize ] );
            memcpy( hashData.get(), qrCodeHashStr.c_str(), hashSize );
            devmenu::CreateQrCode( &imageInfo, &imageSize, hashData.get(), hashSize );
            pView->AddImage( imageInfo.rgbData, imageSize, imageInfo.width, imageInfo.height );
            delete[] imageInfo.rgbData;

#endif // defined( NN_DEVMENULOTCHECK )
        }
    }
    pView->AddButton( "Close" );
    StartModal( pView, true );
}


void CatalogScene::UpdateStorageSize( bool isSdCardCheckRequired, const nn::Result& checkResult ) NN_NOEXCEPT
{
    m_Footer.UpdateStorageSize( isSdCardCheckRequired, checkResult );
}

glv::View* CatalogScene::GetFocusableChild() NN_NOEXCEPT
{
    return ( nullptr == m_pScrollContainer->child )
    ? static_cast< glv::View* >( m_pInstallSceneButton )
    : static_cast< glv::View* >( m_pScrollContainer->child );
}

void CatalogScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );

    m_ListItemCopy = m_Model.Get();

    auto refreshedCount = m_Model.GetRefreshedCount();
    if ( refreshedCount != m_ModelRefreshedCount && false == this->GetRootSurfaceContext()->IsModalViewRunning() )
    {
        if ( m_ListItemCopy.empty() )
        {
            m_pLabelNoItem->enable( glv::Property::t::Visible );
            m_pLabelNoItem->bringToFront();
            m_Footer.disable( glv::Property::Visible );
        }
        else
        {
            m_pLabelNoItem->disable( glv::Property::t::Visible );
            m_Footer.enable( glv::Property::Visible );
        }

        // Update the count
        m_ModelRefreshedCount = refreshedCount;

        // Refresh the Application List Scroll Container
        ApplyApplicationInfoToList();
    }

    // Pad の状態を確認してユーザーアクションのコールバック関数を決定します。
    {
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        const int count = events.GetAvailableBasicPadCount();
        bool isPlusMinusButtonPressed = false;
        bool isAButtonPressed = false;
        bool isYButtonPressed = false;

        if ( true == dpad.HasAnyEvent() )
        {
            isPlusMinusButtonPressed = dpad.IsButtonUp< nn::hid::DebugPadButton::Start >() || dpad.IsButtonUp< nn::hid::DebugPadButton::Select >();
            isAButtonPressed = dpad.IsButtonUp< nn::hid::DebugPadButton::A >();
            isYButtonPressed = dpad.IsButtonUp< nn::hid::DebugPadButton::Y >(); // exhibition mode 時のみ検知可能
        }
        else if ( count > 0 )
        {
            for ( int i = 0; i < count; ++i )
            {
                const glv::BasicPadEventType& bpad = events.GetBasicPad( i );
                isPlusMinusButtonPressed |= bpad.IsButtonUp< glv::BasicPadEventType::Button::Start >() | bpad.IsButtonUp< glv::BasicPadEventType::Button::Select >();
                isAButtonPressed |= bpad.IsButtonUp< glv::BasicPadEventType::Button::A >();
                isYButtonPressed |= bpad.IsButtonUp< glv::BasicPadEventType::Button::Y >();  // exhibition mode 時のみ検知可能
            }
        }

        if ( isPlusMinusButtonPressed )
        {
            m_ListUserActionCallback = [&](){ m_Op.OpenApplicationInfo(); };
        }
        else if ( isAButtonPressed )
        {
            m_ListUserActionCallback = [&](){ m_Op.SelectApplication(); };
        }
        else if ( isYButtonPressed )
        {
            m_ListUserActionCallback = [&](){ m_Op.OpenAutoBootSetting(); };
        }
        else
        {
            return; // タッチ操作でのユーザーアクションを想定して何もしない
        }
        m_IsNotifiedByTouch = false;
    }
}

void CatalogScene::ApplyApplicationInfoToList() NN_NOEXCEPT
{
    // Reset the selected application property
    m_pSelectedApplicationProperty = nullptr;

    // Refresh the Application Tile List
    m_pScrollContainer->Refresh();

    const auto innerRect = m_pScrollContainer->GetInnerRect();
    int tileListIndex = 0;
    int applicationCountOnList = 0;

    // Loop through applications
    for ( const auto& item : m_ListItemCopy )
    {
        if ( applicationCountOnList >= MaxNumberOfApplicationsSupported )
        {
            // Reached the limit of supported apps on the list, break out of the for loop
            break;
        }

        // Create container for this application
        TileContainer* pTileContainer = new TileContainer( this, innerRect.width(), 80.0f, item, m_ApplicationTilesList.at( tileListIndex ) );

        // Add notification event
        pTileContainer->attach( OnApplicationTileViewClickNotification, glv::Update::Clicked, this );

        // Add container to the scrollable list
        *m_pScrollContainer << pTileContainer;

        // Check if this application id was the last ID that had been focused, so we can reset the focus correctly
        if ( m_SelectedApplicationId == item.view.id )
        {
            // This was the last application that had focus. Reset the page focus to this element if the list is visible
            if ( enabled( glv::Property::Visible ) )
            {
                this->SetFocus( pTileContainer );
                SetLastFocusedView( pTileContainer );
            }
        }

        // Update list index and total amount of apps added to the list
        tileListIndex++;
        applicationCountOnList++;
        if ( MaxNumberOfVisibleApplicationTiles <= tileListIndex )
        {
            // Reached the end of the app vector, reset index
            tileListIndex = 0;
        }
    }
}

/**
 * @brief アプリケーションメインループからのコールバックです。
 *
 * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
 */
void CatalogScene::OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}

void CatalogScene::OnChangeIntoBackground() NN_NOEXCEPT
{
    // Update スレッドは Application ページ外での処理が必要なので OnNotificationMessageReceived() で対応
}

void CatalogScene::OnChangeIntoForeground() NN_NOEXCEPT
{
    // Update スレッドは Application ページ外での処理が必要なので OnNotificationMessageReceived() で対応
}

void CatalogScene::Refresh() NN_NOEXCEPT
{
#if !defined( NN_DEVMENULOTCHECK )
    m_pWriteScene->ClearScreenText();
#endif
}

void CatalogScene::OnApplicationTileViewClickNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    const auto self = notification.receiver< CatalogScene >();
    if ( self->m_IsNotifiedByTouch )
    {
        if ( self->m_SelectedApplicationId == self->m_PreviousSelectedApplicationId )
        {
            self->m_ListUserActionCallback = [&](){ self->m_Op.SelectApplication(); };
        }
        self->m_PreviousSelectedApplicationId = self->m_SelectedApplicationId;
    }

    const std::function< void() > callback = self->m_ListUserActionCallback;
    if ( nullptr != callback )
    {
        callback();
        self->m_ListUserActionCallback = nullptr;
        self->m_IsNotifiedByTouch = true;
    }
}

bool CatalogScene::CanUpdateItemList() NN_NOEXCEPT
{
    return m_pParentPage->IsInFocusState();
}

void CatalogScene::UpdateApplicationListAndStorageSizeIfRequired() NN_NOEXCEPT
{
    if ( !m_IsUpdateOfApplicationListEnabled || !CanUpdateItemList() )
    {
        return;
    }

    ApplicationListModel::ExitReason reason = ApplicationListModel::ExitReason_Normal;
    NN_UTIL_SCOPE_EXIT
    {
        switch ( reason )
        {
        case ApplicationListModel::ExitReason_Normal:
            break;
        case ApplicationListModel::ExitReason_Canceled:
            // キャンセル時は後に更新可能になった時点で更新されるようにする
            m_IsRefreshOfApplicationListRequired = true;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    };

    // アプリケーション記録が更新された場合は全ての item を再取得する
    if ( m_IsRefreshOfApplicationListRequired )
    {
        m_IsRefreshOfApplicationListRequired = false;
        reason = m_Model.Refresh();
        UpdateStorageSize();
        return;
    }

    // それ以外のケースでは、更新が必要な item のみ更新する
    reason = m_Model.UpdateIfNecessary();
}

void CatalogScene::StartUpdateThread() NN_NOEXCEPT
{
    m_Model.ResetCancel();

    if ( !m_IsUpdateOfApplicationListEnabled )
    {
        return;
    }

    if ( m_IsUpdateThreadRunning )
    {
        return;
    }

    static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[ 16 * 1024 ];
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_UpdateThread, []( void* param )
    {
        nn::os::SystemEvent refreshEvent;
        nn::ns::GetApplicationRecordUpdateSystemEvent( &refreshEvent );

        nn::os::TimerEvent updateCheckTimer( nn::os::EventClearMode_ManualClear );
        updateCheckTimer.StartPeriodic( nn::TimeSpan::FromSeconds( 0 ), nn::TimeSpan::FromSeconds( 1 ) );

        auto pSelf = reinterpret_cast< CatalogScene* >( param );

        while ( NN_STATIC_CONDITION( true ) )
        {
            auto index = nn::os::WaitAny(
                pSelf->m_StopEvent.GetBase(),
                refreshEvent.GetBase(),
                updateCheckTimer.GetBase() );

            switch ( index )
            {
            case 0:
                {
                    pSelf->m_StopEvent.Clear();
                    return;
                }
                break;
            case 1:
                {
                    refreshEvent.Clear();
                    pSelf->m_IsRefreshOfApplicationListRequired = true;
                    pSelf->UpdateApplicationListAndStorageSizeIfRequired();
                }
                break;
            case 2:
                {
                    updateCheckTimer.Clear();
                    pSelf->UpdateApplicationListAndStorageSizeIfRequired();
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    }, this, s_Stack, sizeof( s_Stack ), nn::os::DefaultThreadPriority ) );
    nn::os::StartThread( &m_UpdateThread );
    m_IsUpdateThreadRunning = true;
}

void CatalogScene::StopUpdateThread() NN_NOEXCEPT
{
    if ( !m_IsUpdateOfApplicationListEnabled )
    {
        return;
    }

    m_StopEvent.Signal();
    m_Model.Cancel();
    nn::os::WaitThread( &m_UpdateThread );
    nn::os::DestroyThread( &m_UpdateThread );
    m_IsUpdateThreadRunning = false;
}

void CatalogScene::CancelUpdateOfApplicationList() NN_NOEXCEPT
{
    m_Model.Cancel();
}

void CatalogScene::OnNotificationMessageReceived( NotificationMessageReceiver::Message message ) NN_NOEXCEPT
{
    switch ( message )
    {
    case NotificationMessageReceiver::Message_ChangeIntoBackground:
        StopUpdateThread();
        break;
    case NotificationMessageReceiver::Message_ChangeIntoForeground:
        // BG でコンテンツのインストールやアンインストールをされるとリストが更新されないので、FG 遷移時に明示的に更新する
        // Note: System Applet として動作している場合は、BG 遷移時に更新スレッドを止めない選択肢もある。
        //       そうすれば基本的には FG 遷移時に必要なケースでのみリストを更新することができるが、
        //       DevMenuApp をはじめとするアプリケーション記録の更新イベントを取得するアプリケーションによって
        //       更新イベントをクリアされると、FG 遷移時に更新があったことを検知できない。
        {
            // ToDo: DevMenu が起動した LA からの戻り時は実行しない方がよいので要調整
            m_IsRefreshOfApplicationListRequired = true;
            StartUpdateThread();
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

#if defined( NN_DEVMENULOTCHECK )
void CatalogScene::DisplayInstallTicketDialog() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    pView->AddMessage( "Install all ticket?" );

    pView->AddButton( "Cancel" );

    pView->AddButton(
        "Install",
        [&](void* pParam, nn::TimeSpan& timespan)
        {
            nn::nifm::SubmitNetworkRequestAndWait();
            if (!nn::nifm::IsNetworkAvailable())
            {
                DEVMENU_LOG("Network is not available.\n");
                auto pView = new MessageView( true );
                pView->AddMessage("Network is not available.");
                pView->AddButton("Close");
                StartModal(pView, true, true);
                return;
            }

            // BuildInUser にインストールされているコンテンツのチケットをダウンロード
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::BuildInUser, nn::ncm::ContentMetaType::Application));
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::BuildInUser, nn::ncm::ContentMetaType::Patch));
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::BuildInUser, nn::ncm::ContentMetaType::AddOnContent));
            }
            // SdCard にインストールされているコンテンツのチケットをダウンロード
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::SdCard, nn::ncm::ContentMetaType::Application));
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::SdCard, nn::ncm::ContentMetaType::Patch));
                NN_ABORT_UNLESS_RESULT_SUCCESS(DownloadAllInstalledTicket(nn::ncm::StorageId::SdCard, nn::ncm::ContentMetaType::AddOnContent));
            }
        },
        nullptr,
        MessageView::ButtonTextColor::Green
    );

    StartModal( pView, true );
}

nn::Result CatalogScene::DownloadAllInstalledTicket(nn::ncm::StorageId storageId, nn::ncm::ContentMetaType contentMetaType) NN_NOEXCEPT
{
    const int maxKeys = 512;
    const int bufferSize = 16 << 10;
    std::unique_ptr<nn::ncm::ContentMetaKey[]> keys(new nn::ncm::ContentMetaKey[maxKeys]);
    std::unique_ptr<uint8_t[]> data(new uint8_t[bufferSize]);

    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_TRY(OpenContentMetaDatabase(&db, storageId))
        NN_RESULT_CATCH(nn::ncm::ResultContentMetaDatabaseNotActive)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    auto listcount = db.ListContentMeta(keys.get(), maxKeys, contentMetaType, nn::ncm::ContentMetaDatabase::AnyApplicationId);

    nn::ncm::ContentStorage storage;
    nn::ncm::OpenContentStorage(&storage, storageId);

    for (int i = 0; i < listcount.listed; ++i)
    {
        size_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(db.Get(&size, data.get(), bufferSize, keys[i]));
        nn::ncm::ContentMetaReader reader(data.get(), size);

        auto contentCount = reader.CountContent();
        for (int j = 0; j < contentCount; j++)
        {
            auto contentInfo = reader.GetContentInfo(j);

            nn::ncm::RightsId rightsId;
            nn::Result result = storage.GetRightsId(&rightsId, contentInfo->GetId());

            if (result.IsSuccess())
            {
                nn::es::RightsIdIncludingKeyId esRightsId = nn::es::RightsIdIncludingKeyId::Construct(rightsId.id);
                if (!esRightsId.IsExternalKey())
                {
                    continue;
                }

                nn::ec::system::AsyncResult async;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ec::system::RequestDownloadTicket(&async, esRightsId));

                nn::Result downloadTicketResult = async.Get();
                if (downloadTicketResult.IsFailure())
                {
                    const int RightsIdStringLength = 33;
                    char rightsIdString[RightsIdStringLength];

                    for (size_t k = 0; k < sizeof(nn::es::RightsIdIncludingKeyId); k++)
                    {
                        nn::util::SNPrintf(&rightsIdString[k * 2], 3, "%02x", esRightsId._data[k]);
                    }
                    rightsIdString[RightsIdStringLength - 1] = '\0';

                    if (nn::nim::ResultTicketNotFound::Includes(downloadTicketResult))
                    {
                        DEVMENU_LOG("Downloading ticket(0x%s) failed. Ticket not found.", rightsIdString);
                    }
                    else if (nn::nim::ResultDeviceAccountNotRegisteredForDownloadTicket::Includes(downloadTicketResult))
                    {
                        DEVMENU_LOG("Downloading ticket(0x%s) failed. Device account not registered.", rightsIdString);
                    }
                    else
                    {
                        DEVMENU_LOG("Downloading ticket(0x%s) failed. result = %0x8x .", rightsIdString, downloadTicketResult.GetInnerValueForDebug());
                    }
                }
            }
        }
    }

    NN_RESULT_SUCCESS;
};
#endif

void CatalogScene::SetSelectedApplicationProperty( const ApplicationListModel::Item& item ) NN_NOEXCEPT
{
    m_pSelectedApplicationProperty = &item;
}

void CatalogScene::SetSelectedApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    m_SelectedApplicationId = applicationId;
}

void CatalogScene::SetPreviousSelectedApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    m_PreviousSelectedApplicationId = applicationId;
}

void CatalogScene::UpdateFooterLaunchButtonText( const char* text ) NN_NOEXCEPT
{
    m_Footer.UpdateLaunchButtonText( text );
}

void CatalogScene::UpdateFooterLaunchButtonFocus( bool isEnabled ) NN_NOEXCEPT
{
    m_Footer.UpdateLaunchButtonFocus( isEnabled );
}

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

void CatalogScene::LaunchOrResumeApplication( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    launcher::SetManualLaunchSuccessCallback( [&]{ m_pScrollContainer->ScrollToTop(); } );
    launcher::RequestApplicationLaunchOrResume( applicationId, false );
}

#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

/*******************************************
 * class CatalogScene::ApplicationTileView
 *******************************************/
CatalogScene::ApplicationTileView::ApplicationTileView( glv::space_t width, glv::space_t height ) NN_NOEXCEPT
    : glv::View( glv::Rect( width, height ) )
    , m_ApplicationIcon( 64, 64, 7.0f, 8.0f )
    , m_pGameCardIcon( new devmenu::IconLabel( devmenu::IconCodePoint::GameCard, glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, 20.0f ) ) )
    , m_pApplicationName( new glv::Label( "N/A", glv::Label::Spec( glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize ) ) )
    , m_pApplicationState( new glv::Label( "N/A", glv::Label::Spec( glv::Place::TR, -10.0f, 10.0f, 16.0f ) ) )
    , m_pApplicationId( new glv::Label( "N/A" ) )
    , m_pApplicationVersion( new glv::Label( "N/A" ) )
    , m_pApplicationNameTable ( new glv::Table( "< < <", 0, 0 ) )
    , m_pInfoTable( new glv::Table( ". < < < - - - - -,"
                               "| | | < < < < < < ", 1.0f, 1.0f, glv::Rect( width, height ) ) )
    , m_Item( ApplicationListModel::Item() )
{
    // Create default labels
    auto pIdLabel = new glv::Label( "ID: " );
    auto pVersionLabel = new glv::Label( "Version: " );
    auto pSeparator = new glv::Label( " | " );
    auto pStartSpacer = new devmenu::Spacer( 8.0f, 16.0f );
    const glv::space_t startPos = 70.0f; // IconWidth + α
    auto pEndSpacer = new devmenu::Spacer( width - pIdLabel->width() - pVersionLabel->width() - pSeparator->width() - pStartSpacer->width() - startPos, 16.0f );

    // Add Game Card and Application name to a Group so that they can be added together to the Info Table
    auto pIconSpacer = new devmenu::Spacer( 8.0f, 16.0f );
    *m_pApplicationNameTable << m_pApplicationName << pIconSpacer << m_pGameCardIcon;
    m_pApplicationNameTable->arrange();
    m_pApplicationNameTable->disable( glv::Property::HitTest );

    // Add items to the layout table
    *m_pInfoTable << m_ApplicationIcon << pStartSpacer << m_pApplicationNameTable << pIdLabel << m_pApplicationId << pSeparator << pVersionLabel << m_pApplicationVersion << pEndSpacer;

    // Add table to the app view
    *this << m_pInfoTable << m_pApplicationState;
    m_pInfoTable->arrange().fit( false );
    m_pInfoTable->disable( glv::Property::HitTest );
    this->disable( glv::Property::DrawBorder | glv::Property::HitTest | glv::Property::Controllable | glv::Property::DrawBack );
}

char* CatalogScene::ApplicationTileView::GetApplicationStateString() NN_NOEXCEPT
{
    if ( launcher::GetActiveApplicationId() == m_Item.view.id )
    {
        // App is Running
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "Running" );
    }
    else if ( m_Item.view.IsDownloading() )
    {
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "%s %lld / %lld KB",
            GetApplicationDownloadStateString( m_Item.view.progress.state ),
            m_Item.view.progress.downloaded / 1024, m_Item.view.progress.total / 1024 );
    }
    else if ( m_Item.view.IsApplyingDelta() )
    {
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "%s %lld / %lld",
            GetApplyDeltaStateString( m_Item.view.applyProgress.state ),
            m_Item.view.applyProgress.applied, m_Item.view.applyProgress.total );
    }
    else if ( m_Item.view.IsGameCard() && !m_Item.view.HasGameCardEntity() )
    {
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "Missing Game Card" );
    }
    else if ( !m_Item.view.IsLaunchable() )
    {
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "Not Launchable" );
    }
    else
    {
        nn::util::SNPrintf( m_StateBuffer,
            sizeof( m_StateBuffer ),
            "" );
    }

    return m_StateBuffer;
}

void CatalogScene::ApplicationTileView::UpdateApplicationTileValues( const ApplicationListModel::Item& item ) NN_NOEXCEPT
{
    const auto& nextItem = item;
    const auto& currentItem = m_Item;

    NN_UTIL_SCOPE_EXIT
    {
        // Update the item member
        m_Item = nextItem;
    };

    if ( !nextItem.property )
    {
        m_pApplicationName->setValue( "N/A" );
        m_pApplicationId->setValue( "N/A" );
        m_pApplicationVersion->setValue( "N/A" );
        m_ApplicationIcon.ClearTexture();
        m_pGameCardIcon->disable( glv::Property::Visible );
        m_pInfoTable->arrange().fit( false );
        return;
    }

    char labelBuffer[ 128 ];

    // Get Application Title name
    if ( nextItem.property->isNameCut )
    {
        nn::util::SNPrintf( labelBuffer, sizeof( labelBuffer ), "%s...", nextItem.property->name );
    }
    else
    {
        nn::util::SNPrintf( labelBuffer, sizeof( labelBuffer ), "%s", nextItem.property->name );
    }
    const char* name = labelBuffer;
    m_pApplicationName->setValue( name );

    // Check if we need to display the Game Card Icon
    if ( nextItem.view.IsGameCard() )
    {
        if ( !nextItem.view.HasGameCardEntity() )
        {
            // Display missing Game Card Icon
            m_pGameCardIcon->SetValue( devmenu::IconCodePoint::GameCardMissing );
        }
        else
        {
            // Display Game Card Icon
            m_pGameCardIcon->SetValue( devmenu::IconCodePoint::GameCard );
        }
        // Make sure the Game Card Icon is visible
        m_pGameCardIcon->enable( glv::Property::Visible );
    }
    else
    {
        // Game Card Icon should not be visible
        m_pGameCardIcon->disable( glv::Property::Visible );
    }

    // If any values are same, skip the display updates.
    if ( currentItem.property && nextItem.view.version == currentItem.view.version && nextItem.view.id == currentItem.view.id &&
         0 == nn::util::Strncmp( nextItem.property->displayVersion, currentItem.property->displayVersion, sizeof( nextItem.property->displayVersion ) ) &&
         0 == nn::util::Strncmp( nextItem.property->name, currentItem.property->name, sizeof( nextItem.property->name ) ) )
    {
        return;
    }

    m_pApplicationNameTable->arrange();

    // Get Application ID
    nn::util::SNPrintf( labelBuffer, sizeof( labelBuffer ), "0x%016llx", nextItem.view.id.value );
    const char* applicationIdStr = labelBuffer;
    m_pApplicationId->setValue( applicationIdStr );

    // Get Application Version
    nn::util::SNPrintf( labelBuffer, sizeof( labelBuffer ), nextItem.property->displayVersion );
    const char* versionStr = labelBuffer;
    m_pApplicationVersion->setValue( versionStr );

    // Update Texture and Get Application Icon
    std::unique_ptr< char[] > buffer;
    size_t dataSize;
    const auto result = GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, nextItem.view.id );
    if ( result.IsSuccess() )
    {
        nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
        m_ApplicationIcon.SetTexture( accessor.GetIconData(), accessor.GetIconSize() );
    }
    else
    {
        m_ApplicationIcon.ClearTexture();
    }

    // Rearrange table
    m_pInfoTable->arrange().fit( false );
}

void CatalogScene::ApplicationTileView::onDraw( glv::GLV& glvContext ) NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC( glv::Style, s_CurrentStyle );
    if ( launcher::GetActiveApplicationId() == m_Item.view.id )
    {
        // This app is running, change App name and state color to green
        s_CurrentStyle = glv::Style::standard();
        s_CurrentStyle.color.text.set( 0.1f, 0.85f, 0.2f );
    }
    else
    {
        // App is not running, use default styles
        s_CurrentStyle = glv::Style::standard();
    }

    // Get current state
    const char* currentApplicationStateString = GetApplicationStateString();

    // Check if the state has changed
    if ( currentApplicationStateString != m_pApplicationState->getValue() )
    {
        // Get current label's bounds
        const glv::space_t rightBound = m_pApplicationState->right();
        const glv::space_t topBound = m_pApplicationState->top();
        // Update the Label to the new value
        m_pApplicationState->setValue( currentApplicationStateString );
        // Reposition Label
        m_pApplicationState->pos( rightBound - m_pApplicationState->width(), topBound );
    }

    // Set the style of the App Name and State
    m_pGameCardIcon->style( &s_CurrentStyle );
    m_pApplicationName->style( &s_CurrentStyle );
    m_pApplicationState->style( &s_CurrentStyle );

    // Call the parent view's Draw
    glv::View::onDraw( glvContext );
}

const char* CatalogScene::ApplicationTileView::GetApplicationDownloadStateString( nn::ns::ApplicationDownloadState state ) NN_NOEXCEPT
{
    switch ( state )
    {
    case nn::ns::ApplicationDownloadState::Runnable:        return "Runnable";
    case nn::ns::ApplicationDownloadState::Fatal:           return "Fatal";
    case nn::ns::ApplicationDownloadState::NotEnoughSpace:  return "NotEnoughSpace";
    case nn::ns::ApplicationDownloadState::Finished:        return "Finished";
    case nn::ns::ApplicationDownloadState::Suspended:       return "Suspended";
    default: NN_UNEXPECTED_DEFAULT;
    }
}
const char* CatalogScene::ApplicationTileView::GetApplyDeltaStateString( nn::ns::ApplicationApplyDeltaState state ) NN_NOEXCEPT
{
    switch ( state )
    {
    case nn::ns::ApplicationApplyDeltaState::Applying:       return "Applying";
    case nn::ns::ApplicationApplyDeltaState::Suspended:      return "Suspended";
    case nn::ns::ApplicationApplyDeltaState::NotEnoughSpace: return "NotEnoughSpace";
    case nn::ns::ApplicationApplyDeltaState::Fatal:          return "Fatal";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/************************************
 * class CatalogScene::TileContainer
 ************************************/

CatalogScene::TileContainer::TileContainer( CatalogScene* pParent, glv::space_t width, glv::space_t height, const ApplicationListModel::Item& item, ApplicationTileView* pApplicationTile ) NN_NOEXCEPT
    : glv::View( glv::Rect( width, height ) )
    , m_Item( item )
    , m_pApplicationTile( pApplicationTile )
    , m_pParent( pParent )
{
    this->enable( glv::Property::DrawBorder | glv::Property::FocusHighlight | glv::Property::HitTest | glv::Property::Controllable );

    BasicPadClickDetector::ButtonSetType basicButtons( glv::BasicPadEventType::Button::A::Mask | glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
    DebugPadClickDetector::ButtonSetType debugButtons( glv::DebugPadEventType::Button::A::Mask | glv::DebugPadEventType::Button::Start::Mask | glv::DebugPadEventType::Button::Select::Mask );

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    if ( exhibition::IsExhibitionModeEnabled() )
    {
        basicButtons = ( glv::BasicPadEventType::Button::A::Mask | glv::BasicPadEventType::Button::Y::Mask | glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
        debugButtons = ( glv::DebugPadEventType::Button::A::Mask | glv::DebugPadEventType::Button::Y::Mask | glv::DebugPadEventType::Button::Start::Mask | glv::DebugPadEventType::Button::Select::Mask );
    }
#endif
    changePadClickDetectableButtons( basicButtons );
    changePadClickDetectableButtons( debugButtons );
}

CatalogScene::TileContainer::~TileContainer() NN_NOEXCEPT
{
    // Detach the app tile so that it doesn't get destroyed on Scrollable container refresh
    m_pApplicationTile->remove();
    m_pApplicationTile = nullptr;
}

void CatalogScene::TileContainer::onDraw( glv::GLV& glvContext ) NN_NOEXCEPT
{
    // Check if the the given tile index is already attached to this container
    if ( m_pApplicationTile->parent != this )
    {
        // If it is not, update the tile contents by passing this container's item information
        m_pApplicationTile->UpdateApplicationTileValues( m_Item );
        // Attach the ApplicationTile to this container
        *this << m_pApplicationTile;
    }
    // Update check is required. When m_pApplicationTile was added to this instance, property is sometimes nullopt.
    else if ( m_Item.property )
    {
        m_pApplicationTile->UpdateApplicationTileValues( m_Item );
    }

    // Update the Footer button's String
    m_pParent->UpdateFooterLaunchButtonText(
        m_pParent->GetSelectedApplicationId() != launcher::GetActiveApplicationId()
        ? " A: Launch "
        : " A: Resume " );

    // Call the parent view's Draw
    glv::View::onDraw( glvContext );
}

bool CatalogScene::TileContainer::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    NN_UNUSED( glvRoot );

    switch ( event )
    {
    case glv::Event::FocusGained:
        {
            // Update the selected Application Property and ID
            m_pParent->SetSelectedApplicationProperty( m_Item );
            m_pParent->SetSelectedApplicationId( m_Item.view.id );

            // Update the Footer button's focus style
            m_pParent->UpdateFooterLaunchButtonFocus( m_Item.view.IsLaunchable() );
        }
        break;
    case glv::Event::FocusLost:
        // Update what was the previously selected app
        m_pParent->SetPreviousSelectedApplicationId( m_Item.view.id );
        break;
    default: break;
    }

    return true;
}

/*********************************
 * class CatalogScene::Footer
 *********************************/
CatalogScene::Footer::Footer( CatalogScene* pParent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::Group( rect )
    , m_pParent( pParent )
    , m_LaunchButton( " A: Launch ",  [&]{ m_pParent->m_Op.SelectApplication(); }, CommonValue::InitialFontSize, 0.0f )
    , m_OptionButton( " -/+: Option ", [&]{ m_pParent->m_Op.OpenApplicationInfo(); }, CommonValue::InitialFontSize, 0.0f )
    , m_BuiltInUserSize( "", glv::Label::Spec( glv::Place::TR, 0, 0.f, CommonValue::InitialFontSize ) )
    , m_SdCardSize( "", glv::Label::Spec( glv::Place::TR, 0.0f, 32.0f, CommonValue::InitialFontSize ) )
{
    anchor( glv::Place::BL ).pos( CatalogScene::ListMarginLeft, -CatalogScene::FooterRegion + 4.0f );

    // 操作説明表示
    auto pFooterButtons = new glv::Group( glv::Rect( width(), CatalogScene::FooterRegion - 8.0f ) );
    pFooterButtons->anchor( glv::Place::TL ).pos( 0.0f, 0.0f );
    {
        glv::space_t posOptionButton = CatalogScene::ListMarginLeft + 16.0f;

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

        // 起動ボタン
        m_LaunchButton.anchor( glv::Place::TL ).pos( CatalogScene::ListMarginLeft, 0.0f );
        *pFooterButtons << m_LaunchButton;
        posOptionButton += m_LaunchButton.right();

        if ( exhibition::IsExhibitionModeEnabled() )
        {
            // 自動起動アプリ設定ボタン
            auto pAutoBootSettingButton = new devmenu::Button( " Y: Set Auto Boot Application ", [&]{ m_pParent->m_Op.OpenAutoBootSetting(); }, CommonValue::InitialFontSize, 0.0f );
            pAutoBootSettingButton->anchor( glv::Place::TL ).pos( m_LaunchButton.left(), m_LaunchButton.bottom() + 4.0f );
            *pFooterButtons << pAutoBootSettingButton;

            pAutoBootSettingButton->disable( glv::Property::FocusHighlight );
        }

#endif
        // オプションボタン
        m_OptionButton.anchor( glv::Place::TL ).pos( posOptionButton, 0.0f );
        *pFooterButtons << m_OptionButton;

        // フォーカス調整
        m_LaunchButton.disable( glv::Property::FocusHighlight );
        m_OptionButton.disable( glv::Property::FocusHighlight );

        *this << pFooterButtons;
    }

    // BuiltInUser, SdCard の容量表示
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    const size_t storageContainerWidth = 512.0f;
    auto pStorageContainer = new glv::Group( glv::Rect( storageContainerWidth , CatalogScene::FooterRegion - 8.0f ) );
    pStorageContainer->anchor( glv::Place::TR ).pos( -( storageContainerWidth + ( CatalogScene::ListMarginRight ) ), 0.0f );
    {
#if !defined ( NN_DEVMENUSYSTEM )
        auto pBuiltInUserLabel = new glv::Label( "System Memory", glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize ) );
#else
        auto pBuiltInUserLabel = new glv::Label( "Built-in", glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize ) );
#endif
        *pStorageContainer << pBuiltInUserLabel << m_BuiltInUserSize;

        auto pSdCardLabel = new glv::Label( "SD Card", glv::Label::Spec( glv::Place::TL, 0.f, 32.0f, CommonValue::InitialFontSize ) );
        *pStorageContainer << pSdCardLabel << m_SdCardSize;

        pStorageContainer->disable( focusableProperty );
        *this << pStorageContainer;
    }

    // Set style for size label
    static glv::Style s_SizeStyle;
    s_SizeStyle.color.text.set( 0.6f, 0.6f, 0.6f );
    m_BuiltInUserSize.style( &s_SizeStyle );
    m_SdCardSize.style( &s_SizeStyle );
}

CatalogScene::Footer::~Footer() NN_NOEXCEPT
{
}

void CatalogScene::Footer::UpdateLaunchButtonText( const char* text ) NN_NOEXCEPT
{
    m_LaunchButton.UpdateLabelText( text );
}

void CatalogScene::Footer::UpdateLaunchButtonFocus( bool isEnabled ) NN_NOEXCEPT
{
    m_LaunchButton.UpdateFocusAndColor( isEnabled );
}

void CatalogScene::Footer::IncreaseLaunchButtonSize( glv::space_t increasedX, glv::space_t increasedY ) NN_NOEXCEPT
{
    m_LaunchButton.IncreaseLabelSize( increasedX, increasedY );
}

void CatalogScene::Footer::UpdateStorageSize( bool isSdCardCheckRequired, const nn::Result& checkResult ) NN_NOEXCEPT
{
    glv::space_t previousWidth;
    int64_t freeSpaceSize = 0LL;
    int64_t totalSpaceSize = 0LL;
    char sizeMessage[ 64 ];
#if !defined ( NN_DEVMENUSYSTEM )
    char freeSpaceSizeString[ 64 ];
    char totalSpaceSizeString[ 64 ];
#endif

    // 本体 NAND
    GetStorageSize( &freeSpaceSize, &totalSpaceSize, nn::ncm::StorageId::BuiltInUser );
#if !defined ( NN_DEVMENUSYSTEM )
    ConvertBytesToLargestType( freeSpaceSizeString, sizeof( freeSpaceSizeString ), freeSpaceSize, false );
    ConvertBytesToLargestType( totalSpaceSizeString, sizeof( totalSpaceSizeString ), totalSpaceSize );
    nn::util::SNPrintf( sizeMessage, sizeof( sizeMessage ), "%s / %s ", freeSpaceSizeString, totalSpaceSizeString );
#else
    nn::util::SNPrintf( sizeMessage, sizeof( sizeMessage ), "%s / %s KB",
        GetDelimitedNumberString( freeSpaceSize / 1024 ).c_str(), devmenu::GetDelimitedNumberString( totalSpaceSize / 1024 ).c_str() );
#endif
    previousWidth = m_BuiltInUserSize.width();
    m_BuiltInUserSize.setValue( std::string( sizeMessage ) );
    m_BuiltInUserSize.left( m_BuiltInUserSize.left() - ( m_BuiltInUserSize.width() - previousWidth ) );

    // SD カード
    const auto result = isSdCardCheckRequired ? nn::ns::CheckSdCardMountStatus() : checkResult;
    previousWidth = m_SdCardSize.width();

    if ( result.IsSuccess() )
    {
        if ( GetStorageSize( &freeSpaceSize, &totalSpaceSize, nn::ncm::StorageId::SdCard ).IsSuccess() )
        {
#if !defined ( NN_DEVMENUSYSTEM )
            ConvertBytesToLargestType( freeSpaceSizeString, sizeof( freeSpaceSizeString ), freeSpaceSize, false );
            ConvertBytesToLargestType( totalSpaceSizeString, sizeof( totalSpaceSizeString ), totalSpaceSize );
            nn::util::SNPrintf( sizeMessage, sizeof( sizeMessage ), "%s / %s ", freeSpaceSizeString, totalSpaceSizeString );
#else
            nn::util::SNPrintf( sizeMessage, sizeof( sizeMessage ), "%s / %s KB",
                GetDelimitedNumberString( freeSpaceSize / 1024 ).c_str(), devmenu::GetDelimitedNumberString( totalSpaceSize / 1024 ).c_str() );
#endif
            m_SdCardSize.setValue( std::string( sizeMessage ) );
        }
        else
        {
            m_SdCardSize.setValue( "Access Failed.\n" );
        }
    }
    else
    {
        if ( nn::ns::ResultSdCardNotInserted::Includes( result ) )
        {
            if( nn::fs::IsSdCardInserted() )
            {
                // ns の仕様により、SD カードが利用可能な状態から抜去を行うと、その後は常に nn::ns::CheckSdCardMountStatus() が NotInserted を返す
                // そのため、インサートされているかを fs に問い合わせて直接確認する（マウントは過剰なので行わない）
                m_SdCardSize.setValue( "NX contents cannot be used.\n" );
            }
            else
            {
                m_SdCardSize.setValue( "Not inserted.\n" );
            }
        }
        else if ( nn::ns::ResultSdCardNoOwnership::Includes( result ) || nn::ns::ResultSdCardDatabaseCorrupted::Includes( result )
            || nn::ns::ResultSdCardNotMounted::Includes( result ) )
        {
            // 表示範囲の都合で具体的な情報を削っている
            m_SdCardSize.setValue( "NX contents cannot be used.\n" );
        }
        else if ( nn::ns::ResultSdCardFileSystemCorrupted::Includes( result ) )
        {
            m_SdCardSize.setValue( "SD card is corrupted.\n" );
        }
    }

    m_SdCardSize.left( m_SdCardSize.left() - ( m_SdCardSize.width() - previousWidth ) );
}

const nn::Result CatalogScene::Footer::GetStorageSize( int64_t* pOutFreeSpaceSize, int64_t* pOutTotalSpaceSize, nn::ncm::StorageId storageId ) NN_NOEXCEPT
{
    NN_UNUSED( storageId );

    *pOutFreeSpaceSize = 0LL;
    *pOutTotalSpaceSize = 0LL;

#if !defined( NN_BUILD_CONFIG_OS_WIN )
    NN_RESULT_DO( nn::ns::GetFreeSpaceSize( pOutFreeSpaceSize, storageId ) );
    NN_RESULT_DO( nn::ns::GetTotalSpaceSize( pOutTotalSpaceSize, storageId ) );
#endif

    NN_RESULT_SUCCESS;
}

/*********************************
 * class ApplicationCatalogPage
 *********************************/

ApplicationCatalogPage::ApplicationCatalogPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
    : devmenu::Page( pageId, pageCaption, rect )
    , m_Op( *this )
    , m_pCatalogScene( nullptr )
    , m_pInstallScene( nullptr )
    , m_pDeleteContentsScene( nullptr )
    , m_pApplicationInfoScene( nullptr )
#if !defined( NN_DEVMENULOTCHECK )
    , m_pWriteScene( nullptr )
#endif
{
}

/**
 * @brief ページがコンテナに追加された後に呼び出されます。
 */
void ApplicationCatalogPage::OnAttachedPage() NN_NOEXCEPT
{
    // カタログシーン
    m_pCatalogScene = new CatalogScene( m_Op, this, glv::Rect( w, h ) );
    GetRootSurfaceContext()->SetStorageSizeUpdateFunction( [&]( const nn::Result& result ){ m_pCatalogScene->UpdateStorageSize( false, result ); } );

    *this << m_pCatalogScene;

    // インストールシーン
    m_pInstallScene = new InstallScene( this, m_pCatalogScene );
    m_pInstallScene->SetBackButtonCallback( [&] { this->SwitchScene( m_pInstallScene, m_pCatalogScene, true ); } );
    m_pInstallScene->anchor( glv::Place::TL ).pos( glv::Place::TL, 0.f , 0.f );
    m_pInstallScene->disable( glv::Property::Visible );
    *this << m_pInstallScene;

#if !defined( NN_DEVMENULOTCHECK )
    // ゲームカード書き込みシーン
    m_pWriteScene = new write::WriteScene( this, m_pCatalogScene );
    m_pWriteScene->SetBackButtonCallback( [&] { this->SwitchScene( m_pWriteScene, m_pCatalogScene, true ); } );
    m_pWriteScene->anchor( glv::Place::TL ).pos( glv::Place::TL, 0.f , 0.f );
    m_pWriteScene->disable( glv::Property::Visible );
    *this << m_pWriteScene;
#endif

    // アプリケーション情報シーン
    m_pApplicationInfoScene = new DetailScene( this, m_Op, glv::Rect( w, h ) );
    m_pApplicationInfoScene->SetBackButtonCallback( [&] { m_Op.CloseApplicationInfo(); } );
    m_pApplicationInfoScene->disable( glv::Property::Visible );
    *this << m_pApplicationInfoScene;

    // コンテンツ削除シーン
    m_pDeleteContentsScene = new DeleteContentsScene( this, m_pApplicationInfoScene, m_pCatalogScene );
    m_pDeleteContentsScene->SetBackButtonCallback( [&] { m_pDeleteContentsScene->GetCloseButton()->CloseEventCallback(); } );
    m_pDeleteContentsScene->anchor( glv::Place::TL ).pos( glv::Place::TL, 0.f , 0.f );
    m_pDeleteContentsScene->disable( glv::Property::Visible );
    *this << m_pDeleteContentsScene;

    m_pCatalogScene->SetInstallScene( m_pInstallScene );
#if !defined( NN_DEVMENULOTCHECK )
    m_pCatalogScene->SetWriteScene( m_pWriteScene );
#endif
}


void ApplicationCatalogPage::OnDetachedPage() NN_NOEXCEPT
{
}

void ApplicationCatalogPage::OnActivatePage() NN_NOEXCEPT
{
    m_pCatalogScene->StartUpdateThread();
    GetRootSurfaceContext()->MoveFocusToMenuTabs();
    m_pCatalogScene->UpdateStorageSize();
}

void ApplicationCatalogPage::OnDeactivatePage() NN_NOEXCEPT
{
    m_pCatalogScene->CancelUpdateOfApplicationList();
}

void ApplicationCatalogPage::OnChangeIntoBackground() NN_NOEXCEPT
{
    m_pCatalogScene->OnChangeIntoBackground();
}

void ApplicationCatalogPage::OnChangeIntoForeground() NN_NOEXCEPT
{
    m_pCatalogScene->OnChangeIntoForeground();
}

/**
 * @brief アプリケーションメインループからのコールバックです。
 *
 * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
 * この時点ではまだ glvコンテキストのレンダリングは開始していません。
 */
void ApplicationCatalogPage::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    m_pCatalogScene->OnLoopBeforeSceneRenderer( context, events );
    m_pDeleteContentsScene->OnLoopBeforeSceneRenderer( context, events );
    m_pInstallScene->UpdateInstallProgress();
#if !defined( NN_DEVMENULOTCHECK )
    m_pWriteScene->UpdateWriteProgress();
#endif
    m_pApplicationInfoScene->OnLoopBeforeSceneRenderer( context, events );
}

/**
 * @brief アプリケーションメインループからのコールバックです。
 *
 * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
 */
void ApplicationCatalogPage::OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );

    m_pCatalogScene->OnLoopAfterSceneRenderer( context, events );
}
/**
 * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* ApplicationCatalogPage::GetFocusableChild() NN_NOEXCEPT
{
    if ( true == m_pApplicationInfoScene->enabled( glv::Property::t::Visible ) )
    {
        return static_cast< glv::View* >( m_pApplicationInfoScene );
    }
    else if ( true == m_pInstallScene->enabled( glv::Property::t::Visible ) )
    {
        return m_pInstallScene->GetFocusableChild();
    }
#if !defined( NN_DEVMENULOTCHECK )
    else if ( true == m_pWriteScene->enabled( glv::Property::t::Visible ) )
    {
        return m_pWriteScene->GetFocusableChild();
    }
#endif
    return m_pCatalogScene->GetFocusableChild();
}

/**
 * @brief シーンの切替え。
 */
void ApplicationCatalogPage::SwitchScene( devmenu::Scene* pCurrentScene, devmenu::Scene* pNextScene, bool isBack ) NN_NOEXCEPT
{
    if ( nullptr != lastFocusedView() )
    {
        pCurrentScene->SetLastFocusedView( lastFocusedView() );
    }
    else
    {
        // TORIAEZU: 何らかの理由で lastFocusedView() が nullptr だった場合は、初期フォーカスをセットする
        pCurrentScene->SetLastFocusedView( pCurrentScene->GetFirstFocusTargetView() );
    }

    pCurrentScene->disable( glv::Property::Visible );
    pNextScene->enable( glv::Property::Visible );

    if ( true == isBack )
    {
        auto* pFocusTargetView = pNextScene->GetLastFocusedView();

        if ( nullptr != pFocusTargetView )
        {
            GetRootSurfaceContext()->setFocus( pFocusTargetView );
        }
        else
        {
            GetRootSurfaceContext()->setFocus( pNextScene->GetFirstFocusTargetView() );
        }
    }
    else
    {
        GetRootSurfaceContext()->setFocus( pNextScene->GetFirstFocusTargetView() );
    }

    pNextScene->Refresh();
}

/**
 * @brief 次にフォーカス表示可能なビューを取得します。
 */
glv::View* ApplicationCatalogPage::NextFocusableView( glv::View* pBase, int flags ) NN_NOEXCEPT
{
    return ViewContainer::nextFocusableView( pBase, flags );
}

/*******************************************
 * struct ApplicationCatalogPage::Operators
 *******************************************/

ApplicationCatalogPage::Operators::Operators( ApplicationCatalogPage& parent ) NN_NOEXCEPT
    : parent( parent )
{
}

void ApplicationCatalogPage::Operators::SelectApplication( const nn::ns::ApplicationView& applicationView ) const NN_NOEXCEPT
{
    auto targetApplicationView = applicationView;

    // 有効な applicationId が指定されていない場合は、選択されている Tile の applicationId を使用する
    if ( application::InvalidApplicationView.id == targetApplicationView.id )
    {
        const auto* pProperty = parent.m_pCatalogScene->GetSelectedApplicationProperty();
        if ( nullptr == pProperty )
        {
            return;
        }
        targetApplicationView = pProperty->view;
    }

    if ( targetApplicationView.IsLaunchable() )
    {
        #if defined ( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
            parent.m_pCatalogScene->LaunchOrResumeApplication( targetApplicationView.id );
        #endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    }
    else
    {
        if ( IsTaskSuspended( targetApplicationView ) )
        {
            auto pView = new MessageView( true );

            pView->AddMessage( "Download is suspended. Resume or cancel the download?" );

            pView->AddButton(
                "Cancel",
                [targetApplicationView]( void* pParam, nn::TimeSpan& timespan )
                {
                    nn::ns::CancelApplicationDownload( targetApplicationView.id );
                },
                nullptr
            );

            pView->AddButton(
                "Resume",
                [ targetApplicationView ]( void* pParam, nn::TimeSpan& timespan )
                {
                    nn::ns::ResumeApplicationDownload( targetApplicationView.id );
                },
                nullptr
            );

            pView->AddButton( "Close" );

            parent.GetRootSurfaceContext()->StartModal( pView, true );
        }
    }
}

void ApplicationCatalogPage::Operators::ManageApplication( const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT
{
    parent.m_pDeleteContentsScene->SetApplicationId( applicationId );
    parent.SwitchScene( parent.m_pApplicationInfoScene, parent.m_pDeleteContentsScene, false );
}

void ApplicationCatalogPage::Operators::OpenApplicationInfo() const NN_NOEXCEPT
{
    const auto* pProperty = parent.m_pCatalogScene->GetSelectedApplicationProperty();

    if ( nullptr != pProperty )
    {
        // Initialize the Application details on the info page, and switch to that scene
        parent.m_pApplicationInfoScene->SetApplicationView( pProperty->view );
        parent.SwitchScene( parent.m_pCatalogScene, parent.m_pApplicationInfoScene, false );
    }
}

void ApplicationCatalogPage::Operators::CloseApplicationInfo() const NN_NOEXCEPT
{
    parent.SwitchScene( parent.m_pApplicationInfoScene, parent.m_pCatalogScene, true );
}

void ApplicationCatalogPage::Operators::ShowLegalInfo( const char* kindOfDocument, const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::settings::system::RegionCode region;

    nn::settings::system::GetRegionCode( &region );

    char url[ 256 ];
    nn::util::SNPrintf( url, sizeof( url ), "%s.htdocs/index.html?r=%d", kindOfDocument, region );
    nn::web::OfflineHtmlPageReturnValue returnValue;
    nn::web::ShowApplicationLegalInformationPageArg arg( nn::ApplicationId{ applicationId.value }, url );
    launcher::ShowApplicationLegalInformationPage( &returnValue, arg );
#endif
}

void ApplicationCatalogPage::Operators::OpenAutoBootSetting() const NN_NOEXCEPT
{
#if defined ( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

    if ( !exhibition::IsExhibitionModeEnabled() )
    {
        return;
    }

    auto pView = new MessageView( true );
    const auto pRootSurface = parent.GetRootSurfaceContext();
    const auto* pProperty = parent.m_pCatalogScene->GetSelectedApplicationProperty();

    // 表示するアプリケーション名を取得
    nn::ns::ApplicationControlProperty acProperty;
    auto result = launcher::GetApplicationControlPropertyForLaunch( &acProperty, pProperty->view.id );

    if ( result.IsSuccess() )
    {
        const std::string applicationNameStr = std::string( "[Application Name]\n" ) + std::string( acProperty.GetDefaultTitle().name );
        const auto isLaunchable = pProperty->view.IsLaunchable();

        // 保存する ApplicationId 文字列を取得
        char applicationIdStr[ 32 ];
        nn::util::SNPrintf( applicationIdStr, sizeof( applicationIdStr ), "0x%016llx", pProperty->view.id.value );

        pView->AddMessage( applicationNameStr );

        pView->AddButton( "Close" );

        pView->AddButton(
            "Set as auto boot application",
            [ pRootSurface, applicationIdStr, isLaunchable ]( void* pParam, nn::TimeSpan& timespan )
            {
                // Application を起動できない場合は設定不可
                if ( !isLaunchable )
                {
                    auto pErrorView = new MessageView( true );
                    const std::string errorMessage( "Unable to launch this application.\nInstall the target application to the device." );
                    pErrorView->AddMessage( errorMessage );
                    DEVMENU_LOG( "%s\n", errorMessage.c_str() );
                    pErrorView->AddButton( "Close" );
                    pRootSurface->StartModal( pErrorView, true, true );
                    return;
                }

                auto result = exhibition::SetAutoBootApplicationId( applicationIdStr );
                if ( result.IsFailure() )
                {
                    auto pErrorView = new MessageView( true );
                    devmenu::CreateErrorMessageView( pErrorView, "Failed to set auto boot application.", result );
                    pRootSurface->StartModal( pErrorView, true, true );
                }
            },
            nullptr,
            MessageView::ButtonTextColor::Green
        );
    }
    else
    {
        const std::string messageStr( "Unable to set this content as auto boot target since it is not launchable.\nMaybe it is an addon content or a patch." );
        devmenu::CreateErrorMessageView( pView, messageStr, result );
    }

    pRootSurface->StartModal( pView, true );

#endif // defined ( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
}

devmenu::RootSurfaceContext* ApplicationCatalogPage::Operators::GetRootSurfaceContext() const NN_NOEXCEPT
{
    return parent.GetRootSurfaceContext();
}


/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class ApplicationCatalogPageCreator : devmenu::PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit ApplicationCatalogPageCreator( const char* pPageName ) NN_NOEXCEPT
        : devmenu::PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new ApplicationCatalogPage( ID, GLV_TEXT_API_WIDE_STRING( "Application" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) ApplicationCatalogPageCreator< _id > g_ApplicationCatalogPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_Application, "ApplicationCatalog" );

}} // ~namespace devmenu::application, ~namespace devmenu
