﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_CommonScrollBox.h"

namespace devmenu {

ScrollableBoxView::ScrollableBoxView( const char* tableLayoutStr, const glv::Rect& rect, glv::space_t scrollBarWidth, glv::space_t marginLeft, glv::space_t marginTop ) NN_NOEXCEPT
    : ScissorBoxView( marginLeft, marginTop, rect.width(), rect.height() )
    , m_Scroll( rect, scrollBarWidth, 10.0f, 5.0f )
    , m_pTable( nullptr )
    , m_InnerRect( m_Scroll.width() - InnerRectMarginHorizontal, m_Scroll.height() - InnerRectMarginVertical )
    , m_pFocusedView( nullptr )
    , m_pAncestorView( nullptr )
    , m_ChildrenFocusTraversalAction ( this )
{
    m_TableLayoutString = tableLayoutStr;

    m_Scroll.disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::FocusHighlight );
    this->enable( glv::Property::Controllable | glv::Property::FocusHighlight );

    this->attach( []( const glv::Notification& notification )->void
    {
        ScrollableBoxView* pSelf = notification.receiver< ScrollableBoxView >();
        pSelf->OnFocused( notification.sender< glv::View >() );
    }, glv::Update::Focus, this );

    this->Refresh();
    this->add( m_Scroll );
}

ScrollableBoxView::ScrollableBoxView( const char* tableLayoutStr, const glv::Rect& rect, glv::View* pAncestorView, glv::space_t scrollBarWidth, glv::space_t marginLeft, glv::space_t marginTop ) NN_NOEXCEPT
    : ScrollableBoxView( tableLayoutStr, rect, scrollBarWidth, marginLeft, marginTop )
{
    NN_ASSERT_NOT_NULL( pAncestorView );
    m_pAncestorView = pAncestorView;
}

glv::View& ScrollableBoxView::operator << ( glv::View* pNewChild ) NN_NOEXCEPT
{
    // Set a Focus event so that we scroll automatically to pNewChild view.
    // This only works if newChild is the one receiving the Focus event (not one of its descendants).
    if ( pNewChild->enabled( glv::Property::FocusHighlight ) )
    {
        pNewChild->attach( []( const glv::Notification& notification )->void
        {
            ScrollableBoxView* pSelf = notification.receiver< ScrollableBoxView >();
            pSelf->OnFocused( notification.sender< glv::View >() );
        }, glv::Update::Focus, this );
    }

    // For all of the descendants, set a different Focus event that will traverse up through it's parents, and set this
    // ScrollableBoxView's position appropriately.
    //
    // This is necessary for when Focus is obtained by objects nested within pNewChild.  If this isn't done, then there is no way
    // to accurately determine the position the ScrollableBoxView should scroll to.
    std::vector< glv::View* > children;
    pNewChild->getChildren( children, m_ChildrenFocusTraversalAction );

    *m_pTable << pNewChild;
    m_pTable->arrange();
    return *m_pTable;
}

void ScrollableBoxView::OnFocused( const glv::View* const pFocusedView ) NN_NOEXCEPT
{
    // Check to make sure the control isn't already visible. If it is visible already, do nothing.
    if ( !this->IsViewDisplayed( pFocusedView ) )
    {
        auto topPosition = this->GetAdjustedTopPosition( pFocusedView );
        if ( topPosition < m_Scroll.height() * 0.5f )
        {
            topPosition = 0.0f;
        }
        this->ScrollTo( topPosition );
    }
    m_pFocusedView = pFocusedView;
}

void ScrollableBoxView::ScrollToTop() NN_NOEXCEPT
{
    m_Scroll.scrollTopTo( 0.0f );
    m_pFocusedView = nullptr;
}

void ScrollableBoxView::ScrollTo( float position ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pTable );

    // Check if we have more content than can be displayed
    if ( m_pTable->height() > m_Scroll.height() )
    {
        // Scroll top to the position given
        m_Scroll.scrollTopTo( position );
    }
}

void ScrollableBoxView::ScrollTo( glv::View* pTargetView ) NN_NOEXCEPT
{
    if ( glv::isRelation( pTargetView, m_pTable ) )
    {
        const auto& visibleRegion = m_pTable->visibleRegion();
        const auto nextFocusedViewCenterPosY = this->GetAdjustedTopPosition( pTargetView ) + pTargetView->height() * 0.5f;
        const auto scrollAreaCenterPosY = visibleRegion.top() + m_Scroll.height() * 0.5f;
        this->ScrollTo( visibleRegion.top() - scrollAreaCenterPosY + nextFocusedViewCenterPosY );
        m_pFocusedView = pTargetView;
    }
}

void ScrollableBoxView::Refresh() NN_NOEXCEPT
{
    if ( nullptr != m_pTable )
    {
        m_pTable->remove();
        delete m_pTable;
    }
    m_pTable = new glv::Table( m_TableLayoutString.c_str(), 0.0f, 12.0f, m_InnerRect );
    m_Scroll << m_pTable;
}

void ScrollableBoxView::ArrangeTable() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pTable );
    if ( nullptr != m_pTable )
    {
        m_pTable->arrange();
    }

    if ( m_pTable->height() < m_Scroll.height() )
    {
        this->disable( glv::Property::FocusHighlight );
    }
}

void ScrollableBoxView::SetRectAndUpdateInnerSize( glv::space_t width, glv::space_t height ) NN_NOEXCEPT
{
    this->width( width );
    this->height( height );
    m_Scroll.extent( width, height );
    m_InnerRect = glv::Rect( m_Scroll.w - InnerRectMarginHorizontal, m_Scroll.h - InnerRectMarginVertical );
}

glv::space_t ScrollableBoxView::GetAdjustedTopPosition( const glv::View* const pView ) NN_NOEXCEPT
{
    // To get the correct position, start at the View that got the Focus event, and traverse up through
    // it's parents.  By summing up their Y positions, you get the final position of the View within
    // the scrollable table.
    float finalPosY = pView->t;
    const auto* pAncestorView = pView->parent;

    while ( nullptr != pAncestorView && m_pTable != pAncestorView )
    {
        finalPosY += pAncestorView->t;
        pAncestorView = pAncestorView->parent;
    }

    return finalPosY;
}

const glv::View* ScrollableBoxView::GetNextFocusableChild( const glv::View* pFocusedView, glv::MoveFlags flags ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( pFocusedView );
    auto const pRootView = ( this != pFocusedView || nullptr == m_pAncestorView ) ? this : m_pAncestorView;
    glv::NextFocusAction actionDown( pFocusedView, pRootView, flags );
    glv::traverseChildren( actionDown, pRootView );
    return actionDown.getTargetView();
}

bool ScrollableBoxView::IsViewDisplayed( const glv::View* const pView ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( pView );
    NN_ASSERT_NOT_NULL( m_pTable );

    const auto& visibleRegion = m_pTable->visibleRegion();

    // Check if all elements are within scroll area and the target view is completely displayed.
    // However, the view is judged as displayed when the hegiht of view is larger than visible region.
    return
        m_pTable->height() <= m_Scroll.height() ||
        std::abs( pView->height() - pView->visibleRegion().height() ) < ScrollableBoxView::AcceptableFloatDelta ||
        ( visibleRegion.height() <= pView->height() &&
          visibleRegion.top() > pView->top() &&
          visibleRegion.bottom() < pView->bottom() );
}

bool ScrollableBoxView::IsDescendant( const glv::View* const pTargetView ) const NN_NOEXCEPT
{
    auto pView = pTargetView;

    while ( NN_STATIC_CONDITION( true ) )
    {
        if ( nullptr == pView->parent )
        {
            break;
        }

        pView = pView->parent;

        if ( m_pTable == pView )
        {
            return true;
        }
    }

    return false;
}

bool ScrollableBoxView::Move( glv::MoveFlags flags, bool isRepeat ) NN_NOEXCEPT
{
    if ( nullptr == m_pFocusedView )
    {
        return true;
    }

    const auto pNextFocusedView =
        this->IsViewDisplayed( m_pFocusedView )
        ? this->GetNextFocusableChild( m_pFocusedView, flags )
        : m_pFocusedView;
    const auto visibleRegion = m_pTable->visibleRegion();

    // Check if next focusable view is a descendant of ScrollBoxView and avoid unnecessary scroll when the bottom of visible region reaches the bottom of ScrollBoxView
    if ( nullptr != pNextFocusedView &&
        this->IsDescendant( pNextFocusedView ) &&
        false == ( std::abs( m_pTable->height() - visibleRegion.bottom() ) < ScrollableBoxView::AcceptableFloatDelta && glv::MoveFlags::MoveDown == flags ) )
    {
        // Center the next focused view
        const auto nextFocusedViewCenterPosY = this->GetAdjustedTopPosition( pNextFocusedView ) + pNextFocusedView->height() * 0.5f;
        const auto scrollAreaCenterPosY = visibleRegion.top() + m_Scroll.height() * 0.5f;
        this->ScrollTo( visibleRegion.top() - scrollAreaCenterPosY + nextFocusedViewCenterPosY );

        return true;
    }

    switch ( flags )
    {
    case glv::MoveFlags::MoveUp:
        if ( std::abs( visibleRegion.top() ) < ScrollableBoxView::AcceptableFloatDelta )
        {
            return true;
        }
        else
        {
            m_Scroll.pageY( this->ForcedScrollRate );
            return false;
        }
    case glv::MoveFlags::MoveDown:
        // Check if bottom of the table is displayed
        if ( visibleRegion.top() + m_Scroll.height() >= m_pTable->height() )
        {
            if ( nullptr != pNextFocusedView )
            {
                return true;
            }
            else
            {
                if ( !isRepeat )
                {
                    // Scroll to top considering the next focasable view exists outside ScrollableBoxView.
                    // If the next focusable view exists and it is the descendant of ScrollableBoxView(),
                    // scroll to top is usually done twice in this place and OnFocused().
                    // There is no way to check the next focusable view exists at this moment.
                    // It is checked in glv::TabPages::onEvent() after this process.
                    this->ScrollToTop();
                    return true;
                }
                else
                {
                    // Avoid moving foucus from bottom to top when input type is repeat
                    return false;
                }
            }
        }
        else
        {
            m_Scroll.pageY( -this->ForcedScrollRate );
            return false;
        }
    default:
        return true;
    }
}

bool ScrollableBoxView::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    if ( event == glv::Event::BasicPadDown || event == glv::Event::BasicPadRepeat )
    {
        if ( glvRoot.getBasicPadEvent().IsButtonDown<   glv::BasicPadEventType::Button::Up >() ||
             glvRoot.getBasicPadEvent().IsButtonRepeat< glv::BasicPadEventType::Button::Up >() )
        {
            return this->Move( glv::MoveFlags::MoveUp, false );
        }
        else if ( glvRoot.getBasicPadEvent().IsButtonDown< glv::BasicPadEventType::Button::Down >() )
        {
            return this->Move( glv::MoveFlags::MoveDown, false );
        }
        else if ( glvRoot.getBasicPadEvent().IsButtonRepeat< glv::BasicPadEventType::Button::Down >())
        {
            return this->Move( glv::MoveFlags::MoveDown, true );
        }
    }

    if ( event == glv::Event::DebugPadDown || event == glv::Event::DebugPadRepeat )
    {
        if ( glvRoot.getDebugPadEvent().IsButtonDown<   glv::DebugPadEventType::Button::Up >() ||
             glvRoot.getDebugPadEvent().IsButtonRepeat< glv::DebugPadEventType::Button::Up >() )
        {
            return this->Move( glv::MoveFlags::MoveUp, false );
        }
        else if ( glvRoot.getDebugPadEvent().IsButtonDown< glv::DebugPadEventType::Button::Down >() )
        {
            return this->Move( glv::MoveFlags::MoveDown, false );
        }
        else if ( glvRoot.getDebugPadEvent().IsButtonRepeat< glv::DebugPadEventType::Button::Down >())
        {
            return this->Move( glv::MoveFlags::MoveDown, true );
        }
    }
    return true;
}

ScrollableBoxView::ChildrenFocusTraversalAction::ChildrenFocusTraversalAction( ScrollableBoxView* pParent ) NN_NOEXCEPT
{
    m_pScrollableBoxView = pParent;
}

bool ScrollableBoxView::ChildrenFocusTraversalAction::operator()( glv::View* pView, int depth ) NN_NOEXCEPT
{
    // This will traverse through all of the children of the sender view inside of the Notification 'n', and attach
    // a callback to the glv::Update::Focus event.  This is done so that when a child of one of the Views added to a
    // ScrollableBoxView gets focus, we are given the opportunity to set the scroll position correctly to that child.
    //
    // If this is not done, the ScrollableBoxView will only be able to scroll if the parent View itself gets Focus.

    if ( pView->enabled( glv::Property::FocusHighlight ) )
    {
        pView->attach([](const glv::Notification& notification)->void
        {
            ScrollableBoxView* pScrollableBoxView = notification.receiver< ScrollableBoxView >();
            pScrollableBoxView->OnFocused(notification.sender< glv::View >());
        }, glv::Update::Focus, m_pScrollableBoxView);
    }

    // Recursively perform these actions for the children of the container passed in to ensure that the Focus event
    // can be triggered for all descendants.
    std::vector< glv::View* > children;
    pView->getChildren( children, *this );

    return true;
}

} // ~namespace devmenu
