﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <string>

#include "../DevMenu_ModalView.h"

namespace devmenu {

/**********************************
 * class AsyncTaskWithProgressView
 **********************************/
class AsyncTaskWithProgressView
{
public:
    enum ExitReason
    {
        ExitReason_Success,               // 処理が成功した
        ExitReason_Failure,               // 処理に失敗した
        ExitReason_CancelOnModalView,     // ModalView 上のキャンセルボタンを押下した
        ExitReason_CancelOnLibraryApplet, // Library Applet で解決できずにキャンセルになった
    };

public:

    /**
     * @brief   コンストラクタ
     *
     * @param[in]  pRootSurfaceContext                   RootSurfaecContext のポインタ (ModalView 表示やフォーカス処理のために必要)
     * @param[in]  executionStr                          実行時に表示する文字列
     * @param[in]  threadFunction                        サブスレッドでの非同期処理で実行する関数
     * @param[in]  completionCallback                    処理の完了後に実行するコールバック関数
     * @param[in]  isViewClosedAutomatically             処理の完了後に ModalView を自動的にクローズするか否か
     * @param[in]  eventClearMode                        終了イベントのクリアモード (デフォルトは nn::os::EventClearMode_ManualClear )
     *
     * @details
     *  threadFunction の引数 pOutExitReason には終了理由を設定してください。pIsCancelRequested は Cancel() 実行で true になります。
     *  threadFunction の処理完了後、設定した返り値 Result と *pOutReason は completeCallback に渡ります。
     *  completionCallback の返り値は「ModalView のクローズ後も InvisibleWall の前面表示を維持するか否か」を設定してください。
     */
    AsyncTaskWithProgressView(
        RootSurfaceContext* pRootSurfaceContext,
        const std::string& executionStr,
        const std::function< nn::Result( ExitReason* pOutExitReason, bool* pIsCancelRequested ) >& threadFunction,
        const std::function< bool( const nn::Result&, ExitReason ) >& completionCallback,
        bool isViewClosedAutomatically,
        nn::os::EventClearMode eventClearMode = nn::os::EventClearMode_ManualClear
    ) NN_NOEXCEPT
        : m_Thread()
        , m_Event( eventClearMode )
        , m_State( ExecutionState_None )
        , m_ProgressModalView( isViewClosedAutomatically )
        , m_pRootSurfaceContext( pRootSurfaceContext )
        , m_ProgressMessageBase( executionStr )
        , m_IsCancelRequested( false )
        , m_ExitReason( ExitReason_Failure )
        , m_Result()
        , m_ThreadFunction( threadFunction )
        , m_CompletionCallback( completionCallback )
    {
        NN_ASSERT_NOT_NULL( pRootSurfaceContext );
        NN_ASSERT_NOT_NULL( threadFunction );
    }

    /**
     * @brief   処理を開始します。
     */
    void Start() NN_NOEXCEPT;

    /**
     * @brief   処理をキャンセルします。
     */
    inline void Cancel() NN_NOEXCEPT
    {
        m_IsCancelRequested = true;
    }

    /**
     * @brief   処理が終了するのを待機します。
     *
     * @details
     *  コンストラクタで指定したイベントクリアモードが nn::os::EventClearMode_ManualClear の場合、
     *  処理の終了を検知後に本関数を呼ぶと即時終了します。
     *  終了イベントを非シグナル状態に戻すには Clear() を実行する必要があります。
     */
    void Wait() NN_NOEXCEPT;

    /**
     * @brief   処理が終了したか否かを判別します。
     *
     * @return  処理が終了した場合は true を返します。
     *
     * @details
     *  コンストラクタで指定したイベントクリアモードが nn::os::EventClearMode_ManualClear の場合、
     *  処理の終了を検知後に本関数を呼ぶと true を返します。
     *  終了イベントを非シグナル状態に戻すには Clear() を実行する必要があります。
     */
    bool TryWait() NN_NOEXCEPT;

    /**
     * @brief   タスクの状態をクリアして初期状態に戻します。
     *
     * @details
     *  再度 Start() を実行してタスクを開始するには、事前に本関数を呼ぶ必要があります。
     */
    void Clear() NN_NOEXCEPT;

    /**
     * @brief   処理の終了理由を返します。
     */
    inline ExitReason GetExitReason() const NN_NOEXCEPT
    {
        return m_ExitReason;
    }

    /**
     * @brief   処理の結果を返します。
     */
    inline const nn::Result& GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }

    /**
     * @brief   ProgressModalView にメッセージをセットします。
     *
     * @param[in]   message  セットする文字列
     */
    void SetProgressViewMessage( const std::string& message ) NN_NOEXCEPT;

    /**
     * @brief   ProgressModalView を更新し、処理が完了したらスレッドを破棄します。
     *
     * @details
     *  本関数はスレッドセーフでない ModalView 操作 API を実行するので、メインスレッドから呼び出す必要があります。
     */
    void UpdateProgressView() NN_NOEXCEPT;

protected:

    /**
     * @brief   ModalView を表示します。
     *
     * @param[in]   isFloatingApplicationLaunchBlocked   アプリケーションのフローティング起動をブロックするか否か
     */
    void StartModal( bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT;

private:
    enum ExecutionState
    {
        ExecutionState_None,
        ExecutionState_Preparing,
        ExecutionState_Running,
        ExecutionState_Completed
    };

private:
    static constexpr int ProgressMessageLoopCount = 4;
    static constexpr int ThreadStackSize = 8 * 1024; // 暫定(ProgressModalScene との調整後に 64 k にする)

private:
    /**
     * @brief   処理を実行するスレッド関数です。
     */
    void ThreadFunction() NN_NOEXCEPT;

    /**
     * @brief   サブスレッド処理が完了した際に UpdateProgressView() 内で呼ばれるコールバック関数です。
     *
     * @param[in]   result       処理の結果
     * @param[in]   exitReason   処理の終了理由
     *
     * @return  終了後に遷移する処理状態を返します。
     */
    ExecutionState OnCompleteExecution( const nn::Result& result, ExitReason exitReason ) NN_NOEXCEPT;

private:
    nn::os::ThreadType                                m_Thread;
    nn::os::Event                                     m_Event;
    ExecutionState                                    m_State;
    ProgressModalView                                 m_ProgressModalView;
    RootSurfaceContext*                               m_pRootSurfaceContext;
    const std::string                                 m_ProgressMessageBase;
    bool                                              m_IsCancelRequested;
    ExitReason                                        m_ExitReason;
    nn::Result                                        m_Result;
    std::function< nn::Result( ExitReason*, bool* ) > m_ThreadFunction;
    std::function
        < bool( const nn::Result&, ExitReason ) >     m_CompletionCallback;
};

} // ~namespace devmenu
