﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsSceneFwdbg.h"
#include "DevMenu_AccountsUiComponents.h"

#include <nn/nn_Abort.h>
#include <nn/account/account_ApiDebug.h>

#include <nn/settings/system/settings_Account.h>
#if defined(DEVMENU_ACCOUNTS_FWDBG_ENABLE)
#include "../Common/DevMenu_CommonSettingsApi.h"
#endif

namespace devmenu { namespace accounts {
namespace {
template <typename T>
T LoadFwdbg(const char* key) NN_NOEXCEPT
{
    T value = {};
#if defined(DEVMENU_ACCOUNTS_FWDBG_ENABLE)
    GetFixedSizeFirmwareDebugSettingsItemValue( &value, "account", key );
#endif
    return value;
}
} // namespace devmenu::accounts::<anonymous>

FwdbgScene::Closer::Closer(const AbstractOperators& op, FwdbgScene& parent, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
    : Button("Back", [&]{ m_Op.PopScene(&m_Parent, m_Parent.m_pPreviousScene); } , r, anchor)
    , m_Op(op)
    , m_Parent(parent)
{
}

FwdbgScene::FwdbgScene( AbstractOperators& op, glv::Rect rect, Scene* pParent ) NN_NOEXCEPT
    : Scene( rect )
    , m_Header(
        "Firmware settings",
        glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize ) )
    , m_Footer(
        GLV_TEXT_API_WIDE_STRING( "↑↓:Focus  A:Select" ),
        glv::Label::Spec( glv::Place::BR, 0.0f, 0.0f, CommonValue::InitialFontSize ) )
    , m_Container(
        "<", glv::Rect( width(), height() - (Scene::HeaderRegion + Scene::FooterRegion) ), 5.0f, 0.0f, HeaderRegion )
    , m_NaRequiredCheckbox( "Requires Nintendo Account for use of network service" )
    , m_NaLicenseVerificationEnabledCheckbox( "Enable Nintendo Account license verification" )
    , m_SkipSelectionCheckbox( "Skip user account selection if possible" )
    , m_InterprogramOpenUserRetentionCheckbox( "Enable inter-program open user retention" )
    , m_SuspendBgDaemonCheckbox( "Suspend background activity temporarily" )
    , m_Suspension( decltype(m_Suspension)() )
    , m_Closer(op, *this, glv::Rect(m_Container.GetInnerRect().width(), 48.0f), glv::Place::CC)
    , m_pPreviousScene(pParent)
{
#if !defined ( NN_CUSTOMERSUPPORTTOOL )
    *this << m_Header << m_Footer << m_Container;

    // Setup "NaRequiredCheckbox"
    m_NaRequiredCheckbox.SetValue( LoadFwdbg<bool>("na_required_for_network_service") );
    m_NaRequiredCheckbox.SetCallback( []( const glv::Notification& n )->void { n.receiver<FwdbgScene>()->UpdateNaRequiredCheckbox( n ); }, this );

    // Setup "m_NaLicenseVerificationEnabledCheckbox"
    m_NaLicenseVerificationEnabledCheckbox.SetValue( LoadFwdbg<bool>("na_license_verification_enabled") );
    m_NaLicenseVerificationEnabledCheckbox.SetCallback( []( const glv::Notification& n )->void { n.receiver<FwdbgScene>()->UpdateNaLicenseVerificationEnabledCheckbox( n ); }, this );

    // Setup "SkipSelectionCheckbox"
    nn::settings::system::AccountSettings cfg;
    nn::settings::system::GetAccountSettings( &cfg );
    m_SkipSelectionCheckbox.SetValue(cfg.userSelectorSettings.flags.Test<nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>());
    m_SkipSelectionCheckbox.SetCallback( []( const glv::Notification& n )->void { n.receiver<FwdbgScene>()->UpdateSkipSelectionCheckbox( n ); }, this );

    // Setup "InterprogramOpenUserRetentionCheckbox"
    m_InterprogramOpenUserRetentionCheckbox.SetValue(false);
    m_InterprogramOpenUserRetentionCheckbox.SetCallback( [](const glv::Notification& n)->void { n.receiver<FwdbgScene>()->UpdateInterprogramOpenUserRetentionCheckbox(n); }, this );

    // Setup "SuspendBdDaemonCheckbox"
    m_SuspendBgDaemonCheckbox.SetValue( false );
    m_SuspendBgDaemonCheckbox.SetCallback( []( const glv::Notification& n )->void { n.receiver<FwdbgScene>()->UpdateSuspendBgDaemonCheckbox( n ); }, this );

    m_Container.Refresh();
    m_Container << m_Closer;
#if defined( DEVMENU_ACCOUNTS_FWDBG_ENABLE )
    m_Container << m_NaRequiredCheckbox;
    m_Container << m_NaLicenseVerificationEnabledCheckbox;
#endif
    m_Container << m_SkipSelectionCheckbox;
    m_Container << m_InterprogramOpenUserRetentionCheckbox;
    m_Container << m_SuspendBgDaemonCheckbox;

#if 0
    glv::Style* pStyle = new glv::Style();
    pStyle->color.set(glv::StyleColor::BlackOnWhite);
    pStyle->color.fore.set(0.8, 0.8, 0.8);
    pStyle->color.back.set(0.0, 0.6, 0.0);
    style(pStyle);
    enable(glv::Property::DrawBack | glv::Property::DrawBorder);
#endif
#endif // !defined ( NN_CUSTOMERSUPPORTTOOL )
}

void FwdbgScene::UpdateNaRequiredCheckbox( const glv::Notification& n ) NN_NOEXCEPT
{
#if defined( DEVMENU_ACCOUNTS_FWDBG_ENABLE )
        auto newValue = m_NaRequiredCheckbox.GetValue();
        SetFixedSizeFirmwareDebugSettingsItemValue( "account", "na_required_for_network_service", newValue );
#else
        NN_UNUSED( n );
        NN_ABORT( "[Accounts] Unexpected operation" );
#endif
}

void FwdbgScene::UpdateNaLicenseVerificationEnabledCheckbox( const glv::Notification& n ) NN_NOEXCEPT
{
#if defined( DEVMENU_ACCOUNTS_FWDBG_ENABLE )
    auto newValue = m_NaLicenseVerificationEnabledCheckbox.GetValue();
    SetFixedSizeFirmwareDebugSettingsItemValue( "account", "na_license_verification_enabled", newValue );
#else
    NN_UNUSED( n );
    NN_ABORT( "[Accounts] Unexpected operation" );
#endif
}

void FwdbgScene::UpdateSkipSelectionCheckbox( const glv::Notification& n ) NN_NOEXCEPT
{
    nn::settings::system::AccountSettings cfg;
    nn::settings::system::GetAccountSettings( &cfg );
    cfg.userSelectorSettings.flags.Set< nn::settings::system::UserSelectorFlag::SkipsIfSingleUser >( m_SkipSelectionCheckbox.GetValue() );
    nn::settings::system::SetAccountSettings( cfg );
}

void FwdbgScene::UpdateInterprogramOpenUserRetentionCheckbox( const glv::Notification& n ) NN_NOEXCEPT
{
    NN_UNUSED( n );
    // Check if the InterprogramOpenUserRetention Checkbox is checked
    if ( m_InterprogramOpenUserRetentionCheckbox.GetValue() )
    {
        m_Retention = nn::account::DebugEnableInterprogramOpenUserRetention();
    }
    else
    {
        m_Retention = decltype(m_Retention)();
    }
}

void FwdbgScene::UpdateSuspendBgDaemonCheckbox( const glv::Notification& n ) NN_NOEXCEPT
{
    NN_UNUSED( n );
    // Check if the SuspendBgDaemon Checkbox is checked
    if ( m_SuspendBgDaemonCheckbox.GetValue() )
    {
        m_Suspension = nn::account::SuspendBackgroundActivity();
    }
    else
    {
        m_Suspension = decltype(m_Suspension)();
    }
}

void FwdbgScene::Close() NN_NOEXCEPT
{
    disable( glv::Property::Visible );
    m_pPreviousScene->enable( glv::Property::Visible );

    m_pPreviousScene->RequireRefresh();
}

void FwdbgScene::Clear() NN_NOEXCEPT
{
}

void FwdbgScene::CloseScene() NN_NOEXCEPT
{
    Close();
}

void FwdbgScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    auto& g = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &g ) );
    g.setFocus( &m_Closer );
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
