﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <cstring>
#include <memory>
#include <string>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Base.h>
#include <nn/os/os_Thread.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/util/util_Color.h>
#if 0
#include <nn/bpc/bpc_BoardPowerControl.h>
#else
#include <nn/oe/oe_Common.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>
#endif
#include <nn/nn_Log.h>
#include "DevKitUpdater.h"

// TODO: クラス化
// TODO: PCブート時にスタブ化

namespace
{
    // 表示設定
    std::string g_DisplayString = "";
    ViewDisplayState g_DisplayState = ViewDisplayState_Other;
    nn::os::Tick g_FinishTick;

    // View Thread 設定
    const size_t                    ThreadStackSize = 0x40000;
    NN_OS_ALIGNAS_THREAD_STACK char g_ViewThreadStack[ ThreadStackSize ];
    nn::os::ThreadType              g_ViewThread;

    struct MakeCommandUserData
    {
        nn::gfx::util::DebugFontTextWriter* pWriter;
        int frame;
    };
    nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
}

static void ViewThreadFunction(void *arg);
static void MakeCommand(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData);

nn::Result InitializeView()
{
    nn::Result result;

    // スレッドを作成する
    result = nn::os::CreateThread( &g_ViewThread, ViewThreadFunction, NULL, g_ViewThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_ViewThread." );
    nn::os::SetThreadName( &g_ViewThread, "View Thread" );

    nn::os::StartThread( &g_ViewThread );

    return nn::ResultSuccess();
}

nn::Result FinalizeView()
{
    // スレッドを終了させる
    // TODO: スレッドが先に終了した場合も問題ないことを確認する
    nn::os::WaitThread( &g_ViewThread );
    nn::os::DestroyThread( &g_ViewThread );

    return nn::ResultSuccess();
}

void ViewSetState(ViewDisplayState setState)
{
    if(setState == ViewDisplayState_Finish)
    {
        g_FinishTick = nn::os::GetSystemTick();
    }

    g_DisplayState = setState;
}

void ViewSetString(std::string setString)
{
    g_DisplayString = setString;
}

void ViewErrorLog(std::string errorString)
{
    ViewSetState(ViewDisplayState_Error);
    ViewSetString(errorString);
}

// TODO: 別ファイルへ移動
void RebootMe()
{
    NN_LOG("Restart now.\n");
#if 0
    nn::bpc::InitializeBoardPowerControl();
    nn::bpc::RebootSystem();
#else
    nn::oe::Initialize();
    nn::oe::RequestToReboot();
#endif
}

static void ViewThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    const int BufferCount = 2;
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // デバッグフォント初期化
    const bool userMemoryPoolEnable = true;    // true にすると、ユーザーのメモリプールを使用します
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetBufferCount(BufferCount);
    info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(gfw.GetDevice(), info);
    nn::util::BytePtr debugFontHeap(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(gfw.GetDevice(), info);

    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    nn::gfx::MemoryPool* pMemoryPool = gfw.GetMemoryPool(type);
    ptrdiff_t memoryPoolOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
    nn::gfx::util::DebugFontTextWriter writer;
    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        memoryPoolOffset = gfw.AllocatePoolMemory(type, debugFontMemoryPoolSize, 1);

        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize,
            pMemoryPool, memoryPoolOffset, debugFontMemoryPoolSize);
    }
    else {
        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize, nullptr, 0, 0);
    }

    int textureDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samplerDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    writer.SetDisplayWidth(gfw.GetDisplayWidth());
    writer.SetDisplayHeight(gfw.GetDisplayHeight());
    writer.SetTextureDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    writer.SetSamplerDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);

    // プリミティブレンダラー初期化
    {
        nns::gfx::PrimitiveRenderer::RendererInfo rendererInfo;
        rendererInfo.SetDefault();
        rendererInfo.SetAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nullptr);
        rendererInfo.SetAdditionalBufferSize(1024 * 4);
        rendererInfo.SetDrawCallCountMax(1024 * 4);
        rendererInfo.SetViewFunctionCallCountMax(1024 * 4);

        rendererInfo.SetMultiBufferQuantity(BufferCount);

        g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(gfw.GetDevice(), rendererInfo);
        g_pPrimitiveRenderer->SetScreenWidth(gfw.GetDisplayWidth());
        g_pPrimitiveRenderer->SetScreenHeight(gfw.GetDisplayHeight());
    }

    // GraphicsFramework によるフレーム処理用の設定
    MakeCommandUserData userData;
    userData.pWriter = &writer;
    userData.frame = 0;

    gfw.SetMakeCommandCallback(MakeCommand, &userData);


    // 毎フレームのレンダリング
    // for (int frame = 0; frame < 60 * 10; ++frame)
    while (g_DisplayState != ViewDisplayState_Close && g_DisplayState != ViewDisplayState_Terminate)
    {
        gfw.ProcessFrame();
        userData.frame++;
    }
    gfw.QueueFinish();


    // デバッグフォント終了
    writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(debugFontHeap.Get());
    debugFontHeap.Reset(nullptr);

    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        gfw.FreePoolMemory(type, memoryPoolOffset);
    }
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureDescriptorIndex);
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samplerDescriptorIndex);

    // フレームワーク終了
    gfw.Finalize();
}


void Print(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type colorWhite = { { 255, 255, 255, 255 } };

    // writer.SetScale(1.0f, 1.0f);

    std::string displayString = g_DisplayString;
    if(g_DisplayState == ViewDisplayState_Progress)
    {
        uint64_t dotNum = 0;
        dotNum = ((uint64_t)frame >> 0x6) & 0x7;
        std::string addString = std::string(dotNum, '.');
        displayString += addString;
    }

    writer.SetTextColor(colorWhite);
    writer.SetCursor(240.f, 120.f);
    writer.SetScale(2.0f, 2.0f);
    writer.Print(displayString.c_str());

    if(g_DisplayState == ViewDisplayState_Finish)
    {
        const int64_t CountDownTime = 15;
        nn::os::Tick currentTick = nn::os::GetSystemTick();

        nn::TimeSpan countdownSpan = nn::os::ConvertToTimeSpan(currentTick - g_FinishTick);
        int64_t countdownSec = CountDownTime - countdownSpan.GetSeconds();

        if(countdownSec > 0)
        {
            std::string str = "Reboot will be held in " + std::to_string(countdownSec) + " seconds.";
            writer.SetTextColor(colorWhite);
            writer.SetCursor(240.f, 180.f);
            writer.SetScale(2.0f, 2.0f);
            writer.Print(str.c_str());
        }
        else
        {
            g_DisplayState = ViewDisplayState_Close;
        }
    }
}

void Draw(nn::gfx::CommandBuffer* rootCommandBuffer)
{
    switch (g_DisplayState)
    {
        case ViewDisplayState_Finish:
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::Green());
            break;
        case ViewDisplayState_Error:
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::Red());
            break;
        case ViewDisplayState_Terminate:
        case ViewDisplayState_Close:
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());
            break;
        default:
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::Blue());
            break;
    }
    g_pPrimitiveRenderer->Draw2DRect(rootCommandBuffer, 0.f, 0.f, 160.f, 720.f);
    g_pPrimitiveRenderer->Draw2DRect(rootCommandBuffer, 1120.f, 0.f, 160.f, 720.f);
}

static void MakeCommand(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>(pUserData);

    // TODO: これ必要？
    g_pPrimitiveRenderer->Update(bufferIndex);

    // フォント表示
    Print(*pData->pWriter, pData->frame);

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // 帯を描画
        Draw(rootCommandBuffer);

        // デバッグフォント用のコマンド生成
        pData->pWriter->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
}
