﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <algorithm>
#include <cmath>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../SimpleGfx_Types.h"
#include "../SimpleGfx_TypesInternal.h"
#include "SimpleGfx_ApiInternal.h"

namespace nns { namespace sgx { namespace detail {

namespace
{

ShaderType  g_CurrentShader  = ShaderType::Simple;
Size        g_CanvasSize     = DefaultCanvasSize;

ShaderPack* g_pSimpleShader  = nullptr;
ShaderPack* g_pTextureShader = nullptr;

}  // anonymous

void CreateShader(
    ShaderPack* pOutShader,
    const GLchar* const* vertexProgram,
    const GLchar* const* fragmentProgram) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutShader);
    NN_ASSERT_NOT_NULL(vertexProgram);
    NN_ASSERT_NOT_NULL(fragmentProgram);

    GLint result;
    GLchar shaderLog[1024];
    GLsizei shaderLogSize;

    // Vertex shader
    auto vertexId = glCreateShader(GL_VERTEX_SHADER);
    NN_ASSERT(vertexId != 0, "Failed to create vertex shader\n");
    glShaderSource(vertexId, 1, vertexProgram, 0);
    glCompileShader(vertexId);
    glGetShaderiv(vertexId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(vertexId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile vertex shader: %s\n", shaderLog);
    }

    // Fragment shader
    auto fragmentId = glCreateShader(GL_FRAGMENT_SHADER);
    NN_ASSERT(fragmentId != 0, "Failed to create fragment shader\n");
    glShaderSource(fragmentId, 1, fragmentProgram, 0);
    glCompileShader(fragmentId);
    glGetShaderiv(fragmentId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(fragmentId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile fragment shader: %s\n", shaderLog);
    }

    // コンパイル
    auto programId = glCreateProgram();
    NN_ASSERT(programId != 0, "Failed to create shader program\n");

    glAttachShader(programId, vertexId);
    glAttachShader(programId, fragmentId);
    glLinkProgram(programId);

    pOutShader->vertexId = vertexId;
    pOutShader->fragmentId = fragmentId;
    pOutShader->programId = programId;
}

void DeleteShader(const ShaderPack& shader) NN_NOEXCEPT
{
    glDetachShader(shader.programId, shader.vertexId);
    glDetachShader(shader.programId, shader.fragmentId);

    glDeleteShader(shader.vertexId);
    glDeleteShader(shader.fragmentId);

    glDeleteProgram(shader.programId);
}

void RegisterShader(
    ShaderPack* pSimpleShader,
    ShaderPack* pTextureShader) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pSimpleShader);
    NN_ASSERT_NOT_NULL(pTextureShader);

    g_pSimpleShader  = pSimpleShader;
    g_pTextureShader = pTextureShader;
}

bool ChangeShader(ShaderType type) NN_NOEXCEPT
{
    if (g_CurrentShader == type)
    {
        return false;
    }

    switch (type)
    {
    case ShaderType::None:
        glUseProgram(0);
        break;

    case ShaderType::Simple:
        glUseProgram(g_pSimpleShader->programId);
        break;

    case ShaderType::Texture:
        glUseProgram(g_pTextureShader->programId);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    bool needsRefresh = g_CurrentShader == ShaderType::None;
    g_CurrentShader = type;

    return needsRefresh;
}

Size GetCanvasSize() NN_NOEXCEPT
{
    return g_CanvasSize;
}

void SetCanvasSize(const Size& size) NN_NOEXCEPT
{
    g_CanvasSize = size;
}

}}}  // nns::sgx::detail
