﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include "sgx/SimpleGfx.h"

#define NN_APP_LOG0(log)    NN_LOG("[ControllerFirmwareUpdater] %s", (log))

//!< サポートしているデバイスか判定します。
bool IsSupportedDevice(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

//!< RomFs のマウント処理
void InitializeFileSystem() NN_NOEXCEPT;

//!< RomFs のマウント解除
void FinalizeFileSystem() NN_NOEXCEPT;

//!< コントローラのファームウェアバージョンを読み込むクラスです。
class FirmwareVersionReader final
{
    NN_DISALLOW_COPY(FirmwareVersionReader);
    NN_DISALLOW_MOVE(FirmwareVersionReader);

public:
    FirmwareVersionReader() NN_NOEXCEPT :
        m_Mutex(true),
        m_UniquePadId(),
        m_Version(),
        m_IsValid(false),
        m_HasValidVersion(false)
    {
    }

    // 対象コントローラーの UniquePadId を設定
    void SetUniquePadId(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT;

    // ファームウェアバージョンを無効化
    void Invalidate() NN_NOEXCEPT;

    // 有効化されているか
    bool IsValid() const NN_NOEXCEPT;

    // 有効なバージョンを取得済みか
    bool HasValidVersion() const NN_NOEXCEPT;

    // 読み込んだバージョン情報を取得
    bool Get(nn::hid::system::FirmwareVersion* pOutVersion) const NN_NOEXCEPT;

    // バージョン情報の更新
    void Update() NN_NOEXCEPT;

private:
    void InvalidateVersion() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex m_Mutex;

    nn::hid::system::UniquePadId        m_UniquePadId;
    nn::hid::system::FirmwareVersion    m_Version;
    bool m_IsValid;
    bool m_HasValidVersion;
};

//!< ウェイトアイコン表示を制御するクラスです。
class WaitIcon final
{
    NN_DISALLOW_COPY(WaitIcon);
    NN_DISALLOW_MOVE(WaitIcon);

public:
    WaitIcon() NN_NOEXCEPT :
        m_FrameCount(0)
    {
    }

    void Update() NN_NOEXCEPT
    {
        m_FrameCount++;
    }

    float Draw(float x, float y) NN_NOEXCEPT
    {
        uint32_t spinner[] = { 0xE020u + (m_FrameCount / 6) % 8, 0u };
        nns::sgx::DrawText(x, y, spinner);

        nns::sgx::Size size;
        nns::sgx::GetTextDrawSize(&size, spinner);
        return size.width;
    }

private:
    uint32_t m_FrameCount;
};
