﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <algorithm>
#include <cstdio>

namespace app
{
//----------------------------------------------------------------
// 範囲内での値調整
//    現在値 value から incValue (正負どちらでも可) 動かしたときの結果を返す。
//    min や max を超えた場合は、範囲内でサイクリックに考慮する
//
int RotateWithinRange( int value, int incValue, int min, int max ) NN_NOEXCEPT
{
    NN_ASSERT( max >= min );

    int range = max - min + 1;
    int v = (value + incValue - min) % range;
    return ((v < 0)? (v + range): v) + min;
}

//----------------------------------------------------------------
// 範囲内での値調整
//   現在値 position から incValue (正負どちらでも可) 動かしたときの結果を返す。
//   アイテムは max 個あるのだが、viewTop から viewLen だけが表示されているので
//   position が動けば(position が表示から出るようなら)表示箇所を変える必要がある。
//   したがって position、viewTop が変化する。
//
void RotateWithinViewRange( int& position, int incValue, int& viewTop, int viewLen, int max ) NN_NOEXCEPT
{
    if ( max == 0 )
    {
        return;
    }

    // 現在のアイテム番号
    int curPos = viewTop + position;
    // 新しいアイテム番号
    int newPos = RotateWithinRange( curPos, incValue, 0, max - 1 );
    // 表示数が viewLen より小さい場合、viewLen をその値とみなす
    if ( max < viewLen )
    {
        viewLen = max;
    }

    if ( incValue > 0 )
    {
        // 余裕があるとき
        if ( position + incValue < viewLen )
        {
            position += incValue;
        }
        // 余裕がないのでずらす
        else
        {
            position = (newPos < viewLen - 1)? newPos: (viewLen - 1);
            viewTop = (newPos < viewLen - 1)? 0: (newPos - viewLen + 1);
        }
    }
    else
    {
        // 余裕があるとき
        if ( position + incValue > 0 )
        {
            position += incValue;
        }
        // 余裕がないのでずらす
        else
        {
            position = (newPos <= max - viewLen)? 0: (newPos + viewLen - max);
            viewTop = (newPos <= max - viewLen)? newPos: (max - viewLen);
        }
    }
}

//----------------------------------------------------------------
// 範囲内での値調整
//   現在値 position から incValue (正負どちらでも可) 動かしたときの結果を返す。
//   カーソルは動かない。全体をスクロールさせる。
//   アイテム数は max 個で、表示量は viewLen。
//   範囲外に超えたり回ったりはしない。
//
void MoveWithinFixViewRange( int& position, int incValue, int viewLen, int max ) NN_NOEXCEPT
{
    // 新しい位置
    int newPos = position + incValue;
    // 表示数が viewLen より小さい場合、viewLen をその値とみなす
    if ( max < viewLen )
    {
        viewLen = max;
    }

    if ( newPos >= max - viewLen )
    {
        newPos = max - viewLen;
    }
    if ( newPos < 0 )
    {
        newPos = 0;
    }
    position = newPos;
}

//----------------------------------------------------------------
// キーリピートなどで、カウント値と最大長から、どれだけ増分させるかを決める
//
int GetRepeatCount( int count, int maxLen ) NN_NOEXCEPT
{
    return ( count <  10 )? 1:
           ( count <  15 )? 2:
           ( count <  20 )? 4:
           ( count <  25 )? 8:
           ( count <  30 )? 16:
           ( count <  35 )? 32:
           ( count <  40 )? 64:
           ( count <  45 )? std::max( 100, maxLen / 25 ): std::max( 100, maxLen / 20 );
}

//----------------------------------------------------------------
// スクロールバー表示用の計算
//    全体サイズ、現在位置、表示量から、バーの座標値(始値、終値)を与え
//    現在の表示分がどの位置になるかを計算する。
//    スクロールバーの最低限の大きさが指定される
//
void CalculateScrollBarPosition( int64_t wholeSize, int64_t displaySize, int64_t currentPosition,
                                 float posStart, float posEnd, float minWidth,
                                 float* pStart, float* pEnd ) NN_NOEXCEPT

{
    if ( displaySize >= wholeSize )
    {
        *pStart = posStart;
        *pEnd = posEnd;
        return;
    }

    // 現在分の幅
    float b = std::max( (posEnd - posStart) * displaySize / wholeSize, minWidth );

    // 現在分の位置
    float s = (posEnd - posStart - b) * currentPosition / (wholeSize - displaySize);

    *pStart = posStart + s;
    *pEnd = posStart + s + b;
}

size_t StrLen16( const char16_t* s ) NN_NOEXCEPT
{
    if ( ! s )
    {
        return 0;
    }
    size_t len = 0;
    while( s[len] )
    {
        len ++;
    }
    return len;
}

namespace {
    const size_t VaBufferSize = 1024;
    static char g_VaBuffer[ VaBufferSize ];
    static char16_t g_VaBuffer16[ VaBufferSize ];

    const char16_t Char16Table[] =
        {
            u'\0', u'\x001', u'\x002', u'\x003', u'\x004', u'\x005', u'\x006', u'\x007',
            u'\x008', u'\x009', u'\x00a', u'\x00b', u'\x00c', u'\x00d', u'\x00e', u'\x00f',
            u'\x010', u'\x011', u'\x012', u'\x013', u'\x014', u'\x015', u'\x016', u'\x017',
            u'\x018', u'\x019', u'\x01a', u'\x01b', u'\x01c', u'\x01d', u'\x01e', u'\x01f',
            u' ', u'!', u'\"', u'#', u'$', u'%', u'&', u'\'',
            u'(', u')', u'*', u'L', u',', u'-', u'.', u'/',
            u'0', u'1', u'2', u'3', u'4', u'5', u'6', u'7',
            u'8', u'9', u':', u';', u'<', u'=', u'>', u'?',
            u'@', u'A', u'B', u'C', u'D', u'E', u'F', u'G',
            u'H', u'I', u'J', u'K', u'L', u'M', u'N', u'O',
            u'P', u'Q', u'R', u'S', u'T', u'U', u'V', u'W',
            u'X', u'Y', u'Z', u'[', u'\\', u']', u'^', u'_',
            u'`', u'a', u'b', u'c', u'd', u'e', u'f', u'g',
            u'h', u'i', u'j', u'k', u'l', u'm', u'n', u'o',
            u'p', u'q', u'r', u's', u't', u'u', u'v', u'w',
            u'x', u'y', u'z', u'{', u'|', u'}', u'~', u'?',
        };
} //namespace

//----------------------------------------------------------------
// char 文字列を char16_t に変換
//
char16_t* ConvertToChar16_t( const char* format, ... ) NN_NOEXCEPT
{
    va_list args;
    va_start( args, format );
    vsnprintf( g_VaBuffer, sizeof(g_VaBuffer), format, args );
    va_end( args );

    char* p = g_VaBuffer;
    char16_t* p16 = g_VaBuffer16;
    while( *p )
    {
        *p16 ++ = ( '\0' <= *p && *p <= '~' ) ? Char16Table[ *p - '\0' ] : u'?';
        p ++;
    }
    *p16 = u'\0';

    return g_VaBuffer16;
}

} //namespace app
