﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"
#include "BcatSystemDebugTool_TopMenu.h"
#include "BcatSystemDebugTool_FileMenu.h"
#include "BcatSystemDebugTool_DataInitMenu.h"
#include "BcatSystemDebugTool_AppList.h"
#include "BcatSystemDebugTool_PushNotificationLogMenu.h"
#include "BcatSystemDebugTool_NetworkConnection.h"
#include "BcatSystemDebugTool_OtherMenu.h"

namespace app
{
void ExecTopMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecTopMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecTopMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawTopMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecTopMenuGroup = {
    ExecTopMenu_Initialize,
    ExecTopMenu,
    ExecTopMenu_Finalize,
    nullptr,

    DrawTopMenu,
    nullptr,
    DrawPriority_TopMenu,
    0
};

namespace
{
    app::Menu g_Menu;

    enum
    {
         MenuIndex_AppList = 0,
         MenuIndex_Unmount,
         MenuIndex_Network,
         MenuIndex_PushNotificationLog,
         MenuIndex_StartSubscribing,
         //MenuIndex_InitSaveData,
         MenuIndex_OtherMenu,
    };

    const int TopMenuConsoleTag = 0x100;
    int g_InitialMenuIndex = 0;
    const int MenuMax = MenuIndex_OtherMenu + 1;

} //namespace

//----------------------------------------------------------------
// トップメニュー用描画コールバック
//
void DrawTopMenu( void* arg ) NN_NOEXCEPT
{
    // タイトル枠
    app::DrawFrameRectangle( Position_Title, DrawColorSet_TitleBack, DrawColorSet_TitleFrame, DrawFrameWidth );

    // タイトル文字列
    SetGlvColor( ColorSet_White );
    float title_w;
    float title_h;
    GetStringBounds( TitleString, TitleStringFontSize, 1.5f, &title_w, &title_h );
    glv::draw::text( TitleString,
                     Position_Title.l + (Position_Title.r - Position_Title.l) / 2 - title_w / 2,
                     Position_Title.t + 10,
                     TitleStringFontSize );

    float ver_w;
    float ver_h;
    GetStringBounds( VersionString, VersionStringFontSize, 1.5f, &ver_w, &ver_h );
    glv::draw::text(  VersionString,
                      Position_Title.l + (Position_Title.r - Position_Title.l) / 2 + title_w / 2 - ver_w - 10,
                      Position_Title.t + 38,
                      VersionStringFontSize );
}

//----------------------------------------------------------------
// トップメニューのメニュー文字列描画
//
void PrintTopMenuItems() NN_NOEXCEPT
{
    app::FixedProportionalConsole<char16_t>* p = g_Menu.GetConsole16();
    if ( p )
    {
        // トップメニュー
        p->PrintfEx( 0, 0, app::ConsoleColor_Green,
                     app::GetText( TextResource_TopMenuTitle ) );

        p->SetAttribute( app::ConsoleColor_White );

        // アプリケーション一覧
        p->PrintfEx( 2, 2 + MenuIndex_AppList, app::ConsoleColor_White,
                     app::GetText( TextResource_ListApplications ) );

        // データ配信キャッシュストレージのアンマウント
        p->PrintfEx( 2, 2 + MenuIndex_Unmount,
                     app::IsMountedCacheStorage()? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     app::GetText( TextResource_UnmountStorage ) );

        // ネットワークに接続／切断
        p->PrintfEx( 2, 2 + MenuIndex_Network,
                     app::IsConnected()? app::GetText( TextResource_DisconnectNetwork ): app::GetText( TextResource_ConnectNetwork ) );

        // プッシュ通知ログをみる
        p->PrintfEx( 2, 2 + MenuIndex_PushNotificationLog,
                     app::GetText( TextResource_ViewPushNotificationLog ) );

        // 購読の開始(未実装)
        p->PrintfEx( 2, 2 + MenuIndex_StartSubscribing, app::ConsoleColor_DarkWhite,
                     app::GetText( TextResource_StartSubscribing ) );

#if 0
        p->PrintfEx( 2, 2 + MenuIndex_InitSaveData,
                     app::ConsoleColor_DarkWhite,
                     u"Initialize the save data ... (not implemented)" );
#endif
        // その他
        p->PrintfEx( 2, 2 + MenuIndex_OtherMenu,
                     app::ConsoleColor_White,
                     app::GetText( TextResource_OtherMenu ) );
    }
}

//----------------------------------------------------------------
// トップメニューの内部コンソール作成
//
void CreateTopMenuConsole() NN_NOEXCEPT
{
    // メニュー作成
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_TopMenu, 100, 6, 0, 2, 24, 4, DrawPriority_TopMenu + 1, TopMenuConsoleTag );
    g_Menu.SetBackColor( DrawColorSet_TopMenuBack, DrawColorSet_TopMenuFrame, app::DrawFrameWidth );
    g_Menu.SetItemParameter( MenuMax, g_InitialMenuIndex );
    PrintTopMenuItems();
}

//----------------------------------------------------------------
// トップメニューの内部コンソール破棄
//
void DestroyTopMenuConsole() NN_NOEXCEPT
{
    // 内部コンソールの破棄
    g_Menu.DestroyConsole();
}

//----------------------------------------------------------------
// トップメニューのキー説明
void PrintTopMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpTopMenu ) );
}

//----------------------------------------------------------------
// トップメニュー(開始処理)
//
void ExecTopMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    static bool isFirstCall = true;
    if ( isFirstCall )
    {
        // SD カードからコンフィグ読み込み
        ReadConfigFromSdCard( "save", "save:/BcatConfig.txt" );
        SetEnglish( IsConfigEnglish() );

        isFirstCall = false;
    }

    // ベース情報表示のセットアップ
    SetUpBaseDisplay();

    // メニュー作成
    CreateTopMenuConsole();

    // キー説明
    PrintTopMenuHelp();
}
//----------------------------------------------------------------
// トップメニュー(終了処理)
//
void ExecTopMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    DestroyTopMenuConsole();
}

namespace {
//----------------------------------------------------------------
// 接続／切断切り替えからの戻り
void AfterNetworkConnection( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    PrintTopMenuItems();
    // キー説明
    PrintTopMenuHelp();
}
} //namespace

//----------------------------------------------------------------
// トップメニュー
//
void ExecTopMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー確認
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    switch( g_Menu.CheckButtonOk() )
    {
        case MenuIndex_AppList:
            g_InitialMenuIndex = MenuIndex_AppList;
            app::sequence::JumpTo( ExecAppListMenuGroup, nullptr );
            return;
        case MenuIndex_Unmount:
            UnmountCacheStorage();
            ResetCurrentApplication();
            break;
        case MenuIndex_Network:
            // ネットワーク接続 / 切断
            app::sequence::Call( ExecNetworkConnectionGroup );
            app::sequence::SetFromCall( AfterNetworkConnection, nullptr );
            break;
        case MenuIndex_PushNotificationLog:
            // プッシュ通信ログメニューへ
            g_InitialMenuIndex = MenuIndex_PushNotificationLog;
            app::sequence::JumpTo( ExecPushNotificationLogMenuGroup, nullptr );
            return;
        case MenuIndex_StartSubscribing:
            {
                // 未実装
            }
#if 0
            {
                // ダイアログ実験
                app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
                p->SetMessage( "Failed to mount the Delivery cache storage." );
                app::sequence::Call( ExecDialogGroup, p );
                return;
            }
#endif
            break;

#if 0
        case MenuIndex_InitSaveData:
            {
                if ( IsCurrentApplicationAvailable() )
                {
                    g_InitialMenuIndex = MenuIndex_InitSaveData;
                    app::sequence::JumpTo( ExecDataInitMenuGroup, nullptr );
                    return;
                }
            }
            break;
#endif

        case MenuIndex_OtherMenu:
            {
                g_InitialMenuIndex = MenuIndex_OtherMenu;
                app::sequence::JumpTo( ExecOtherMenuGroup );
                return;
            }
            break;
        default:
            // do nothing
            break;
    }

    if ( g_Menu.IsUpdated() )
    {
        PrintTopMenuItems();
        g_Menu.ClearUpdated();
    }
}

} // namespace app
