﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"

#include "BcatSystemDebugTool_TopMenu.h"
#include "BcatSystemDebugTool_BinaryDump.h"

#include <nn/bcat/bcat_DeliveryCacheDirectory.h>

namespace app
{

void ExecFileListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileListMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileListMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecFileListMenuGroup = {
    ExecFileListMenu_Initialize,
    ExecFileListMenu,
    ExecFileListMenu_Finalize,
    nullptr,

    nullptr,
    nullptr,
    0,
    0
};

void ExecDirectoryListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryListMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryListMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawFileMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecDirectoryListMenuGroup = {
    ExecDirectoryListMenu_Initialize,
    ExecDirectoryListMenu,
    ExecDirectoryListMenu_Finalize,
    nullptr,

    DrawFileMenu,
    nullptr,
    DrawPriority_SubMenu,
    0
};

namespace
{
    enum MenuState
    {
        MenuState_DirectoryList,
        MenuState_FileList,
        MenuState_BinaryDump,
    };

    MenuState g_MenuState = MenuState_DirectoryList;

    nn::bcat::DirectoryName g_DirectoryName[ nn::bcat::DeliveryCacheDirectoryCountMax ];
    nn::bcat::DeliveryCacheDirectoryEntry g_FileEntry[ nn::bcat::DeliveryCacheFileCountMaxPerDirectory ];

    //---- ファイル/ディレクトリリスト用
    const int ConsoleWidth = 78;
    const int ConsoleHeight = 33;
    const int ConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ConsoleWidth, ConsoleHeight );
    char g_ConsoleBuffer[ ConsoleBufferSize ];
    app::FixedConsole<char> g_Console;
    bool g_IsConsoleCreated = false;

    // ディレクトリ用
    const int DirListViewMax = 17;
    int g_DirListCursor = 0;
    int g_DirListViewTop = 0;
    int g_DirListNum = 0;
    bool g_IsDirListUpdate = false;
    int g_CurDirectory = 0;

    // ファイル用
    const int FileListViewMax = 16;
    int g_FileListCursor = 0;
    int g_FileListViewTop = 0;
    int g_FileListNum = 0;
    bool g_IsFileListUpdate = false;
    int g_CurFile = 0;

    // バイナリ表示用
    app::BinaryDumpParam g_BinaryDumpParam;
} //namespace

void  DrawFileConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric( 110, 110, 0, 24, 16, 24, x, y, string, attr, stringLen, arg );
}
//----------------------------------------------------------------
void DrawFileMenu( void* arg ) NN_NOEXCEPT
{
    if ( g_MenuState == MenuState_DirectoryList || g_MenuState == MenuState_FileList || g_MenuState == MenuState_BinaryDump )
    {
        // フレーム描画
        app::DrawFrameRectangle( Position_FileList, DrawColorSet_FileMenuBack, DrawColorSet_FileMenuFrame );
    }

    if ( g_MenuState == MenuState_DirectoryList )
    {
        if ( g_DirListNum > 0 )
        {
            app::DrawRectangle( 152, 208 + (g_DirListCursor * 24), 1000, 233 + (g_DirListCursor * 24), app::DrawColorSet_ItemCursorBack );

            // スクロールバー
            app::DrawScrollBar( g_DirListNum, DirListViewMax, g_DirListViewTop, ScrollBarDefaultViewWidth, Position_FileListScrollBar );
        }
    }
    else if ( g_MenuState == MenuState_FileList )
    {
        if ( g_FileListNum > 0 )
        {
            app::DrawRectangle( 152, 232 + (g_FileListCursor * 24), 1150, 257 + (g_FileListCursor * 24),  app::DrawColorSet_ItemCursorBack );

            // スクロールバー
            app::DrawScrollBar( g_FileListNum, FileListViewMax, g_FileListViewTop, ScrollBarDefaultViewWidth, Position_FileListScrollBar );
        }
    }

    if ( g_IsConsoleCreated )
    {
        if ( g_MenuState == MenuState_DirectoryList || g_MenuState == MenuState_FileList )
        // ディレクトリ／ファイルリストのコンソール
        g_Console.Display();
    }
}

//----------------------------------------------------------------
// ファイルリストをコンソールにプリント
//
void PrintFileList() NN_NOEXCEPT
{
    g_Console.Clear();

    // ディレクトリ名
    g_Console.PrintfEx( 0, 0, app::ConsoleColor_Yellow, "Directory : %s", g_DirectoryName[ g_CurDirectory ].value );

    // ファイル数
    g_Console.PrintfEx( 0, 1, "File List : %d file(s)", g_FileListNum );

    if ( g_FileListNum == 0 )
    {
        g_Console.PrintfEx( 2, 4, app::ConsoleColor_Yellow, "<No files>" );
        return;
    }

    g_Console.PrintfEx( 0, 3, "  No.     File name                      Size       Digest" );
    g_Console.PrintfEx( 0, 4, "  ---  ------------------------------- --------   -----------" );

    // ファイル名
    for( int i=0; i<FileListViewMax; i++ )
    {
        int num = g_FileListViewTop + i;
        if ( num >= g_FileListNum )
        {
            break;
        }
        nn::bcat::DeliveryCacheDirectoryEntry entry = g_FileEntry[ g_FileListViewTop + i ];

        g_Console.PrintfEx( 2, i + 5, "%3d: %s", num, entry.name );
        g_Console.PrintfEx( 39, i + 5, "%8lld", entry.size );

        // ダイジェスト
        char diStr[17];
        nn::util::SNPrintf( diStr, sizeof(diStr), "%016llX", reinterpret_cast<uint64_t>(entry.digest.value[0]) );
        diStr[8] = '\0'; // 8文字しか使用しないので
        g_Console.PrintfEx( 50, i + 5, "%s...", diStr );
    }
}

//----------------------------------------------------------------
// ファイルリストメニューのキー説明
void PrintFileListMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpFileMenu ) );
}

//----------------------------------------------------------------
// Call からの戻り
namespace
{
void Return_FromCall( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::Return();
}

void SetStateFileList_FromCall( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_MenuState = MenuState_FileList;
    g_IsFileListUpdate = true;
    PrintFileListMenuHelp();
}
} //namespace

//----------------------------------------------------------------
// ファイルリスト(初回処理)
//
void ExecFileListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_FileListNum = -1;

    app::ScrollConsole<char>& c = GetSystemConsole();
    g_Console.Clear();

    nn::bcat::DeliveryCacheDirectory d;
    // 現在ディレクトリ g_CurDirectory はここに来る前にディレクトリリストがセットする

    // オープン
    c.Printf("nn::bcat::DeliveryCacheDirectory::Open() :");
    nn::Result result = d.Open( g_DirectoryName[ g_CurDirectory ] );
    app::PrintErrorCode( result );
    if ( result.IsFailure() )
    {
        app::SetErrorDialog( u"Failed to open directory.", result );
        app::sequence::SetFromCall( Return_FromCall, nullptr );
        return;
    }

    // カウント
    c.Printf("nn::bcat::DeliveryCacheDirectory::GetCount() :");
    int count = d.GetCount();
    c.Printf("%d\n", count);
    g_FileListNum = count;
    g_FileListViewTop = 0;
    g_FileListCursor = 0;

    // エントリ取得
    c.Printf("nn::bcat::DeliveryCacheDirectory::Read() :");
    int outCount;
    result = d.Read( &outCount, g_FileEntry, nn::bcat::DeliveryCacheFileCountMaxPerDirectory );
    app::PrintErrorCode( result );
    if ( result.IsFailure() )
    {
        g_FileListNum = -1;
        d.Close();
        app::SetErrorDialog( u"Failed to read file entries.", result );
        app::sequence::SetFromCall( Return_FromCall, nullptr );
        return;
    }

    g_IsFileListUpdate = true;
    PrintFileListMenuHelp();
}
//----------------------------------------------------------------
// ファイルリスト(終了処理)
//
void ExecFileListMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
}
//----------------------------------------------------------------
// ファイルリスト
//
void ExecFileListMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // A ボタンでファイル詳細メニューへ
    if ( pad.IsButtonDownA() && g_FileListNum > 0 )
    {
        g_MenuState = MenuState_BinaryDump;
        g_CurFile = g_FileListViewTop + g_FileListCursor; // これから開くファイル

        g_BinaryDumpParam.pDirName = reinterpret_cast<const nn::bcat::DirectoryName*>(&g_DirectoryName[ g_CurDirectory ]); // ディレクトリ名
        g_BinaryDumpParam.pFileEntry = reinterpret_cast<const nn::bcat::DeliveryCacheDirectoryEntry*>(&g_FileEntry[ g_CurFile ]); // ファイルエントリ

        app::sequence::Call( ExecFileDetailMenuGroup, reinterpret_cast<void*>(&g_BinaryDumpParam) );
        app::sequence::SetFromCall( SetStateFileList_FromCall, nullptr );
        return;
    }

    // X ボタンでダンプ表示
    if ( pad.IsButtonDownX() && g_FileListNum > 0 )
    {
        g_MenuState = MenuState_BinaryDump;
        g_CurFile = g_FileListViewTop + g_FileListCursor; // これから開くファイル

        g_BinaryDumpParam.pDirName = reinterpret_cast<const nn::bcat::DirectoryName*>(&g_DirectoryName[ g_CurDirectory ]); // ディレクトリ名
        g_BinaryDumpParam.pFileEntry = reinterpret_cast<const nn::bcat::DeliveryCacheDirectoryEntry*>(&g_FileEntry[ g_CurFile ]); // ファイルエントリ

        app::sequence::Call( ExecBinaryDumpGroup, reinterpret_cast<void*>(&g_BinaryDumpParam) );
        app::sequence::SetFromCall( SetStateFileList_FromCall, nullptr );
        return;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::Return();
        return;
    }

    // カーソル上
    if ( (pad.IsButtonDownUp() || pad.IsButtonRepeatUp()) &&  g_FileListNum > 0 )
    {
        RotateWithinViewRange( g_FileListCursor, -1, g_FileListViewTop, FileListViewMax, g_FileListNum );
        g_IsFileListUpdate = true;
    }
    // カーソル下
    if ( (pad.IsButtonDownDown() || pad.IsButtonRepeatDown()) && g_FileListNum > 0 )
    {
        RotateWithinViewRange( g_FileListCursor, 1, g_FileListViewTop, FileListViewMax, g_FileListNum );
        g_IsFileListUpdate = true;
    }
    // カーソル左
    if ( pad.IsButtonDownLeft() && g_FileListNum > 0 )
    {
        g_FileListCursor = 0;
        g_FileListViewTop = 0;
        g_IsFileListUpdate = true;
    }
    // カーソル右
    if ( pad.IsButtonDownRight() && g_FileListNum > 0 )
    {
        g_FileListCursor = (g_FileListNum < FileListViewMax)? (g_FileListNum - 1): (FileListViewMax - 1);
        g_FileListViewTop = (g_FileListNum < FileListViewMax)? 0: (g_FileListNum - FileListViewMax);
        g_IsFileListUpdate = true;
    }

    // リストの更新
    if ( g_IsFileListUpdate )
    {
        g_IsFileListUpdate = false;
        PrintFileList();

        // ファイルがあるならカーソル表示
        if ( g_FileListNum > 0 )
        {
            g_Console.PrintfEx( 1, g_FileListCursor + 5, app::ConsoleColor_Yellow, ">" );
        }
    }
} //NOLINT(impl/function_size)

//----------------------------------------------------------------
// ディレクトリ一覧の取得
//
void EnumerateDirectory() NN_NOEXCEPT
{
    // ディレクトリ情報取得
    app::GetSystemConsole().Printf( "nn::bcat::EnumerateDeliveryCacheDirectory() :" );
    int outCount;
    nn::Result result = nn::bcat::EnumerateDeliveryCacheDirectory( &outCount, g_DirectoryName, nn::bcat::DeliveryCacheDirectoryCountMax );
    app::PrintErrorCode( result );

    if ( result.IsSuccess() )
    {
        app::GetSystemConsole().Printf( "directory num = %d\n", outCount );
        g_MenuState = MenuState_DirectoryList;
        g_DirListNum = outCount;
        g_DirListViewTop = 0;
        g_DirListCursor = 0;
    }
}
//----------------------------------------------------------------
// ファイル用コンソール作成
//
void CreateFileConsole() NN_NOEXCEPT
{
    g_Console.SetBuffer( g_ConsoleBuffer, ConsoleBufferSize, ConsoleWidth, ConsoleHeight );
    g_Console.SetPrinter( DrawFileConsoleString, nullptr );
    g_IsConsoleCreated = true;
}
//----------------------------------------------------------------
// ディレクトリリストをコンソールにプリント
//
void PrintDirectoryList() NN_NOEXCEPT
{
    g_Console.Clear();

    // ディレクトリ数
    g_Console.PrintfEx( 0, 0, "Directory List : %d dir(s)", g_DirListNum );

    if ( g_DirListNum == 0 )
    {
        g_Console.PrintfEx( 2, 3, app::ConsoleColor_Yellow, "<No directories>" );
        return;
    }

    g_Console.PrintfEx( 0, 2, "  No.     Directory name" );
    g_Console.PrintfEx( 0, 3, "  ---  -------------------------------" );

    // ディレクトリ名
    for( int i=0; i<DirListViewMax; i++ )
    {
        int num = g_DirListViewTop + i;
        if ( num >= g_DirListNum )
        {
            break;
        }
        g_Console.PrintfEx( 2, i + 4, "%3d: %s", num, g_DirectoryName[ g_DirListViewTop + i ].value );
    }
}

//----------------------------------------------------------------
// ディレクトリリストメニューのキー説明
void PrintDirectoryListMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpFileMenuDir ) );
}

//----------------------------------------------------------------
// Call からの戻り
namespace
{
void SetStateDirectoryList_FromCall( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::PrintDirectoryListMenuHelp();
    g_MenuState = MenuState_DirectoryList;
    g_IsDirListUpdate = true;
}
} // namespace

//----------------------------------------------------------------
// ディレクトリリスト(初回処理)
void ExecDirectoryListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    CreateFileConsole();
    PrintDirectoryListMenuHelp();

    EnumerateDirectory();
    g_IsDirListUpdate = true;
}
//----------------------------------------------------------------
// ディレクトリリスト(終了処理)
void ExecDirectoryListMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_IsConsoleCreated = false;
}
//----------------------------------------------------------------
// ディレクトリリスト
void ExecDirectoryListMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // A ボタンでディレクトリを開く
    if ( pad.IsButtonDownA() && g_DirListNum > 0 )
    {
        g_MenuState = MenuState_FileList;
        g_CurDirectory = g_DirListViewTop + g_DirListCursor; // これから開くディレクトリ
        g_FileListNum = 0;
        app::sequence::Call( ExecFileListMenuGroup );
        app::sequence::SetFromCall( SetStateDirectoryList_FromCall, nullptr );
        return;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::JumpTo( ExecTopMenuGroup, nullptr );
    }

    // カーソル上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        RotateWithinViewRange( g_DirListCursor, -1, g_DirListViewTop, DirListViewMax, g_DirListNum );
        g_IsDirListUpdate = true;
    }
    // カーソル下
    if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        RotateWithinViewRange( g_DirListCursor, 1, g_DirListViewTop, DirListViewMax, g_DirListNum );
        g_IsDirListUpdate = true;
    }
    // カーソル左
    if ( pad.IsButtonDownLeft() )
    {
        g_DirListCursor = 0;
        g_DirListViewTop = 0;
        g_IsDirListUpdate = true;
    }
    // カーソル右
    if ( pad.IsButtonDownRight() )
    {
        g_DirListCursor = (g_DirListNum < DirListViewMax)? (g_DirListNum - 1): (DirListViewMax - 1);
        g_DirListViewTop = (g_DirListNum < DirListViewMax)? 0: (g_DirListNum - DirListViewMax);
        g_IsDirListUpdate = true;
    }

    // リストの更新
    if ( g_IsDirListUpdate )
    {
        g_IsDirListUpdate = false;
        PrintDirectoryList();

        // ディレクトリがあるならカーソル表示
        if ( g_DirListNum > 0 )
        {
            g_Console.PrintfEx( 1, g_DirListCursor + 4, app::ConsoleColor_Yellow, ">" );
        }
    }
}

} // namespace app

