﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"

#include "BcatSystemDebugTool_TopMenu.h"
#include "BcatSystemDebugTool_ApplicationMenu.h"
#include "BcatSystemDebugTool_FileMenu.h"
#include "BcatSystemDebugTool_SaveAllFile.h"

#include <nn/bcat/bcat_TypesDebug.h>
#include <nn/bcat/bcat_ApiDebug.h>

#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>

#include <vector>

//================================================================
// デバッグ用途のフラグ
// これを定義すると、表示実験のためにたくさん読み込んだことにする
//#define APP_PRETEND_TO_READ_MANY_FOR_DEBUG
//================================================================

namespace app
{

void ExecAppListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecAppListMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecAppListMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawAppList( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecAppListMenuGroup = {
    ExecAppListMenu_Initialize,
    ExecAppListMenu,
    ExecAppListMenu_Finalize,
    nullptr,

    DrawAppList,
    nullptr,
    DrawPriority_SubMenu,
    0
};

void ExecAppListSubMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecAppListSubMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecAppListSubMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecAppListSubMenuGroup = {
    ExecAppListSubMenu_Initialize,
    ExecAppListSubMenu,
    ExecAppListSubMenu_Finalize,
    nullptr,

    nullptr,
    nullptr,
    0,
    0
};

void ExecAppListMenu_AfterApplicationMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecAppListMenu_DumpToSd( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;;
void FromInputString( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;

namespace
{
    // タスク一覧
    nn::bcat::TaskInfo g_TaskInfo[ nn::bcat::TaskCountMax ];
    int g_TaskInfoNum;

    // コンソール
    const int ConsoleWidth = 100;
    const int ConsoleHeight = 33;
    const int ConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ConsoleWidth, ConsoleHeight );
    char g_ConsoleBuffer[ ConsoleBufferSize ];
    app::FixedConsole<char> g_Console;

    // アプリリスト表示用
    const int AppListViewMax = 15;
    int g_AppListCursor = 0;
    int g_AppListViewTop = 0;
    int g_AppListNum = 0;
    bool g_IsAppListUpdate = false;

    // アプリリスト
    struct ApplicationListInfo
    {
        uint64_t saveDataId;
        nn::ApplicationId applicationId;
        uint32_t applicationVersion;
        uint64_t saveDataSize;
        nn::fs::SaveDataSpaceId spaceId;

        int taskIndex; // タスク一覧の中にあるなら 0 以上の値。ない場合は負数
        bool isSaveDataExist; // セーブデータが存在するか

        // 進捗
        nn::bcat::DeliveryCacheProgress* pProgress;
        nn::Result lastResult;
        bool hasDone;

        ApplicationListInfo() NN_NOEXCEPT :
            saveDataId( 0LL ),
            applicationId( { 0LL } ),
            applicationVersion( 0 ),
            saveDataSize( 0LL ),
            spaceId( nn::fs::SaveDataSpaceId::User ),
            taskIndex( -1 ),
            isSaveDataExist( 0 ),
            pProgress( nullptr ),
            lastResult( nn::ResultSuccess() ),
            hasDone( false )
        {
        }

        void SetSaveDataParameter( uint64_t saveDataId0, nn::ApplicationId applicationId0, uint64_t saveDataSize0, nn::fs::SaveDataSpaceId spaceId0 ) NN_NOEXCEPT
        {
            saveDataId = saveDataId0;
            applicationId = applicationId0;
            saveDataSize = saveDataSize0;
            spaceId = spaceId0;
        }
        void SetExist( bool st ) NN_NOEXCEPT
        {
            isSaveDataExist = st;
        }
        void SetTaskIndex( int index ) NN_NOEXCEPT
        {
            taskIndex = index;
        }
        void SetApplicationVersion( uint32_t version ) NN_NOEXCEPT
        {
            applicationVersion = version;
        }
        nn::ApplicationId GetApplicationId() NN_NOEXCEPT
        {
            return applicationId;
        }
    };
    std::vector<ApplicationListInfo*> g_ApplicationListInfoList;
    const int ApplicationListMax = 128;
    int g_ApplicationListNum;

    ApplicationListInfo* GetApplicationListByIndex( int index ) NN_NOEXCEPT
    {
        return (index > g_ApplicationListInfoList.size())? nullptr: g_ApplicationListInfoList[ index ];
    }

    // アプリケーションメニュー呼び出し用
    app::ApplicationMenuParam g_ApplicationMenuParam;

    // アプリリストソート
    enum SortType
    {
        SortType_ByApplicationId,
        SortType_ByTaskStatus,
    };
    SortType g_CurrentSortType = SortType_ByTaskStatus;

    // コンソールと一緒に描画する GLV オブジェクト
    struct GlvObjectInfo
    {
        enum
        {
            GlvObjectInfoType_ProgressBar = 0,
            GlvObjectInfoType_ProgressBase,
        };

        int positionX;
        int positionY;
        int sizeX;
        int sizeY;
        int rate;
        int type;

        void SetProgressBarParameter( int positionX0, int positionY0, int rate0 )  NN_NOEXCEPT
        {
            positionX = positionX0;
            positionY = positionY0;
            rate = rate0;
            type = GlvObjectInfo::GlvObjectInfoType_ProgressBar;
        }
        void SetProgressBaseParameter( int positionX0, int positionY0, int sizeX0, int sizeY0 )  NN_NOEXCEPT
        {
            positionX = positionX0;
            positionY = positionY0;
            sizeX = sizeX0;
            sizeY = sizeY0;
            type = GlvObjectInfo::GlvObjectInfoType_ProgressBase;
        }
    };
    std::list<GlvObjectInfo*> g_GlvObjectInfoList;

    // プログレスコンソール
    const int ProgressConsoleWidth = 100;
    const int ProgressConsoleHeight = 33;
    const int ProgressConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ProgressConsoleWidth, ProgressConsoleHeight );
    char g_ProgressConsoleBuffer[ ProgressConsoleBufferSize ];
    app::FixedProportionalConsole<char> g_ProgressConsole;
    bool g_IsProgressConsoleDrawing = false;

    // 個別プログレス表示スイッチ
    bool g_IsDisplayFileProgress = false;

    char g_DirectoryName[ 32 ];
    char g_SdDirectoryName[ app::PathSdLengthMax ];
    app::SaveAllFileParam g_SaveAllFileParam;

} // namespace

// 前方参照用
void ExecAppListSubMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void SortAppListByApplicationId() NN_NOEXCEPT;
void SortAppListByTaskStatus() NN_NOEXCEPT;

//----------------------------------------------------------------
// アプリリストコンソールのプリンタ
//
void  DrawAppListConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric( 80, 110, 2, 24, 16, 24, x, y, string, attr, stringLen, arg );
}
//----------------------------------------------------------------
// プログレスコンソールのプリンタ
//
void  DrawProgressConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawProportionalConsoleGeneric( 132, 217, 2, 18, 16, 24, x, y, string, attr, stringLen, arg );
}

//----------------------------------------------------------------
// アプリリストの描画コールバック
//
void DrawAppList( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrameRectangle( Position_AppList, DrawColorSet_AppListBack, DrawColorSet_AppListFrame );

    if ( g_IsProgressConsoleDrawing )
    {
    }

    if ( g_AppListNum > 0 )
    {
        app::DrawRectangle( 122, 216 + (g_AppListCursor * (24 + 2)), 1125, 241 + (g_AppListCursor * (24 + 2)), DrawColorSet_ItemCursorBack );
    }

    g_Console.Display();

    for( auto iter = g_GlvObjectInfoList.begin(); iter != g_GlvObjectInfoList.end(); iter++ )
    {
        GlvObjectInfo* p = *iter;

        if ( p->type == GlvObjectInfo::GlvObjectInfoType_ProgressBase )
        {
            // 進捗バック
            app::RectanglePosition backPosition( p->positionX, p->positionY, p->positionX + p->sizeX, p->positionY + p->sizeY );
            app::DrawRectangle( backPosition, DrawColorSet_FileProgressBack );
        }

        if ( p->type == GlvObjectInfo::GlvObjectInfoType_ProgressBar )
        {
            // 枠
            app::RectanglePosition framePosition( p->positionX, p->positionY, p->positionX + 40, p->positionY + 12 );
            app::DrawFrame( framePosition, DrawColorSet_ProgressBarFrame );

            // 中身
            int rw = 37.F * p->rate / 100.F;
            app::RectanglePosition currentPosition( p->positionX + 2, p->positionY + 2, p->positionX + rw, p->positionY + 10 );
            app::DrawRectangle( currentPosition, DrawColorSet_ProgressBar );
        }

        delete p;
    }
    g_GlvObjectInfoList.clear();

    // スクロールバー
    app::DrawScrollBar( g_AppListNum, AppListViewMax, g_AppListViewTop, ScrollBarDefaultViewWidth, Position_AppListScrollBar );

    // プログレスコンソール描画
    if (  g_IsDisplayFileProgress )
    {
        g_ProgressConsole.Display();
    }
}

//----------------------------------------------------------------
// アプリリスト用のコンソールを準備する
//
void CreateAppListConsole() NN_NOEXCEPT
{
    g_Console.SetBuffer( g_ConsoleBuffer, ConsoleBufferSize, ConsoleWidth, ConsoleHeight );
    g_Console.SetPrinter( DrawAppListConsoleString );

    g_ProgressConsole.SetBuffer( g_ProgressConsoleBuffer, ProgressConsoleBufferSize, ProgressConsoleWidth, ProgressConsoleHeight );
    g_ProgressConsole.SetPrinter( DrawProgressConsoleString );
}

//----------------------------------------------------------------
// アプリリスト用のカーソルをコンソールにプリントする
//
void PrintAppCursor() NN_NOEXCEPT
{
    g_Console.PrintfEx( 1, g_AppListCursor + 4, app::ConsoleColor_Yellow, ">" );
}

//----------------------------------------------------------------
// アプリ一覧用：bcat の購読状態による表示の色と文字列
namespace
{
    struct ResourceForSubscriptionStatusDisplay
    {
        nn::bcat::SubscriptionStatus status;
        int                         attribute;
        const char*                 string;
    };
    const ResourceForSubscriptionStatusDisplay ResourceListForSubscriptionStatusDisplay[] = {
        { nn::bcat::SubscriptionStatus_Empty,      app::ConsoleColor_DarkWhite, "---" },
        { nn::bcat::SubscriptionStatus_Subscribed, app::ConsoleColor_White,     "Sub" },
        { nn::bcat::SubscriptionStatus_Error,      app::ConsoleColor_Red,       "Err" },
    };
    const int   ResourceAttributeForSubscriptionStatusDisplay_Default = app::ConsoleColor_Magenta;
    const char* ResourceStringForSubscriptionStatusDisplay_Default = "???";
} //namespace

void GetPrintResourceForSubscriptionStatus( nn::bcat::SubscriptionStatus st, int* pAttribute, const char** pString ) NN_NOEXCEPT
{
    NN_ASSERT( pAttribute != nullptr );
    NN_ASSERT( pString != nullptr );
    for( int i=0; i < sizeof(ResourceListForSubscriptionStatusDisplay) / sizeof(ResourceForSubscriptionStatusDisplay); i++ )
    {
        if ( st == ResourceListForSubscriptionStatusDisplay[i].status )
        {
            *pAttribute = ResourceListForSubscriptionStatusDisplay[i].attribute;
            *pString    = ResourceListForSubscriptionStatusDisplay[i].string;
            return;
        }
    }
    *pAttribute = ResourceAttributeForSubscriptionStatusDisplay_Default;
    *pString    = ResourceStringForSubscriptionStatusDisplay_Default;
}

//----------------------------------------------------------------
// アプリ一覧用：bcat のバックグラウンドタスクの状態による表示の色と文字列
namespace
{
    struct ResourceForBgTaskStatusDisplay
    {
        nn::bcat::TaskStatus status;
        int                  attribute;
        const char*          string;
    };
    const ResourceForBgTaskStatusDisplay ResourceListForBgTaskStatusDisplay[] = {
        { nn::bcat::TaskStatus::TaskStatus_Empty,                     app::ConsoleColor_Magenta, "----" },
        { nn::bcat::TaskStatus::TaskStatus_Runnable,                  app::ConsoleColor_Yellow,  "Runnable" },
        { nn::bcat::TaskStatus::TaskStatus_Running,                   app::ConsoleColor_Yellow,  "Running" },
        { nn::bcat::TaskStatus::TaskStatus_Wait,                      app::ConsoleColor_Cyan,    "Wait" },
        { nn::bcat::TaskStatus::TaskStatus_Done,                      app::ConsoleColor_White,   "Done" },
        { nn::bcat::TaskStatus::TaskStatus_Error,                     app::ConsoleColor_Red,     "Error" },
    };
    const int   ResourceAttributeForBgTaskStatusDisplay_Default = app::ConsoleColor_Red;
    const char* ResourceStringForBgTaskStatusDisplay_Default = "< Unknown >";
} //namespace

void GetPrintResourceForBgTaskStatus( nn::bcat::TaskStatus st, int* pAttribute, const char** pString ) NN_NOEXCEPT
{
    NN_ASSERT( pAttribute != nullptr );
    NN_ASSERT( pString != nullptr );
    for( int i=0; i < sizeof(ResourceListForBgTaskStatusDisplay) / sizeof(ResourceForBgTaskStatusDisplay); i++ )
    {
        if ( st == ResourceListForBgTaskStatusDisplay[i].status )
        {
            *pAttribute = ResourceListForBgTaskStatusDisplay[i].attribute;
            *pString    = ResourceListForBgTaskStatusDisplay[i].string;
            return;
        }
    }
    *pAttribute = ResourceAttributeForBgTaskStatusDisplay_Default;
    *pString    = ResourceStringForBgTaskStatusDisplay_Default;
}

//----------------------------------------------------------------
// アプリ一覧用：bcat のデータ配信の進捗状態による表示の色と文字列
namespace
{
    struct ResourceForDeliveryCacheProgressStatusDisplay
    {
        nn::bcat::DeliveryCacheProgressStatus status;
        int                                   attribute;
        const char*                           string;
    };
    const ResourceForDeliveryCacheProgressStatusDisplay ResourceListForDeliveryCacheProgressStatusDisplay[] = {
        { nn::bcat::DeliveryCacheProgressStatus_None,        app::ConsoleColor_DarkWhite,  "-" },
        { nn::bcat::DeliveryCacheProgressStatus_Queued,      app::ConsoleColor_Yellow,  "Queued" },
        { nn::bcat::DeliveryCacheProgressStatus_Connect,     app::ConsoleColor_Magenta, "Connect" },
        { nn::bcat::DeliveryCacheProgressStatus_ProcessList, app::ConsoleColor_Magenta, "ProcList" },
        { nn::bcat::DeliveryCacheProgressStatus_Download,    app::ConsoleColor_Green,   "Download" },
        { nn::bcat::DeliveryCacheProgressStatus_Commit,      app::ConsoleColor_Cyan,    "Commit" },
        { nn::bcat::DeliveryCacheProgressStatus_Done,        app::ConsoleColor_White,   "Done" },
    };
    const int   ResourceAttributeForDeliveryCacheProgressStatusDisplay_Default = app::ConsoleColor_Red;
    const char* ResourceStringForDeliveryCacheProgressStatusDisplay_Default = "??";
} //namespace

void GetPrintResourceForDeliveryCacheProgress( nn::bcat::DeliveryCacheProgressStatus st, int* pAttribute, const char** pString ) NN_NOEXCEPT
{
    NN_ASSERT( pAttribute != nullptr );
    NN_ASSERT( pString != nullptr );
    for( int i=0; i < sizeof(ResourceListForDeliveryCacheProgressStatusDisplay) / sizeof(ResourceForDeliveryCacheProgressStatusDisplay); i++ )
    {
        if ( st == ResourceListForDeliveryCacheProgressStatusDisplay[i].status )
        {
            *pAttribute = ResourceListForDeliveryCacheProgressStatusDisplay[i].attribute;
            *pString    = ResourceListForDeliveryCacheProgressStatusDisplay[i].string;
            return;
        }
    }
    *pAttribute = ResourceAttributeForDeliveryCacheProgressStatusDisplay_Default;
    *pString    = ResourceStringForDeliveryCacheProgressStatusDisplay_Default;
}

//----------------------------------------------------------------
// コンソールと一緒に描画する GLV オブジェクトを追加：プログレスバー
//
void AddProgressBar( int64_t wholeSize, int64_t currentSize, int positionX, int positionY )
{
    float rate = currentSize * 100.F / wholeSize;
    rate = (rate < 0.F)? 0.F: (rate>100.F)? 100.F: rate;

    GlvObjectInfo* p = new GlvObjectInfo;
    p->SetProgressBarParameter( positionX, positionY, rate );
    g_GlvObjectInfoList.push_back( p );
}
//----------------------------------------------------------------
// コンソールと一緒に描画する GLV オブジェクトを追加：プログレスベース
//
void AddProgressBase( int positionX, int positionY, int sizeX, int sizeY )
{
    GlvObjectInfo* p = new GlvObjectInfo;
    p->SetProgressBaseParameter( positionX, positionY, sizeX, sizeY );
    g_GlvObjectInfoList.push_back( p );
}

//----------------------------------------------------------------
// BG タスクの状態をコンソールにプリント
//
void PrintBgTaskStatus( int taskIndex, int position ) NN_NOEXCEPT
{
    g_Console.PrintfEx( 28, position + 4, app::ConsoleColor_White, "%8d", g_TaskInfo[ taskIndex ].appVersion );

    int attribute;
    const char* string;

    // 購読状態をコンソールにプリント
    nn::bcat::SubscriptionStatus ssst = g_TaskInfo[ taskIndex ].subscription;
    GetPrintResourceForSubscriptionStatus( ssst, &attribute, &string );
    g_Console.PrintfEx( 38, position + 4, attribute, "%s", string );

    // ステータスをコンソールにプリント
    nn::bcat::TaskStatus st = g_TaskInfo[ taskIndex ].status;
    GetPrintResourceForBgTaskStatus( st, &attribute, &string );
    g_Console.PrintfEx( 42, position + 4, attribute, "%s", string );
}
//----------------------------------------------------------------
// BG タスクの Progress の状態をコンソールにプリント
//
void PrintBgTaskProgressStatus( int position, ApplicationListInfo* pApp ) NN_NOEXCEPT
{
    nn::bcat::DeliveryCacheProgressStatus dst = (pApp->pProgress)? pApp->pProgress->GetStatus() :
                                                (pApp->hasDone)?   nn::bcat::DeliveryCacheProgressStatus_Done :
                                                                   nn::bcat::DeliveryCacheProgressStatus_None;

    int attribute;
    const char* string;
    GetPrintResourceForDeliveryCacheProgress( dst, &attribute, &string );
    if ( dst == nn::bcat::DeliveryCacheProgressStatus_Done && pApp->lastResult.IsFailure() )
    {
        attribute = app::ConsoleColor_Red;
        string = "Error";
    }
    g_Console.PrintfEx( 57, position + 4, attribute, "%s", string );

    switch( dst )
    {
        case nn::bcat::DeliveryCacheProgressStatus_Download:  // 状態 Download 時の描画
            {
                if ( g_IsDisplayFileProgress )
                {
                    int64_t r0 = pApp->pProgress->GetCurrentTotal();
                    int64_t r1 = pApp->pProgress->GetCurrentDownloaded();
                    float rate = r1 * 100.F / r0;
                    rate = (rate < 0.F)? 0.F: (rate>100.F)? 100.F: rate;

                    g_ProgressConsole.PrintfEx( 0, position, app::ConsoleColor_White, "%s/%s",
                                                pApp->pProgress->GetCurrentDirectoryName().value,
                                                pApp->pProgress->GetCurrentFileName().value );

                    g_ProgressConsole.PrintfEx( 30, position, app::ConsoleColor_White, "%8lld/%8lld",
                                                r1, r0 );

                    g_ProgressConsole.PrintfEx( 45, position, app::ConsoleColor_White, "%3d%%", static_cast<int>(rate) );

                    AddProgressBase( 128, 215 + position * 26, 852, 29 );
                    AddProgressBar( pApp->pProgress->GetCurrentTotal(), pApp->pProgress->GetCurrentDownloaded(), 910, 224 + position * 26 );
                }
                AddProgressBar( pApp->pProgress->GetWholeTotal(), pApp->pProgress->GetWholeDownloaded(), 1130, 224 + position * 26 );
            }
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Commit: // 状態 Commit 時の描画
            {
                if ( g_IsDisplayFileProgress )
                {
                    g_ProgressConsole.PrintfEx( 0, position, app::ConsoleColor_White, "%s/",
                                                pApp->pProgress->GetCurrentDirectoryName().value );
                    AddProgressBase( 128, 215 + position * 26, 852, 29 );
                }
                AddProgressBar( pApp->pProgress->GetWholeTotal(), pApp->pProgress->GetWholeDownloaded(), 1130, 224 + position * 26 );
            }
            break;
        default:
            break;
    }
}

//----------------------------------------------------------------
// アプリリスト一覧をコンソールにプリントする
//
void PrintSubscribedAppList() NN_NOEXCEPT
{
    g_Console.Clear();
    g_ProgressConsole.Clear();

    // アプリ数
    g_Console.PrintfEx( 0, 0, "Application List : %d app(s)", g_AppListNum );

    if ( g_AppListNum == 0 )
    {
        g_Console.PrintfEx( 2, 2, app::ConsoleColor_Yellow, "<No applications>" );
        return;
    }

    // アプリ一覧
    g_Console.PrintfEx( 0, 2, "  No.    Application ID      AppVer   Sub  Status        Progress" );
    g_Console.PrintfEx( 0, 3, "  ---  ------------------   --------  --- -------------- --------" );
    for( int viewIndex = 0; viewIndex < AppListViewMax; viewIndex ++ )
    {
        int num = g_AppListViewTop + viewIndex;
        if ( num >= g_AppListNum )
        {
            break;
        }
        int consolePositionY = viewIndex + 4;

        ApplicationListInfo* pApp = GetApplicationListByIndex( num );
        NN_ASSERT( pApp );

        g_Console.PrintfEx( 2, consolePositionY, app::ConsoleColor_White, "%3d:", num );

        // アプリID
        uint64_t appId = pApp->applicationId.value;
        int taskIndex = pApp->taskIndex;

        //g_Console.SetAttribute( (g_AppList[ num ].isSaveDataExist == false )? app::ConsoleColor_Green :
        g_Console.SetAttribute( (pApp->isSaveDataExist == false )? app::ConsoleColor_Green :
                                (taskIndex < 0 )?                  app::ConsoleColor_White :
                                                                   app::ConsoleColor_White );
        g_Console.PrintfEx( 7, consolePositionY, "0x%016llx", appId );

        // BG タスクがある場合
        if ( taskIndex >= 0 )
        {
            // タスクの状態表示と、Progress の状態表示
            PrintBgTaskStatus( taskIndex, viewIndex );
            PrintBgTaskProgressStatus( viewIndex, pApp );
        }
        else
        {
            g_Console.PrintfEx( 28, consolePositionY, app::ConsoleColor_DarkWhite, "       -  --- no task" );
        }

        // 選択されているアプリはマークを付ける
        if ( app::IsCurrentApplicationAvailable() && app::GetCurrentApplication().value == appId )
        {
            g_Console.PrintfEx( 26, consolePositionY, "*" );
        }
    }
}

//----------------------------------------------------------------
// 進捗を解放する
//
void ReleaseAllProgress() NN_NOEXCEPT
{
    // GLV オブジェクトクリア
    for( auto iter = g_GlvObjectInfoList.begin(); iter != g_GlvObjectInfoList.end(); iter++ )
    {
        GlvObjectInfo* p = *iter;
        delete p;
    }
    g_GlvObjectInfoList.clear();

    // アプリリストクリア
    for( auto iter = g_ApplicationListInfoList.begin(); iter != g_ApplicationListInfoList.end(); iter++ )
    {
        ApplicationListInfo* p = *iter;

        if ( p->pProgress )
        {
            delete p->pProgress;
        }
        delete p;
    }
    g_ApplicationListInfoList.clear();
}

#ifdef APP_PRETEND_TO_READ_MANY_FOR_DEBUG
//----------------------------------------------------------------
// (DEBUG) 表示の実験のためにたくさん読み込んだことにする
//
void PretendToReadManyTaskInfo() NN_NOEXCEPT
{
    for( int i=1; i<25; i++ )
    {
        g_TaskInfo[i].appId.value = 0x240000 * i;
        g_TaskInfo[i].appVersion = (i * 2) % 10;
        g_TaskInfo[i].subscription = nn::bcat::SubscriptionStatus_Error;

        switch(i % 6)
        {
            case 0: g_TaskInfo[i].status = nn::bcat::TaskStatus_Empty; break;
            case 1: g_TaskInfo[i].status = nn::bcat::TaskStatus_Runnable; break;
            case 2: g_TaskInfo[i].status = nn::bcat::TaskStatus_Running; break;
            case 3: g_TaskInfo[i].status = nn::bcat::TaskStatus_Wait; break;
            case 4: g_TaskInfo[i].status = nn::bcat::TaskStatus_Done; break;
            default: g_TaskInfo[i].status = nn::bcat::TaskStatus_Error; break;
        }
    }
    g_TaskInfoNum = 25;
}
#endif

//----------------------------------------------------------------
// アプリケーション一覧の作成
//
nn::Result EnumerateApplicationList() NN_NOEXCEPT
{
    app::ScrollConsole<char>& c = GetSystemConsole();

    // リソース初期化
    ReleaseAllProgress();

    std::unique_ptr< nn::fs::SaveDataIterator > saveIter;
    nn::fs::SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::User;

    // オープン
    c.Printf("nn::fs::OpenSaveDataIterator() :" );
    nn::Result result = nn::fs::OpenSaveDataIterator( &saveIter, spaceId );
    app::PrintErrorCode( result );
    NN_RESULT_DO( result );

    // セーブデータリストを順次読む
    nn::fs::SaveDataInfo info[1];
    int64_t count;

    g_ApplicationListNum = 0;

#ifdef APP_PRETEND_TO_READ_MANY_FOR_DEBUG
    // (DEBUG) 表示の実験のためにたくさん読み込んだことにする
    int overRead = 18;
    int tmpNum = 0;
#endif

    while( NN_STATIC_CONDITION( true ) )
    {
        result = saveIter->ReadSaveDataInfo( &count, info, 1 );
        if ( result.IsFailure() )
        {
            c.Printf("nn::fs::ReadSaveDataInfo() :" );
            app::PrintErrorCode( result );
            return result;
        }

#ifdef APP_PRETEND_TO_READ_MANY_FOR_DEBUG
        // (DEBUG) 表示の実験のためにたくさん読み込んだことにする
        if ( count == 0 && overRead > 0 )
        {
            overRead --;
            tmpNum ++;

            info[0].saveDataType = nn::fs::SaveDataType::Bcat;
            info[0].saveDataId = 0x000000001 + tmpNum;
            info[0].applicationId.value = 0x120000 * tmpNum;
            info[0].saveDataSize = 300 * tmpNum;
            info[0].saveDataSpaceId = nn::fs::SaveDataSpaceId::User;

            count = 1;
        }
#endif
        if ( count == 0 )
        {
            break;
        }

        // BCAT タイプのものだけをピックアップしていく
        if ( info[0].saveDataType == nn::fs::SaveDataType::Bcat && g_ApplicationListNum < ApplicationListMax )
        {
            ApplicationListInfo* p = new ApplicationListInfo;
            nn::ApplicationId appId = { info[0].applicationId.value };
            p->SetSaveDataParameter( info[0].saveDataId, appId, info[0].saveDataSize, info[0].saveDataSpaceId );
            p->SetExist( true );
            g_ApplicationListInfoList.push_back( p );

            g_ApplicationListNum ++;
        }
    }

    // 全部成功だったことをプリント
    c.Printf("All of nn::fs::ReadSaveDataInfo() :" );
    result = nn::ResultSuccess();
    app::PrintErrorCode( result );

    // 次に BCAT のタスク一覧を読み込む
    c.Printf("nn::bcat::EnumrateBackgroundDeliveryTask() :" );
    result = nn::bcat::EnumerateBackgroundDeliveryTask( &g_TaskInfoNum, g_TaskInfo, NN_ARRAY_SIZE(g_TaskInfo) );
    app::PrintErrorCode( result );

#ifdef APP_PRETEND_TO_READ_MANY_FOR_DEBUG
    // (DEBUG) 表示の実験のためにたくさん読み込んだことにする
    PretendToReadManyTaskInfo();
#endif

    // セーブデータから作成したリストと、BCAT タスク一覧を突き合わせて
    // タスク一覧に各リストの ID が存在するのならインデックスを覚える。
    // タスクがない場合は インデックスとして -1 を入れておく。
    for( auto iter = g_ApplicationListInfoList.begin(); iter != g_ApplicationListInfoList.end(); iter++ )
    {
        ApplicationListInfo* p = *iter;
        for( int j=0; j<g_TaskInfoNum; j++ )
        {
            if ( g_TaskInfo[j].appId == p->GetApplicationId() )
            {
                p->SetTaskIndex( j );
                p->SetApplicationVersion( g_TaskInfo[j].appVersion );
                break;
            }
        }
    }

    // 逆にタスク一覧にあるが、セーブデータが存在しない場合もある。
    // それを考慮する。
    for( int j=0; j<g_TaskInfoNum; j++ )
    {
        uint64_t id = g_TaskInfo[j].appId.value;
        bool isMatched = false;
        for( auto iter = g_ApplicationListInfoList.begin(); iter != g_ApplicationListInfoList.end(); iter++ )
        {
            ApplicationListInfo* p = *iter;
            if ( p->GetApplicationId().value == id )
            {
                isMatched = true;
                break;
            }
        }
        if ( ! isMatched && g_ApplicationListNum < ApplicationListMax )
        {
            // リストに追加する
            ApplicationListInfo* p = new ApplicationListInfo;
            p->SetSaveDataParameter( 0, g_TaskInfo[j].appId, 0, nn::fs::SaveDataSpaceId::User/*dummy*/ );
            p->SetExist( false );
            p->SetTaskIndex( j );
            g_ApplicationListInfoList.push_back( p );

            g_ApplicationListNum ++;
        }
    }

    // リスト用変数の調整
    c.Printf( "Background download task num = %d\n", g_ApplicationListNum );
    g_AppListNum = g_ApplicationListNum;
    g_AppListViewTop = 0;
    g_AppListCursor = 0;
    g_IsAppListUpdate = true;

    // ソートしておく
    if ( g_CurrentSortType == SortType_ByApplicationId )
    {
        SortAppListByApplicationId();
    }
    else if ( g_CurrentSortType == SortType_ByTaskStatus )
    {
        SortAppListByTaskStatus();
    }

    return nn::ResultSuccess();
} //NOLINT(impl/function_size)

//----------------------------------------------------------------
// アプリリストメニューのキー説明
void PrintAppListMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpAppList ) );
}

//----------------------------------------------------------------
// 文字列入力のキー説明
void PrintInputStrHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpAppListInputStr ) );
}


//----------------------------------------------------------------
namespace
{
void AfterCall_FromSubMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_IsAppListUpdate = true;
    PrintAppListMenuHelp();
}

void AfterCall_ToAppList( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_IsAppListUpdate = true;
    PrintAppListMenuHelp();
    app::sequence::SlideTo( ExecAppListMenu );
}

void AfterCall_ToTopMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::JumpTo( ExecTopMenuGroup, nullptr );
}

void AfterCall_UnmountToAppList( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::UnmountCacheStorage();
    app::ResetCurrentApplication();
    PrintAppListMenuHelp();
    app::sequence::SlideTo( ExecAppListMenu );
}
} //namespace

//----------------------------------------------------------------
// アプリリストメニューのキー処理
//   戻り値 false なら戻ってすぐに抜けるべき
bool ProcessKeyForAppLisMenu( const glv::HidEvents& events )
{
    app::Pad pad( events );

    // A ボタンでアプリケーションメニューへ
    if ( pad.IsButtonDownA() )
    {
        if ( g_AppListNum > 0 )
        {
            g_ApplicationMenuParam.m_PositionX = 300;

            g_ApplicationMenuParam.m_PositionY = ( g_AppListCursor < 7 )?
                240 + g_AppListCursor * 26:
                  4 + g_AppListCursor * 26;

            //g_ApplicationMenuParam.m_ApplicationId = g_AppList[ g_AppListViewTop + g_AppListCursor ].applicationId;
            //g_ApplicationMenuParam.m_IsTaskExist = (g_AppList[ g_AppListViewTop + g_AppListCursor ].taskIndex >= 0);

            ApplicationListInfo* pApp = GetApplicationListByIndex( g_AppListViewTop + g_AppListCursor );
            g_ApplicationMenuParam.m_ApplicationId = pApp->GetApplicationId().value;
            g_ApplicationMenuParam.m_IsTaskExist = (pApp->taskIndex >= 0);

            app::sequence::Call( ExecApplicationMenuGroup, &g_ApplicationMenuParam, 0 );
            app::sequence::SlideTo( ExecAppListMenu_AfterApplicationMenu );
            return false;
        }
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::JumpTo( ExecTopMenuGroup, nullptr );
        return false;
    }

    // Y ボタンで個別表示切替
    if ( pad.IsButtonDownY() )
    {
        g_IsDisplayFileProgress = (g_IsDisplayFileProgress == false);
        return false;
    }

    // X ボタンでサブメニュー
    if ( pad.IsButtonDownX() )
    {
        app::sequence::Call( ExecAppListSubMenuGroup );
        app::sequence::SetFromCall( AfterCall_FromSubMenu, nullptr );
        return false;
    }

    // カーソル上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        RotateWithinViewRange( g_AppListCursor, -1, g_AppListViewTop, AppListViewMax, g_AppListNum );
        g_IsAppListUpdate = true;
    }
    // カーソル下
    if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        RotateWithinViewRange( g_AppListCursor, 1, g_AppListViewTop, AppListViewMax, g_AppListNum );
        g_IsAppListUpdate = true;
    }
    // カーソル左
    if ( pad.IsButtonDownLeft() )
    {
        g_AppListCursor = 0;
        g_AppListViewTop = 0;
        g_IsAppListUpdate = true;
    }
    // カーソル右
    if ( pad.IsButtonDownRight() )
    {
        g_AppListCursor = (g_AppListNum < AppListViewMax)? (g_AppListNum - 1): (AppListViewMax - 1);
        g_AppListViewTop = (g_AppListNum < AppListViewMax)? 0: (g_AppListNum - AppListViewMax);
        g_IsAppListUpdate = true;
    }
    return true;
}

//----------------------------------------------------------------
// リスト一覧をコンソールへのプリント
//
void UpdateAppListConsole() NN_NOEXCEPT
{
    if ( ! g_IsAppListUpdate )
    {
        return;
    }

    g_IsAppListUpdate = false;

    // アプリ一覧
    PrintSubscribedAppList();

    // アプリがあるならカーソル表示
    if ( g_AppListNum > 0 )
    {
        PrintAppCursor();
    }
}

//----------------------------------------------------------------
// Progress の更新
//
void UpdateAllProgress() NN_NOEXCEPT
{
    int taskInfoNum;
    nn::Result result = nn::bcat::EnumerateBackgroundDeliveryTask( &taskInfoNum, g_TaskInfo, NN_ARRAY_SIZE(g_TaskInfo) );
    if ( result.IsFailure() )
    {
        NN_LOG("nn::bcat::EnumerateBackgroundDeliveryTask() : Failed\n");
        return;
    }
    if ( taskInfoNum != g_TaskInfoNum )
    {
        EnumerateApplicationList();
    }

#ifdef APP_PRETEND_TO_READ_MANY_FOR_DEBUG
    // (DEBUG) 表示の実験のためにたくさん読み込んだことにする
    PretendToReadManyTaskInfo();
#endif

    // 常に g_TaskUpdateInfo は同じ位置になると思われるが
    // 念のため ApplicationList と突き合わせる
    for( auto iter = g_ApplicationListInfoList.begin(); iter != g_ApplicationListInfoList.end(); iter++ )
    {
        bool isFound = false;
        ApplicationListInfo* p = *iter;
        for( int j=0; j<g_TaskInfoNum; j++ )
        {
            if ( g_TaskInfo[j].appId == p->GetApplicationId() )
            {
                isFound = true;
                p->SetTaskIndex( j );

                if ( p->pProgress )
                {
                    p->pProgress->Update();
                    if ( p->pProgress->GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Done )
                    {
                        p->lastResult = p->pProgress->GetResult();
                        p->hasDone = true;

                        app::GetSystemConsole().Printf("Application ID = 0x%016llx\n", p->GetApplicationId().value );
                        app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheProgres::GetResult() :");
                        app::PrintErrorCode( p->lastResult );

                        // リソース確保のため、Done になったら pProgress を削除する
                        delete p->pProgress;
                        p->pProgress = nullptr;
                    }
                }
                break;
            }
        }
        // pProgress が初期化されているのに見つからなかった場合は削除しておく
        if ( ! isFound && p->pProgress )
        {
            delete p->pProgress;
            p->pProgress = nullptr;
            p->SetTaskIndex( -1 );
        }
    }

    g_IsAppListUpdate =true;
}

//----------------------------------------------------------------
// アプリリストメニューの実行コールバック(初回処理)
//
void ExecAppListMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // アプリリスト用コンソール準備
    CreateAppListConsole();

    // キー説明
    PrintAppListMenuHelp();

    // アプリ一覧取得
    nn::Result result = EnumerateApplicationList();
    if ( result.IsFailure() )
    {
        app::SetErrorDialog( u"Failed to read bcat save data.", result );
        app::sequence::SetFromCall( AfterCall_ToTopMenu, nullptr );
        return;
    }

    g_IsAppListUpdate = true;
}
//----------------------------------------------------------------
// アプリリストメニューの実行コールバック(終了処理)
//
void ExecAppListMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    ReleaseAllProgress();
}
//----------------------------------------------------------------
// アプリリストメニュー
//
void ExecAppListMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // パッド処理
    if ( ! ProcessKeyForAppLisMenu( events ) )
    {
        return;
    }

    // Progress の更新
    UpdateAllProgress();

    // アプリリストのコンソール更新
    UpdateAppListConsole();
}

//----------------------------------------------------------------
// アプリリストメニュー(一覧取得)
//
void ExecAppListMenu_EnumerateList( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // アプリ一覧取得
    nn::Result result = EnumerateApplicationList();
    if ( result.IsFailure() )
    {
        app::SetErrorDialog( u"Failed to read bcat save data.", result );
        app::sequence::SetFromCall( AfterCall_ToTopMenu, nullptr );
    }

    g_IsAppListUpdate = true;
    app::sequence::SlideTo( ExecAppListMenu );
}

//----------------------------------------------------------------
// アプリリストメニュー(アプリケーションメニューからの戻り)
//
void ExecAppListMenu_AfterApplicationMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    ApplicationListInfo* pApp = GetApplicationListByIndex( g_AppListViewTop + g_AppListCursor );

    // 現在のアプリケーションID とアプリケーションバージョン
    nn::ApplicationId appId = pApp->applicationId;
    uint32_t appVersion = pApp->applicationVersion;
    app::ApplicationMenuReturnParam* p = reinterpret_cast<app::ApplicationMenuReturnParam*>( app::sequence::GetReturnArg() );
    NN_ASSERT( p );

    // キャンセルされた
    if ( p && p->m_IsCanceled )
    {
        PrintAppListMenuHelp();
        app::sequence::SlideTo( ExecAppListMenu );
        return;
    }

    // ディレクトリ一覧へ進む指示
    if ( p && p->m_IsListDirectory )
    {
        // マウント
        app::UnmountCacheStorage();
        app::ResetCurrentApplication();
        app::SetCurrentApplication( appId );
        nn::Result result = app::MountCacheStorage();
        if ( result.IsSuccess() )
        {
            app::sequence::JumpTo( ExecDirectoryListMenuGroup, nullptr );
            return;
        }
        else
        {
            // 対象アプリのクリア
            ResetCurrentApplication();

            // エラーダイアログ
            app::SetErrorDialog( u"Failed to mount this application storage.", result );
            app::sequence::SetFromCall( AfterCall_ToAppList, nullptr );
            return;
        }
    }

    // 即時同期処理の指示
    if ( p && p->m_IsRequestSync )
    {
        app::GetSystemConsole().Printf("nn::bcat::RequestSyncDeliveryCache( 0x%016llx, %u ) :", appId.value, appVersion );

        if ( pApp->pProgress )
        {
            delete pApp->pProgress;
            pApp->pProgress = nullptr;
        }

        nn::bcat::DeliveryCacheProgress* pProgress = new nn::bcat::DeliveryCacheProgress;
        NN_ASSERT_NOT_NULL( pProgress );

        nn::Result result = nn::bcat::RequestSyncDeliveryCache( pProgress, appId, appVersion );
        app::PrintErrorCode( result );

        if ( result.IsSuccess() )
        {
            pApp->pProgress = pProgress;
            pApp->lastResult = nn::ResultSuccess();
            pApp->hasDone = false;

            app::sequence::SlideTo( ExecAppListMenu );
            return;
        }
        else
        {
            delete pProgress;
            pApp->lastResult = result;
            pApp->hasDone = true;

            // エラーダイアログ
            app::SetErrorDialog( u"Failed to request to sync delivery cache.", result );
            app::sequence::SetFromCall( AfterCall_ToAppList, nullptr );
            return;
        }
    }
    // 即時同期のキャンセルの指示
    if ( p && p->m_IsCancelSyncRequest )
    {
        app::GetSystemConsole().Printf("nn::bcat::CancelSyncDeliveryCacheRequest\n" );
        nn::bcat::CancelSyncDeliveryCacheRequest();

        app::sequence::SlideTo( ExecAppListMenu_EnumerateList );
    }
    // タスクの Unregister 指示
    if ( p && p->m_IsUnregisterTask )
    {
        app::GetSystemConsole().Printf("nn::bcat::UnregisterBackgroundDeliveryTask( 0x%016llx ) :", appId.value );
        nn::Result result = nn::bcat::UnregisterBackgroundDeliveryTask( appId );
        app::PrintErrorCode( result );
        if ( result.IsSuccess() )
        {
            // 読み直し
            app::sequence::SlideTo( ExecAppListMenu_EnumerateList );
            return;
        }
        else
        {
            // エラーダイアログ
            app::SetErrorDialog( u"Failed to unregister background delivery task.", result );
            app::sequence::SetFromCall( AfterCall_ToAppList, nullptr );
            return;
        }
    }
    // SD カードへのダンプ指示
    if ( p && p->m_IsDumpToSdCard )
    {
        PrintInputStrHelp();
        app::sequence::SlideTo( ExecAppListMenu_DumpToSd );
        return;
    }
    // ストレージクリア指示
    if ( p && p->m_IsClearStorage )
    {
        app::GetSystemConsole().Printf("nn::bcat::ClearDeliveryCacheStorage( 0x%016llx ) :", appId.value );
        nn::Result result = nn::bcat::ClearDeliveryCacheStorage( appId );
        app::PrintErrorCode( result );
        if ( result.IsSuccess() )
        {
            // 読み直し
            app::sequence::SlideTo( ExecAppListMenu_EnumerateList );
            return;
        }
        else
        {
            // エラーダイアログ
            app::SetErrorDialog( u"Failed to clear delivery cache storage.", result );
            app::sequence::SetFromCall( AfterCall_ToAppList, nullptr );
            return;
        }
    }

    PrintAppListMenuHelp();
    app::sequence::SlideTo( ExecAppListMenu );
} //NOLINT(impl/function_size)

//----------------------------------------------------------------
// SDカードのデフォルト保存フォルダ名を時刻から取得
//
void GetSdCardFolder( char* pName ) NN_NOEXCEPT
{
    int year, month, day, hour, minute, second;

    if ( ! GetCurrentTime( &year, &month, &day, &hour, &minute, &second ) )
    {
        strcpy( pName, "XXXX-XX-XX_XX-XX-XX" );
        return;
    }

    sprintf( pName, "%4d-%02d-%02d_%02d-%02d-%02d", year, month, day, hour, minute, second );
}

//----------------------------------------------------------------
// アプリリストメニュー(SDカードへのダンプ)
//
void ExecAppListMenu_DumpToSd( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    nn::ApplicationId appId = GetApplicationListByIndex( g_AppListViewTop + g_AppListCursor )->applicationId;
    app::InputStringParam* p = app::AllocInputStringParam();

    p->GetConsole()->PrintfEx( 0, 0, GetText( TextResource_SdCardDir1 ) );
    p->GetConsole()->PrintfEx( 0, 1, app::ConvertToChar16_t( "( sd:/bcat/DeliveryCache/%lx/", appId.value ) );
    p->GetConsole()->PrintfEx( 0, 2, GetText( TextResource_SdCardDir2 ) );

    p->SetLength( 31 );
    p->SetReturnDelay( 5 );
    p->SetMessageLines( 4 ); // メッセージ領域

    char s[32];
    GetSdCardFolder( s );
    p->SetInitialString( s );

    app::sequence::Call( app::ExecInputStringGroup, p );
    app::sequence::SetFromCall( FromInputString, nullptr );
}

//----------------------------------------------------------------
// 文字列入力ダイアログからの戻り
//  (ストレージメニューに戻るか、SDカード保存を行うか)
//
void FromInputString( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::InputStringReturnParam* p = reinterpret_cast<app::InputStringReturnParam*>( app::sequence::GetReturnArg() );

    // キャンセルだった
    if ( ! p->IsValid() )
    {
        PrintAppListMenuHelp();
        app::sequence::SlideTo( ExecAppListMenu );
        return;
    }

    // 文字列を入れていない(キャンセル扱い)
    if ( strlen( p->GetString() ) == 0 )
    {
        app::DialogParam* dp = app::AllocDialogParam( app::DialogParam::InfoType_Caution );
        // "ディレクトリ名が不正です"
        dp->SetMessage16( GetText( TextResource_AppListInvalidDirName ) );
        app::sequence::Call( app::ExecDialogGroup, dp );
        app::sequence::SlideTo( ExecAppListMenu );
        return;
    }
    strncpy( g_DirectoryName, p->GetString(), 32 );
    g_DirectoryName[31] = '\0';

    // マウントをしておく
    nn::ApplicationId appId = GetApplicationListByIndex( g_AppListViewTop + g_AppListCursor )->applicationId;
    app::SetCurrentApplication( appId );
    nn::Result result = app::MountCacheStorage();
    if ( result.IsFailure() )
    {
        // "マウントに失敗しました"
        app::SetErrorDialog( GetText( TextResource_AppListFailedMount ), result );
        app::sequence::SlideTo( AfterCall_UnmountToAppList );
        return;
    }

    // SDカードへ保存
    g_SaveAllFileParam.saveTarget = app::SaveAllFileTarget_SdCard;

    // SD カードに保存するディレクトリ名( bcat/DeliveryCache/<AppId=16byte>/<InputString=max31byte> )
    nn::util::SNPrintf( g_SdDirectoryName, sizeof(g_SdDirectoryName), "bcat/DeliveryCache/%016llx/%s", appId.value, g_DirectoryName );
    g_SaveAllFileParam.pDirectoryName = g_SdDirectoryName;

    app::sequence::Call( app::ExecSaveAllFileGroup, &g_SaveAllFileParam );
    app::sequence::SlideTo( AfterCall_UnmountToAppList );
}

//================================================================================
// サブメニュー
namespace
{
    app::Menu g_Menu;

    enum SubMenuItem
    {
        MenuItem_SortByApplicationId = 0,
        MenuItem_SortByTaskStatus,
    };
} //namespace

//----------------------------------------------------------------
void AppListSubMenuConsoleCallback( app::FixedProportionalConsole<char16_t>* console, void* ) NN_NOEXCEPT
{
    console->PrintfEx( 1, 0, app::ConsoleColor_Yellow, GetText( TextResource_SortAppListTitle ) );
    console->SetAttribute( app::ConsoleColor_White );
    console->PrintfEx( 2, 2 + MenuItem_SortByApplicationId, (g_CurrentSortType == SortType_ByApplicationId)?
                       GetText( TextResource_SortAppListByAppCurrent ) : GetText( TextResource_SortAppListByApp ) );
    console->PrintfEx( 2, 2 + MenuItem_SortByTaskStatus, (g_CurrentSortType == SortType_ByTaskStatus)?
                       GetText( TextResource_SortAppListByBgTaskCurrent ) : GetText( TextResource_SortAppListByBgTask ) );
}

//----------------------------------------------------------------
// サブメニューのキー説明
//
void PrintAppListSubMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpAppListSubMenu ) );
}

//----------------------------------------------------------------
// ソートの比較関数
//
bool CompFunction_ForSortByApplicationId( ApplicationListInfo* lhs, ApplicationListInfo* rhs ) NN_NOEXCEPT
{
    return ( lhs->applicationId.value < rhs->applicationId.value );
}
bool CompFunction_ForShowBgTaskUpward( ApplicationListInfo* lhs, ApplicationListInfo* rhs ) NN_NOEXCEPT
{
    int lhsVal = (! lhs->isSaveDataExist)? 1: (lhs->taskIndex >= 0)? 0: 2;
    int rhsVal = (! rhs->isSaveDataExist)? 1: (rhs->taskIndex >= 0)? 0: 2;

    if ( lhsVal != rhsVal )
    {
        return (lhsVal < rhsVal);
    }

    if ( lhsVal == 1 || lhsVal == 0 )
    {
        int s1 = static_cast<int>( g_TaskInfo[lhs->taskIndex].status );
        int s2 = static_cast<int>( g_TaskInfo[rhs->taskIndex].status );

        if ( s1 != s2 )
        {
            return (s1 < s2);
        }
    }
    return ( lhs->applicationId.value < rhs->applicationId.value );
}

//----------------------------------------------------------------
// ソート： アプリID でアプリリストをソート
//
void SortAppListByApplicationId() NN_NOEXCEPT
{
    sort( g_ApplicationListInfoList.begin(), g_ApplicationListInfoList.end(), CompFunction_ForSortByApplicationId );
}

//----------------------------------------------------------------
// ソート： タスク状態でアプリリストをソート
//
void SortAppListByTaskStatus() NN_NOEXCEPT
{
    sort( g_ApplicationListInfoList.begin(), g_ApplicationListInfoList.end(), CompFunction_ForShowBgTaskUpward );
}

//----------------------------------------------------------------
// アプリリストメニューのサブメニュー(初回処理)
//
void ExecAppListSubMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    const int AppListConsoleTag = 0x200;
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_AppListSubMenu, 80, 3, 1, 2, 24, 4, app::DrawPriority_SubMenu + 1, AppListConsoleTag );
    g_Menu.SetItemParameter( 2, 0 );
    g_Menu.SetCallback16( AppListSubMenuConsoleCallback, nullptr );

    // 説明
    PrintAppListSubMenuHelp();
}
//----------------------------------------------------------------
// アプリリストメニューのサブメニュー(終了処理)
//
void ExecAppListSubMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}
//----------------------------------------------------------------
// アプリリストメニューのサブメニュー
//
void ExecAppListSubMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー確認
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    switch( g_Menu.CheckButtonOk() )
    {
        case MenuItem_SortByApplicationId:
            {
                g_CurrentSortType = SortType_ByApplicationId;
                SortAppListByApplicationId();
                app::sequence::Return();
                return;
            }
            break;
        case MenuItem_SortByTaskStatus:
            {
                g_CurrentSortType = SortType_ByTaskStatus;
                SortAppListByTaskStatus();
                app::sequence::Return();
                return;
            }
            break;
        default:
            break;
    }

    //---- キャンセル
    if ( g_Menu.CheckButtonCancel() )
    {
        app::sequence::Return();
        return;
    }
}

} //namespace app
