﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/os.h>
#include <cstdlib>

namespace
{
    const int DefaultCaptureCount = 30;
    const ::nn::TimeSpan DefaultCaptureInterval = ::nn::TimeSpan::FromMilliSeconds(1000);
    const ::nn::capsrv::AlbumStorageType DefaultCaptureStorage = ::nn::capsrv::AlbumStorage_Nand;
    const ::nn::TimeSpan SamplingSpan = ::nn::TimeSpan::FromMilliSeconds(15);
    //SamplingSpan + AdditionalPressTime > 30 ms 以上になるようにする
    const ::nn::TimeSpan AdditionalPressTime = ::nn::TimeSpan::FromMilliSeconds(35);

    void ActiveCaptureButton(
        ::nn::hid::debug::CaptureButtonAutoPilotState* pState) NN_NOEXCEPT
    {
        pState->buttons.Reset();
        pState->buttons.Set<
            ::nn::hid::system::CaptureButton::Active>(true);
    }

    void InactiveCaptureButton(
        ::nn::hid::debug::CaptureButtonAutoPilotState* pState) NN_NOEXCEPT
    {
        pState->buttons.Reset();
        pState->buttons.Set<
            ::nn::hid::system::CaptureButton::Active>(false);
    }

} // namespace

extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    int captureCount = DefaultCaptureCount;
    ::nn::TimeSpan captureInterval = DefaultCaptureInterval;
    ::nn::capsrv::AlbumStorageType captureStorage = DefaultCaptureStorage;

    if (argc == 2)
    {
        captureCount = atoi(argv[1]);
    }
    else  if (argc == 3)
    {
        captureCount = atoi(argv[1]);
        captureInterval = ::nn::TimeSpan::FromMilliSeconds(
            static_cast<int64_t>(atoi(argv[2])));
    }
    else if (argc == 4)
    {
        captureCount = atoi(argv[1]);
        captureInterval = ::nn::TimeSpan::FromMilliSeconds(
            static_cast<int64_t>(atoi(argv[2])));

        if (strcmp(argv[3], "--SD") == 0)
        {
            NN_LOG("Output Storage is changed for SD\n");
            captureStorage = ::nn::capsrv::AlbumStorage_Sd;
        }
    }
    else  if (argc >= 5)
    {
        NN_LOG("ERROR: Too many argvs.\n");
        return;
    }
    const ::nn::TimeSpan proceedTime = (SamplingSpan + SamplingSpan + AdditionalPressTime);
    captureInterval = captureInterval - proceedTime;

    if (captureCount < 0)
    {
        NN_LOG("ERROR: Capture count is negative integer.\n");
        return;
    }

    if (captureInterval.GetMilliSeconds() < 0)
    {
        NN_LOG("ERROR: Interval is too small. needs at least %d (ms)\n", proceedTime.GetMilliSeconds());
        return;
    }

    ::nn::capsrv::InitializeAlbumAccess();
    for(::nn::capsrv::AlbumStorageType storage = 0; storage < ::nn::capsrv::AlbumStorageCount; storage++)
    {
        if (storage != captureStorage)
        {
            ::nn::capsrv::ForceAlbumUnmounted(storage);
        }
        else
        {
            ::nn::capsrv::ResetAlbumMountStatus(storage);
        }
    }
    ::nn::capsrv::RefreshAlbumCache(captureStorage);
    ::nn::capsrv::FinalizeAlbumAccess();

    ::nn::hid::system::InitializeCaptureButton();

    ::nn::hid::debug::CaptureButtonAutoPilotState autoPilotState;

    for (int i = 0; i < captureCount; ++i)
    {
        ActiveCaptureButton(&autoPilotState);
        ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(SamplingSpan + AdditionalPressTime);

        InactiveCaptureButton(&autoPilotState);
        ::nn::hid::debug::SetCaptureButtonAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(SamplingSpan);

        ::nn::os::SleepThread(captureInterval);
    }

    ::nn::hid::debug::FinalizeCaptureButton();
}
