﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fs_ImageDirectory.h>

enum Action
{
    Action_PrintHelp,
    Action_Upload,
    Action_Download,
    Action_Clean,
};


class ProgramOption
{
public:
    static const size_t PathSizeMax = 512;

public:
    ProgramOption()
        : m_Action()
        , m_IsHostDirectorySpecified(false)
        , m_IsForceEnabled(false)
        , m_IsRawModeEnabled(false)
    {
    }

    // @brief コマンドラインオプションを解析します。
    static bool TryParseArgument(ProgramOption* pOutValue, int argc, char** argv) NN_NOEXCEPT;

    Action GetAction() const NN_NOEXCEPT
    {
        return m_Action;
    }

    const nn::util::optional<nn::fs::ImageDirectoryId>& GetStorage() const NN_NOEXCEPT
    {
        return m_Storage;
    }

    const char* GetStorageName() const NN_NOEXCEPT
    {
        if(m_Storage == nullptr)
        {
            return "NotSpecified";
        }
        switch(*m_Storage)
        {
        case nn::fs::ImageDirectoryId::Nand:
            {
                return "NAND";
            }
        case nn::fs::ImageDirectoryId::SdCard:
            {
                return "SD";
            }
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetHostDirectory() const NN_NOEXCEPT
    {
        return m_IsHostDirectorySpecified ? m_HostDirectory : nullptr;
    }

    bool IsForceEnabled() const NN_NOEXCEPT
    {
        return m_IsForceEnabled;
    }

    bool IsRawModeEnabled() const NN_NOEXCEPT
    {
        return m_IsRawModeEnabled;
    }

private:
    Action m_Action;

    nn::util::optional<nn::fs::ImageDirectoryId> m_Storage;

    bool m_IsHostDirectorySpecified;
    char m_HostDirectory[PathSizeMax];

    bool m_IsForceEnabled;
    bool m_IsRawModeEnabled;
};
