﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumFileMaker_FileIo.h"

#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "AlbumFileMaker_Log.h"

namespace nn{ namespace album{

    nn::Result LoadFileImpl(std::vector<char>* pOutData, const std::string& path) NN_NOEXCEPT
    {
         NN_RESULT_DO(nn::fs::MountHostRoot());
         NN_UTIL_SCOPE_EXIT{ nn::fs::UnmountHostRoot(); };

         nn::fs::FileHandle h = {};
         NN_RESULT_DO(nn::fs::OpenFile(&h, path.c_str(), nn::fs::OpenMode_Read));
         NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(h); };

         int64_t size = 0;
         NN_RESULT_DO(nn::fs::GetFileSize(&size, h));

         pOutData->resize(static_cast<size_t>(size));

         NN_RESULT_DO(nn::fs::ReadFile(h, 0, pOutData->data(), pOutData->size()));

         NN_RESULT_SUCCESS;
    }

    bool LoadFile(std::vector<char>* pOutData, const std::string& path) NN_NOEXCEPT
    {
        auto result = LoadFileImpl(pOutData, path);

        if(result.IsFailure())
        {
            NN_ALBUM_LOG_ERR("failed to load file(%d-%d)\n", result.GetModule(), result.GetDescription());
            NN_ALBUM_LOG_ERR("  path: %s\n", path.c_str());
            return false;
        }

        return true;
    }

    //-----------------------------------------

    nn::Result SaveFileImpl(const std::string& path, const std::vector<char>& data) NN_NOEXCEPT
    {
         NN_RESULT_DO(nn::fs::MountHostRoot());
         NN_UTIL_SCOPE_EXIT{ nn::fs::UnmountHostRoot(); };

         NN_RESULT_DO(nn::fs::CreateFile(path.c_str(), static_cast<int64_t>(data.size())));

         nn::fs::FileHandle h = {};
         NN_RESULT_DO(nn::fs::OpenFile(&h, path.c_str(), nn::fs::OpenMode_Write));
         NN_UTIL_SCOPE_EXIT{
             nn::fs::FlushFile(h);
             nn::fs::CloseFile(h);
         };

         NN_RESULT_DO(nn::fs::WriteFile(h, 0, data.data(), data.size(), nn::fs::WriteOption::MakeValue(0)));

         NN_RESULT_SUCCESS;
    }

    bool SaveFile(const std::string& path, const std::vector<char>& data) NN_NOEXCEPT
    {
        auto result = SaveFileImpl(path, data);

        if(result.IsFailure())
        {
            NN_ALBUM_LOG_ERR("failed to save file(%d-%d)\n", result.GetModule(), result.GetDescription());
            NN_ALBUM_LOG_ERR("  path: %s\n", path.c_str());
            return false;
        }

        return true;
    }


}}

