﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ldr/ldr_Result.public.h>
#include <nn/lr/lr_Result.public.h>
#include <nn/ae.h>

#include <nn/manu/server/manu_Log.h>

#include "manu_Command.h"

namespace nn { namespace manu { namespace command {

namespace
{
    nn::os::ProcessId processId  = os::ProcessId::GetInvalidId();

    bool WaitProcessExit(nn::os::ProcessId& id, nn::TimeSpan timeout)
    {
        bool isExit = false;
        os::SystemEvent shellEvent;
        ns::GetShellEvent(&shellEvent);

        while(!isExit)
        {
            auto isTimedOut = !shellEvent.TimedWait(timeout);
            if(isTimedOut)
            {
                return isTimedOut;
            }

            ns::ShellEventInfo sei;
            ns::GetShellEventInfo(&sei);

            if( (sei.event == ns::ShellEvent_Exit || sei.event == ns::ShellEvent_Exception) && sei.processId == id)
            {
                NN_MANU_INFO_LOG("Exit process (PID : 0x%08x%08x)\n", static_cast<uint32_t>((sei.processId.value)>>32), static_cast<uint32_t>(sei.processId.value));
                nn::ns::TerminateProcess(sei.processId);
                id = os::ProcessId::GetInvalidId();
                isExit = true;
            }
        }
        return isExit;
    }

    nn::Result ProcessLaunchApplicationCommand(const void* body, size_t bodySize)
    {
        auto programId = *static_cast<const nn::ncm::ProgramId*>(body);
        const char* arguments = reinterpret_cast<const char*>(body) + sizeof(nn::Bit64);
        const int argumentSize = bodySize - sizeof(nn::Bit64);

        NN_MANU_INFO_LOG("Program ID : 0x%08x%08x\n", static_cast<uint32_t>((programId.value)>>32), static_cast<uint32_t>(programId.value));
        NN_MANU_INFO_LOG("Arguments  : %s\n", arguments);
        NN_MANU_INFO_LOG("Args Size  : %d\n", argumentSize);

        if(processId != os::ProcessId::GetInvalidId())
        {
            NN_MANU_INFO_LOG("Terminate last launch process (PID : 0x%08x%08x)\n", static_cast<uint32_t>((processId.value)>>32), static_cast<uint32_t>(processId.value));
            nn::ns::TerminateProcess(processId);
        }

        auto result = nn::ldr::SetProgramArgument(programId, arguments, argumentSize);
        if( result <= ResultArgumentCountOverflow() )
        {
            result = nn::ldr::FlushArguments();
            if( result.IsSuccess() )
            {
                result = nn::ldr::SetProgramArgument(programId, arguments, argumentSize);
            }
        }
        if( result.IsFailure() )
        {
            return result;
        }

        const int32_t flags = ns::LaunchProgramFlags_NotifyExit;
        nn::ns::ProgramLaunchProperty launchProperty = { programId, 0, nn::ncm::StorageId::BuildInSystem };
        result = nn::ns::LaunchProgram(&processId, launchProperty, flags);

        // 直前の引数を残さないように argv[0] に既定値の空文字列を上書きしておく
        nn::ldr::SetProgramArgument(programId, "", 1);
        if( result.IsFailure() )
        {
            if(result <= ResultProgramNotFound())
            {
                NN_MANU_ERROR_LOG("Program not found\n");
            }
            else
            {
                NN_MANU_ERROR_LOG("Failed to launch program (InnerValue : 0x%08X)", result.GetInnerValueForDebug());
            }
            return result;
        }
        WaitProcessExit(processId, nn::TimeSpan::FromDays(10));

        NN_RESULT_SUCCESS;
    }

    void ProcessShutdownCommand()
    {
        nn::ae::InvokeSystemAppletMain(
            nn::ae::AppletId_SystemAppletMenu,
            [](nn::ae::SystemAppletParameters*)
            {
                nn::ae::StartShutdownSequence();
            });
    }

    void ProcessRebootCommand()
    {
        nn::ae::InvokeSystemAppletMain(
            nn::ae::AppletId_SystemAppletMenu,
            [](nn::ae::SystemAppletParameters*)
            {
                nn::ae::StartRebootSequence();
            });
    }
}

nn::Result ProcessCommand(const CommandHeader& header, const void* body) NN_NOEXCEPT
{
    switch(header.Command)
    {
        case CommandId::Nop:
        {
            NN_MANU_INFO_LOG("[NOP]\n");
            break;
        }
        case CommandId::LaunchApplication:
        {
            NN_MANU_INFO_LOG("[LaunchApplication]\n");
            NN_RESULT_DO(ProcessLaunchApplicationCommand(body, header.BodySize));
            break;
        }
        case CommandId::Shutdown:
        {
            NN_MANU_INFO_LOG("[Shutdown]\n");
            ProcessShutdownCommand();
            break;
        }
        case CommandId::Reboot:
        {
            NN_MANU_INFO_LOG("[Reboot]\n");
            ProcessRebootCommand();
            break;
        }
        default :
        {
            NN_MANU_INFO_LOG("Unknown Command ID : 0x%08x\n", header.Command);
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

}}}
