﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_MemoryManagement.h>

#include <nn/ldr/ldr_ShellApi.h>
#include <nn/lr/lr_Service.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_InitializationApi.h>

#include <nn/htc/tenv/htc_TenvIServiceManager.h>
#include <nn/htc/tenv/htc_TenvServiceManager.h>

#include "../../../../Eris/Sources/Processes/scs/scs_Htcs.h"
#include "../../../../Eris/Sources/Processes/scs/scs_Infrastructure.h"
#include "../../../../Eris/Sources/Processes/scs/scs_Shell.h"
#include "../../../../Eris/Sources/Processes/scs/scs_ShellServer.h"
#include "../../../../Eris/Sources/Processes/scs/scs_TenvServiceManager.h"

#include "cs_CommandProcessor.h"
#include "cs_AudioServer.h"
#include "cs_HidServer.h"

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace
{
    const size_t StackSize = 4 * 1024;

    //!< cs プロセスが使用するソケット数
    const int SocketCountForAudio = 2;

    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_ShellServerStack1[StackSize];
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_ShellServerStack2[StackSize];

    nn::cs::CommandProcessor g_CommandProcessor;

    nn::scs::ShellServer g_ShellServer1;
    nn::scs::ShellServer g_ShellServer2;

    nn::sf::UnmanagedServiceObject<nn::htc::tenv::IServiceManager, nn::htc::tenv::ServiceManager> g_TenvServiceManager;

    template <typename Interface>
    void RegisterObjectForPort(nn::sf::SharedPointer<Interface> p, nn::scs::Port port, const char* pPortName)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::scs::GetServerManager()->RegisterObjectForPort(p, nn::scs::SessionCount[port], pPortName));
    }
}

extern "C" void nninitStartup() NN_NOEXCEPT
{
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
}

namespace
{
    const size_t ExpHeapSize = 32 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ExpHeapStack[ExpHeapSize];
    nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;
    struct StaticMutex
    {
        nn::os::MutexType mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex(&mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex(&mutex);
        }
    } g_ExpHeapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

    void InitializeExpHeap()
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        if( g_ExpHeapHandle == nullptr )
        {
            g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill);
            NN_ABORT_UNLESS(g_ExpHeapHandle);
        }
    }

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        return nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
    }
} // namespace

void * operator new ( std::size_t size )
{
    void* p = Allocate(size);
    return p;
}

void operator delete ( void * ptr ) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

extern "C" void nnMain() NN_NOEXCEPT
{
    InitializeExpHeap();
    nn::fs::SetAllocator(Allocate, Deallocate);

    nn::lr::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ldr::InitializeForShell() );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ns::InitializeForDevelop() );

    nn::ns::Initialize();

    nn::scs::InitializeHtcs(SocketCountForAudio);

    nn::cs::InitializeAudioServer();

    nn::cs::InitializeHidServer();

    nn::scs::InitializeShell();

    nn::scs::InitializeTenvServiceManager();

    g_CommandProcessor.Initialize();

    g_ShellServer1.Initialize("iywys@$cs", g_ShellServerStack1, sizeof(g_ShellServerStack1), &g_CommandProcessor);
    g_ShellServer1.Start();

    g_ShellServer2.Initialize("iywys@$csForRunnerTools", g_ShellServerStack2, sizeof(g_ShellServerStack2), &g_CommandProcessor);
    g_ShellServer2.Start();

    RegisterObjectForPort(g_TenvServiceManager.GetShared(), nn::scs::Port_HtcTenv, nn::htc::tenv::ServiceName);

    nn::scs::GetServerManager()->Start();
    nn::scs::StartServer();

    // TargetManager との接続イベント実装までの W/A (SIGLO-43364)
    while(nn::fs::MountHostRoot().IsFailure())
    {
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }
    nn::scs::NotifyTenvMountHostSuccess();

    for (;;)
    {
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1000) );
    }
}
