﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/web/detail/web_SessionApiDetail.h>
#include <nn/web/detail/web_SessionMessageTypesDetail.h>

namespace nn { namespace web { namespace detail {


//------------------------------------------------------------------------
SessionImpl::SessionImpl() NN_NOEXCEPT
: mMutex(false)
, mLibraryAppletHandle(nn::applet::InvalidLibraryAppletHandle)
, mPushCount()
, mPushStorageSize()
{
}

//------------------------------------------------------------------------
bool SessionImpl::trySend(const SessionMessageHeader& header, const void* body) NN_NOEXCEPT
{
    {
        std::lock_guard<decltype(mMutex)> lock(mMutex);

        // セッションが切れていたら何もせず返します。
        if (!isSessionAvaliable())
        {
            return false;
        }

        // 送信可能か確認します。
        auto sendStorageSize = sizeof(SessionMessageHeader) + header.size;
        if (!canSend(sendStorageSize))
        {
            return false;
        }

        // 送信します。
        sendImpl(header, body);
    }

    return true;
}

//------------------------------------------------------------------------
bool SessionImpl::tryRecive(SessionMessage* pOutMessage) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutMessage);

    std::lock_guard<decltype(mMutex)> lock(mMutex);

    // セッションが切れていたら何もせず返します。
    if (!isSessionAvaliable())
    {
        return false;
    }

    bool hasReceivedMessage = false;

    // メッセージ受信
    for (;!hasReceivedMessage;)
    {
        nn::applet::StorageHandle storage;

        // 受信すべきメッセージがあるか確認
        if (!nn::applet::TryPopFromInteractiveOutChannel(&storage, mLibraryAppletHandle))
        {
            return false;
        }

        // ACK かどうかのチェックします。
        SessionMessageHeader header;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storage, 0, &header, sizeof(SessionMessageHeader)));

        // ACK 処理
        if (header.kind == uint32_t(SessionMessageKindDetail::Ack))
        {
            NN_ABORT_UNLESS(mPushCount > 0);
            --mPushCount;

            SessionAck ack;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storage, 0, &ack, sizeof(SessionAck)));
            NN_ABORT_UNLESS((mPushStorageSize - ack.receivedSize) >= 0);
            mPushStorageSize -= ack.receivedSize;
        }
        else
        // Message なら受信処理
        if (header.kind < uint32_t(SessionMessageKind::SessionMessageKind_Max))
        {
            SessionMessageHeader header;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storage, 0, &header, sizeof(SessionMessageHeader)));

            // SessionMessageHeader + MessageSize 分のデータが Storage に入っているか確認します。
            auto realSize = nn::applet::GetStorageSize(storage);
            NN_ABORT_UNLESS((sizeof(SessionMessageHeader) + header.size) == realSize);

            // 読み込み
            {
                pOutMessage->header.kind = header.kind;
                pOutMessage->header.size = header.size;

                // bodyがあれば読み込みます。
                if (header.size > 0)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storage, sizeof(SessionMessageHeader), &pOutMessage->body, pOutMessage->header.size));
                }
                hasReceivedMessage = true;
            }

            // ACK 送信
            sendAck(realSize);
        }

        // Storage を解放します。
        nn::applet::ReleaseStorage(storage);
    }

    return hasReceivedMessage;
}

//------------------------------------------------------------------------
void SessionImpl::setLibraryAppletHandle(const nn::applet::LibraryAppletHandle& aHandle) NN_NOEXCEPT
{
    // Show***Page の呼び出しスレッドから呼び出されます。
    std::lock_guard<decltype(mMutex)> lock(mMutex);
    mLibraryAppletHandle = aHandle;
}

//------------------------------------------------------------------------
bool SessionImpl::canSend(size_t sendStorageSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(mMutex.IsLockedByCurrentThread());
    if (mPushCount == PushCountMax)
    {
        return false;
    }

    if ((mPushStorageSize + sendStorageSize) > PushStorageSizeMax)
    {
        return false;
    }
    return true;
}

//------------------------------------------------------------------------
bool SessionImpl::isSessionAvaliable() NN_NOEXCEPT
{
    // LibraryAppletHandle が有効かつ LA が終了していなければセッションは有効とします。
    NN_SDK_ASSERT(mMutex.IsLockedByCurrentThread());
    return isLibraryAppletHandleAvaliable() && !nn::os::TryWaitSystemEvent(nn::applet::GetLibraryAppletExitEvent(mLibraryAppletHandle));
}

//------------------------------------------------------------------------
bool SessionImpl::isLibraryAppletHandleAvaliable() NN_NOEXCEPT
{
    NN_SDK_ASSERT(mMutex.IsLockedByCurrentThread());
    return mLibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle;
}

//------------------------------------------------------------------------
void SessionImpl::sendImpl(const SessionMessageHeader& header, const void* body) NN_NOEXCEPT
{
    NN_SDK_ASSERT(mMutex.IsLockedByCurrentThread());

    auto sendStorageSize = sizeof(SessionMessageHeader) + header.size;
    NN_ABORT_UNLESS(canSend(sendStorageSize));

    nn::applet::StorageHandle storage;
    size_t storageSize = sizeof(SessionMessageHeader) + header.size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storage, storageSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storage, 0, &header, sizeof(SessionMessageHeader)));
    if (body != nullptr && header.size > 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storage, sizeof(SessionMessageHeader), body, header.size));
    }
    nn::applet::PushToInteractiveInChannel(mLibraryAppletHandle, storage);

    // Storage 数、量をカウントします。
    ++mPushCount;
    mPushStorageSize = mPushStorageSize + sizeof(SessionMessageHeader) + header.size;
}

//------------------------------------------------------------------------
void SessionImpl::sendAck(uint32_t receivedSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(mMutex.IsLockedByCurrentThread());

    auto ack = SessionAck::Create();
    ack.receivedSize = receivedSize;

    nn::applet::StorageHandle storage;
    size_t storageSize = sizeof(SessionAck);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storage, storageSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storage, 0, &ack, storageSize));
    nn::applet::PushToInteractiveInChannel(mLibraryAppletHandle, storage);
}

}}} // namespace nn::web::detail
