﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include <nn/rid/rid_SystemUpdateApi.h>

namespace nn { namespace rid {

    SystemUpdater::SystemUpdater() NN_NOEXCEPT : m_Mutex(false), m_IsCancelled(false)
    {
        m_Progress = { SystemUpdateProgress::State::DoNothing, 0, 0 };
    }

    Result SystemUpdater::Execute() NN_NOEXCEPT
    {
        Result result = ExecuteImpl();
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultSystemUpdateCancelled)
            {
                UpdateProgress(SystemUpdateProgress::State::Cancelled);
            }
            NN_RESULT_CATCH_ALL
            {
                UpdateProgress(SystemUpdateProgress::State::Failed);
            }
        NN_RESULT_END_TRY

        return result;
    }

    void SystemUpdater::Cancel() NN_NOEXCEPT
    {
        m_IsCancelled = true;
    }

    SystemUpdateProgress SystemUpdater::GetProgress() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        return m_Progress;
    }

    Result SystemUpdater::ExecuteImpl() NN_NOEXCEPT
    {
        m_IsCancelled = false;
        UpdateProgress(SystemUpdateProgress::State::DoNothing, 0, 0);

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            NN_RESULT_THROW(ResultNetworkNotConnected());
        }

        // 本体更新情報の取得
        ns::LatestSystemUpdate latest;
        {
            ns::SystemUpdateControl control;
            NN_RESULT_DO(control.Occupy());

            UpdateProgress(SystemUpdateProgress::State::Checking);

            ns::AsyncLatestSystemUpdate asyncLatest;
            NN_RESULT_DO(control.RequestCheckLatestUpdate(&asyncLatest));

            while (!asyncLatest.TryWait())
            {
                if (m_IsCancelled)
                {
                    NN_RESULT_THROW(ResultSystemUpdateCancelled());
                }

                os::SleepThread(TimeSpan::FromSeconds(1));
            }

            NN_RESULT_DO(asyncLatest.Get(&latest));
        }

        // ローカルのバージョンが最新の場合
        if (latest == ns::LatestSystemUpdate::UpToDate)
        {
            UpdateProgress(SystemUpdateProgress::State::NeedNoUpdate);
            NN_RESULT_SUCCESS;
        }
        // 最新の本体更新をダウンロードする必要がある場合
        else if (latest == ns::LatestSystemUpdate::NeedsDownload)
        {
            {
                ns::SystemUpdateControl control;
                NN_RESULT_DO(control.Occupy());

                UpdateProgress(SystemUpdateProgress::State::Downloading, 0, 0);

                // 最新の本体更新をダウンロードする
                ns::AsyncResult asyncResult;
                NN_RESULT_DO(control.RequestDownloadLatestUpdate(&asyncResult));

                while (!asyncResult.TryWait())
                {
                    if (m_IsCancelled)
                    {
                        NN_RESULT_THROW(ResultSystemUpdateCancelled());
                    }

                    auto progress = control.GetDownloadProgress();

                    if (progress.total != 0)
                    {
                        UpdateProgress(SystemUpdateProgress::State::Downloading, progress.loaded, progress.total);
                    }

                    os::SleepThread(TimeSpan::FromSeconds(1));
                }

                NN_RESULT_DO(asyncResult.Get());

                auto progress = control.GetDownloadProgress();

                UpdateProgress(SystemUpdateProgress::State::Downloading, progress.loaded, progress.total);
            }
        }

        // 本体更新を適用する
        {
            ns::SystemUpdateControl control;
            NN_RESULT_DO(control.Occupy());

            if (m_IsCancelled)
            {
                NN_RESULT_THROW(ResultSystemUpdateCancelled());
            }

            UpdateProgress(SystemUpdateProgress::State::Applying);
            control.ApplyDownloadedUpdate();
        }

        UpdateProgress(SystemUpdateProgress::State::Completed);

        NN_RESULT_SUCCESS;
    }

    void SystemUpdater::UpdateProgress(SystemUpdateProgress::State state, int64_t loaded, int64_t total) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        m_Progress.state = state;
        m_Progress.loaded = loaded;
        m_Progress.total = total;
    }
}}
