﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <vector>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include <nn/rid/rid_ApplicationUpdateApi.h>
#include "detail/rid_NetworkConnection.h"
#include "detail/rid_Settings.h"

namespace nn { namespace rid {

    namespace {
        const int MaxApplicationCount = 256;
        const int MaxListContentCount = 128;
        static ncm::StorageContentMetaKey g_KeyList[MaxListContentCount];

        // AppicationId と追加コンテンツのインデックスから ContentMetaKey(version = 0) を生成する
        ncm::ContentMetaKey MakeAocContentMetaKeyVer0(ncm::ApplicationId id, int aocIndex) NN_NOEXCEPT
        {
            return ncm::ContentMetaKey::Make(id.value + 0x1000 + aocIndex, 0, ncm::ContentMetaType::AddOnContent);
        }

        // AppicationId と追加コンテンツのインデックスからインストール済みの ContentMetaKey を取得する
        Result GetInstalledAocContentMetaKey(util::optional<ncm::ContentMetaKey>* outValue, ncm::ApplicationId id, int aocIndex) NN_NOEXCEPT
        {
            int keyCount;
            int offset = 0;

            while (NN_STATIC_CONDITION(true))
            {
                NN_RESULT_DO(ns::ListApplicationRecordInstalledContentMeta(&keyCount, g_KeyList, MaxListContentCount, id, offset));
                if (keyCount <= 0)
                {
                    break;
                }

                for (int i = 0; i < keyCount; i++)
                {
                    if (g_KeyList[i].key.type == ncm::ContentMetaType::AddOnContent && g_KeyList[i].key.id == MakeAocContentMetaKeyVer0(id, aocIndex).id)
                    {
                        outValue->emplace();
                        *outValue = g_KeyList[i].key;

                        NN_RESULT_SUCCESS;
                    }
                }

                offset += MaxListContentCount;
            }

            NN_RESULT_SUCCESS;
        }

        bool Exists(ncm::ApplicationId id, ApplicationAsset assetList[], int count) NN_NOEXCEPT
        {
            for (int i = 0; i < count; i++)
            {
                if (assetList[i].demoApplicationId == id)
                {
                    return true;
                }
            }
            return false;
        }

        bool Exists(ncm::AddOnContentId aocId, ApplicationAsset assetList[], int count) NN_NOEXCEPT
        {
            for (int i = 0; i < count; i++)
            {
                if (MakeAocContentMetaKeyVer0(assetList[i].aocApplicationId, assetList[i].index).id == aocId.value)
                {
                    return true;
                }
            }
            return false;
        }

        Result RequestDownloadAddOnContent(ncm::ApplicationId id, ncm::ContentMetaKey keyList[], int count) NN_NOEXCEPT
        {
            NN_RESULT_TRY(ns::CancelApplicationDownload(id))
                NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound) {}
            NN_RESULT_END_TRY

            ns::AsyncResult async;
            NN_RESULT_DO(ns::RequestDownloadAddOnContent(&async, id, keyList, count, ncm::StorageId::BuiltInUser));
            NN_RESULT_DO(async.Get());

            NN_RESULT_SUCCESS;
        }

        Result RequestDownloadApplication(ncm::ApplicationId id) NN_NOEXCEPT
        {
            NN_RESULT_TRY(ns::CancelApplicationDownload(id))
                NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound) {}
            NN_RESULT_END_TRY

            ns::AsyncResult async;
            NN_RESULT_DO(ns::RequestDownloadApplication(&async, id));
            NN_RESULT_DO(async.Get());

            NN_RESULT_SUCCESS;
        }

        Result HasNoRightsApplication(bool* outValue) NN_NOEXCEPT
        {
            std::unique_ptr<ns::ApplicationRecord> records(new ns::ApplicationRecord[MaxApplicationCount]);
            int listedCount = ns::ListApplicationRecord(records.get(), MaxApplicationCount, 0);

            for (int i = 0; i < listedCount; i++)
            {
                ns::ApplicationView view;
                ncm::ApplicationId id = (records.get() + i)->id;
                NN_RESULT_DO(ns::GetApplicationView(&view, &id, 1));

                if (view.IsLaunchable()                                    // Launchable である
                    && id != GetMenuApplicationId()                        // 試遊台メニューではない
                    && ns::CheckApplicationLaunchRights(id).IsFailure())   // 権利チェックが失敗
                {
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
            }

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        Result HasNoRightsAddOnContent(bool* outValue) NN_NOEXCEPT
        {
            ns::AsyncVerifyAddOnContentsResult async;
            NN_RESULT_DO(ns::RequestVerifyAddOnContentsRights(&async, GetMenuApplicationId()));

            if (async.Get().IsFailure())
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        Result HasNoRightsContent(bool* outValue) NN_NOEXCEPT
        {
            NN_RESULT_DO(HasNoRightsApplication(outValue));

            if (!*outValue)
            {
                NN_RESULT_DO(HasNoRightsAddOnContent(outValue));
            }

            NN_RESULT_SUCCESS;
        }

        Result SyncTicket() NN_NOEXCEPT
        {
            ec::system::AsyncProgressResult async;
            NN_RESULT_DO(ec::system::RequestSyncTicket(&async));
            NN_RESULT_DO(async.Get());

            NN_RESULT_SUCCESS;
        }

        Result IsEnoughFreeSpace(bool* outValue, int64_t requiredSize) NN_NOEXCEPT
        {
            int64_t freeSize;
            NN_RESULT_DO(ns::GetFreeSpaceSize(&freeSize, ncm::StorageId::BuildInUser));
            *outValue = freeSize >= requiredSize;

            NN_RESULT_SUCCESS;
        }
    }

    ApplicationUpdater::ApplicationUpdater() NN_NOEXCEPT : m_Mutex(false), m_IsCancelled(false), m_Current(0), m_Total(0)
    {
        m_Progress = { ApplicationUpdateProgress::State::DoNothing, 0, 0 };
    };

    Result ApplicationUpdater::Execute(int* outResultListCount, DownloadResult outResultList[], int resultListCount, ApplicationAsset assetList[], int assetListCount, bool deletesUnlistedContents, int64_t requiredFreeSpaceSize) NN_NOEXCEPT
    {
        Result result = ExecuteImpl(outResultListCount, outResultList, resultListCount, assetList, assetListCount, deletesUnlistedContents, requiredFreeSpaceSize);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultDownloadApplicationCancelled)
            {
                UpdateProgress(ApplicationUpdateProgress::State::Cancelled);
            }
            NN_RESULT_CATCH_ALL
            {
                UpdateProgress(ApplicationUpdateProgress::State::Failed);
            }
        NN_RESULT_END_TRY

        return result;
    }

    void ApplicationUpdater::Cancel() NN_NOEXCEPT
    {
        m_IsCancelled = true;
    }

    const ApplicationUpdateProgress ApplicationUpdater::GetProgress() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        return m_Progress;
    }

    bool ApplicationUpdater::IsCommitRequired() NN_NOEXCEPT
    {
        const int MaxListApplicationCount = 32;
        ns::ApplicationRecord recordList[MaxListApplicationCount];
        int applicationCount;
        int offset = 0;

        while (NN_STATIC_CONDITION(true))
        {
            applicationCount = ns::ListApplicationRecord(recordList, MaxListApplicationCount, offset);
            if (applicationCount <= 0)
            {
                break;
            }

            for (int i = 0; i < applicationCount; i++)
            {
                ns::ApplicationView view;
                NN_ABORT_UNLESS_RESULT_SUCCESS(ns::GetApplicationView(&view, &recordList[i].id, 1));

                if (view.IsWaitingCommit())
                {
                    return true;
                }
            }

            offset += applicationCount;
        }

        return false;
    }

    Result ApplicationUpdater::ExecuteImpl(int* outResultListCount, DownloadResult outResultList[], int resultListCount, ApplicationAsset assetList[], int count, bool deletesUnlistedContents, int64_t requiredFreeSpaceSize) NN_NOEXCEPT
    {
        ncm::ApplicationId menuApplicationId = GetMenuApplicationId();

        bool hasFailedContent = false;
        *outResultListCount = 0;
        m_IsCancelled = false;
        m_Current = 0;
        m_Total = count * 2;

        if (deletesUnlistedContents)
        {
            // リストにない Asset の削除
            {
                std::vector<ncm::ContentMetaKey> deleteList;
                int keyCount;
                int offset = 0;

                while (NN_STATIC_CONDITION(true))
                {
                    NN_RESULT_DO(ns::ListApplicationRecordInstalledContentMeta(&keyCount, g_KeyList, MaxListContentCount, menuApplicationId, offset));
                    if (keyCount <= 0)
                    {
                        break;
                    }

                    for (int i = 0; i < keyCount; i++)
                    {
                        ncm::AddOnContentId aocId = { g_KeyList[i].key.id };

                        if (g_KeyList[i].key.type == ncm::ContentMetaType::AddOnContent && !Exists(aocId, assetList, count))
                        {
                            deleteList.push_back(g_KeyList[i].key);
                        }
                    }

                    offset += MaxListContentCount;
                }

                for (int i = 0; i < deleteList.size(); i++)
                {
                    NN_RESULT_DO(ns::DeleteApplicationContentEntity(deleteList[i]));
                }
            }

            // リストにないアプリケーションの削除
            {
                const int MaxListApplicationCount = 32;
                ns::ApplicationRecord recordList[MaxListApplicationCount];
                std::vector<ncm::ApplicationId> deleteList;

                int applicationCount;
                int offset = 0;

                while (NN_STATIC_CONDITION(true))
                {
                    applicationCount = ns::ListApplicationRecord(recordList, MaxListApplicationCount, offset);
                    if (applicationCount <= 0)
                    {
                        break;
                    }

                    for (int i = 0; i < applicationCount; i++)
                    {
                        ncm::ApplicationId id = recordList[i].id;

                        if (id!= menuApplicationId &&       // 試遊台メニューではない
                            !Exists(id, assetList, count))  // リストに存在しない
                        {
                            ns::ApplicationView view;
                            NN_RESULT_DO(ns::GetApplicationView(&view, &id, 1));
                            if (!view.IsGameCard())         // ゲームカードではない
                            {
                                deleteList.push_back(id);
                            }
                        }
                    }

                    offset += applicationCount;
                }

                for (int i = 0; i < deleteList.size(); i++)
                {
                    NN_RESULT_DO(ns::DeleteApplicationCompletely(deleteList[i]));
                }
            }
        }

        NetworkConnection connection;
        if (!connection.IsAvailable())
        {
            NN_RESULT_THROW(ResultNetworkNotConnected());
        }

        {
            Result currentResult = ResultSuccess();
            NN_UTIL_SCOPE_EXIT
            {
                if (*outResultListCount < resultListCount)
                {
                    outResultList[*outResultListCount].applicationId = menuApplicationId;
                    outResultList[*outResultListCount].result = currentResult;
                    (*outResultListCount)++;
                }
            };

            // Asset のダウンロード
            for (int i = 0; i < count; i++, m_Current++)
            {
                if (m_IsCancelled)
                {
                    currentResult = ResultDownloadApplicationCancelled();
                    NN_RESULT_THROW(ResultDownloadApplicationCancelled());
                }

                connection.SubmitRequestAndWait();
                if (!connection.IsAvailable())
                {
                    currentResult = ResultNetworkNotConnected();
                    NN_RESULT_THROW(ResultNetworkNotConnected());
                }

                ncm::ContentMetaKey keyVer0 = MakeAocContentMetaKeyVer0(menuApplicationId, assetList[i].index);

                NN_RESULT_TRY(RequestDownloadAddOnContent(menuApplicationId, &keyVer0, 1))
                    NN_RESULT_CATCH(ns::ResultAlreadyUpToDate)
                    {
                        continue;
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        hasFailedContent = true;
                        currentResult = NN_RESULT_CURRENT_RESULT;
                        continue;
                    }
                NN_RESULT_END_TRY

                NN_RESULT_TRY(WaitDownload(menuApplicationId, &connection))
                    NN_RESULT_CATCH(ResultDownloadApplicationCancelled)
                    {
                        currentResult = NN_RESULT_CURRENT_RESULT;
                        NN_RESULT_RETHROW;
                    }
                    NN_RESULT_CATCH(ResultNotEnoughSpaceForDownloadApplication)
                    {
                        currentResult = NN_RESULT_CURRENT_RESULT;
                        NN_RESULT_RETHROW;
                    }
                    NN_RESULT_CATCH(ResultNetworkNotConnected)
                    {
                        currentResult = NN_RESULT_CURRENT_RESULT;
                        NN_RESULT_RETHROW;
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        hasFailedContent = true;
                        currentResult = NN_RESULT_CURRENT_RESULT;
                    }
                NN_RESULT_END_TRY

                bool isEnoughFreeSpace;
                NN_RESULT_DO(IsEnoughFreeSpace(&isEnoughFreeSpace, requiredFreeSpaceSize));
                if (!isEnoughFreeSpace)
                {
                    util::optional<ncm::ContentMetaKey> installedKey;
                    NN_RESULT_DO(GetInstalledAocContentMetaKey(&installedKey, menuApplicationId, assetList[i].index));
                    NN_SDK_ASSERT(installedKey);

                    NN_RESULT_DO(ns::DeleteApplicationContentEntity(*installedKey));
                    currentResult = ResultNotEnoughSpaceForDownloadApplication();
                    NN_RESULT_THROW(ResultNotEnoughSpaceForDownloadApplication());
                }
            }
        }

        // アプリケーションのダウンロード
        for (int i = 0; i < count; i++, m_Current++)
        {
            if (assetList[i].demoApplicationId == InvalidApplicationId)
            {
                continue;
            }

            Result currentResult = ResultSuccess();
            NN_UTIL_SCOPE_EXIT
            {
                if (*outResultListCount < resultListCount)
                {
                    outResultList[*outResultListCount].applicationId = assetList[i].demoApplicationId;
                    outResultList[*outResultListCount].result = currentResult;
                    (*outResultListCount)++;
                }
            };

            if (m_IsCancelled)
            {
                currentResult = ResultDownloadApplicationCancelled();
                NN_RESULT_THROW(ResultDownloadApplicationCancelled());
            }

            connection.SubmitRequestAndWait();
            if (!connection.IsAvailable())
            {
                currentResult = ResultNetworkNotConnected();
                NN_RESULT_THROW(ResultNetworkNotConnected());
            }

            NN_RESULT_TRY(RequestDownloadApplication(assetList[i].demoApplicationId))
                NN_RESULT_CATCH(ns::ResultAlreadyUpToDate)
                {
                    continue;
                }
                NN_RESULT_CATCH_ALL
                {
                    hasFailedContent = true;
                    currentResult = NN_RESULT_CURRENT_RESULT;
                    continue;
                }
            NN_RESULT_END_TRY

            NN_RESULT_TRY(WaitDownload(assetList[i].demoApplicationId, &connection))
                NN_RESULT_CATCH(ResultDownloadApplicationCancelled)
                {
                    currentResult = NN_RESULT_CURRENT_RESULT;
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH(ResultNotEnoughSpaceForDownloadApplication)
                {
                    currentResult = NN_RESULT_CURRENT_RESULT;
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH(ResultNetworkNotConnected)
                {
                    currentResult = NN_RESULT_CURRENT_RESULT;
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    hasFailedContent = true;
                    currentResult = NN_RESULT_CURRENT_RESULT;
                }
            NN_RESULT_END_TRY

            bool isEnoughFreeSpace;
            NN_RESULT_DO(IsEnoughFreeSpace(&isEnoughFreeSpace, requiredFreeSpaceSize));
            if (!isEnoughFreeSpace)
            {
                NN_RESULT_DO(ns::DeleteApplicationCompletely(assetList[i].demoApplicationId));
                currentResult = ResultNotEnoughSpaceForDownloadApplication();
                NN_RESULT_THROW(ResultNotEnoughSpaceForDownloadApplication());
            }
        }

        // 権利のないコンテンツがある場合にチケット同期する
        bool hasNoRightsContent;
        NN_RESULT_DO(HasNoRightsContent(&hasNoRightsContent));
        if (hasNoRightsContent)
        {
            NN_RESULT_DO(SyncTicket());
        }

        if (hasFailedContent)
        {
            NN_RESULT_THROW(ResultDownloadApplicationFailed());
        }

        // チケット同期はサーバにある権利を同期するだけなので、期待しているチケットがダウンロードされているとは限らない
        // そのため、再度権利の確認を行う
        NN_RESULT_DO(HasNoRightsContent(&hasNoRightsContent));
        if (hasNoRightsContent)
        {
            NN_RESULT_THROW(ResultNoRightsContentExists());
        }

        UpdateProgress(ApplicationUpdateProgress::State::Completed, 1000, 1000);
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    void ApplicationUpdater::UpdateProgress(ApplicationUpdateProgress::State state, int64_t done, int64_t total) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        if (state == ApplicationUpdateProgress::State::Downloading)
        {
            m_Progress.state = ApplicationUpdateProgress::State::Downloading;

            if (m_Progress.done == 0 && m_Progress.total == 0)
            {
                m_Progress.done = done;
                m_Progress.total = total;
            }

            double progressRate = (static_cast<double>(done) / total);
            double lastProgressRate = (static_cast<double>(m_Progress.done) / m_Progress.total);

            if (progressRate > lastProgressRate)
            {
                m_Progress.done = done;
                m_Progress.total = total;
            }
        }
        else
        {
            m_Progress.state = state;
            m_Progress.done = done;
            m_Progress.total = total;
        }
    }

    Result ApplicationUpdater::WaitDownload(ncm::ApplicationId id, NetworkConnection* connection) NN_NOEXCEPT
    {
        ns::ApplicationView view;
        NN_RESULT_DO(ns::GetApplicationView(&view, &id, 1));

        int unavailableCount = 0;
        while (view.IsDownloading())
        {
            if (m_IsCancelled)
            {
                NN_RESULT_DO(ns::CancelApplicationDownload(id));
                NN_RESULT_THROW(ResultDownloadApplicationCancelled());
            }

            NN_RESULT_DO(ns::GetApplicationView(&view, &id, 1));

            if (view.progress.state == ns::ApplicationDownloadState::Fatal || view.progress.state == ns::ApplicationDownloadState::Suspended)
            {
                NN_RESULT_DO(ns::CancelApplicationDownload(id));
                NN_RESULT_THROW(view.progress.lastResult);
            }
            else if (view.progress.state == ns::ApplicationDownloadState::NotEnoughSpace)
            {
                NN_RESULT_THROW(ResultNotEnoughSpaceForDownloadApplication());
            }

            if (view.progress.total != 0)
            {
                UpdateProgress(ApplicationUpdateProgress::State::Downloading, 100 * m_Current + 100 * view.progress.downloaded / view.progress.total, 100 * m_Total);
            }

            if (!connection->IsAvailable())
            {
                if (unavailableCount > 60)
                {
                    NN_RESULT_DO(ns::CancelApplicationDownload(id));
                    NN_RESULT_THROW(ResultNetworkNotConnected());
                }
                unavailableCount++;

                connection->SubmitRequestAndWait();
            }
            else
            {
                unavailableCount = 0;
            }

            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        NN_RESULT_SUCCESS;
    }
}}
