﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/repair/repair_Sdcard.h>

namespace nn {
    namespace repair {
        nn::Result Sdcard::WaitAttach() NN_NOEXCEPT
        {
            NN_RESULT_DO(this->WaitAttachDetach(true));
            NN_RESULT_SUCCESS;
        }

        nn::Result Sdcard::WaitDetach() NN_NOEXCEPT
        {
            NN_RESULT_DO(this->WaitAttachDetach(false));
            NN_RESULT_SUCCESS;
        }

        bool Sdcard::IsInserted() NN_NOEXCEPT
        {
            return nn::fs::IsSdCardInserted();
        }

        nn::Result Sdcard::WaitAttachDetach(bool isWaitAttach) NN_NOEXCEPT
        {
            if (this->IsInserted() == isWaitAttach)
            {
                NN_RESULT_SUCCESS;
            }

            std::unique_ptr<nn::fs::IEventNotifier> eventNotifier;
            NN_RESULT_DO(OpenSdCardDetectionEventNotifier(&eventNotifier));

            nn::os::SystemEventType event;
            NN_RESULT_DO(eventNotifier->BindEvent(&event, nn::os::EventClearMode_ManualClear));

            // マイクロ SD カードの挿抜の状態が LoopCountMax 回数分維持されたらループを抜ける
            const int HoldTime = 2000;   // 維持されることを期待する時間(ミリ秒)
            const int IntervalTime = 200; // 確認する間隔(ミリ秒)
            const int LoopCountMax = HoldTime / IntervalTime;   // 維持を期待するループ回数
            int loopCount = LoopCountMax;
            while (loopCount)
            {
                if (this->IsInserted() == isWaitAttach)
                {
                    --loopCount;
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(IntervalTime));
                }
                else
                {
                    loopCount = LoopCountMax;
                    nn::os::WaitSystemEvent(&event);
                    nn::os::ClearSystemEvent(&event);
                }
            }

            NN_RESULT_SUCCESS;
        }
    }
}
