﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/repair/repair_IFile.h>
#include <nn/repair/repair_FileSystem.h>

#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/manu/manu_Api.h>

#include <nn/nn_SdkLog.h>

#include <memory>

namespace nn
{
    namespace repair
    {
        class ManuFile : public IFile
        {
        public:
            ManuFile() : m_Path(nullptr) {}
            virtual ~ManuFile() {}

            virtual nn::Result Open(const char *path);
            virtual void Close() override;
            virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, bool flush) override;
            virtual nn::Result Read(size_t* pOut, int64_t offset, void* buffer, size_t size) override;
            virtual nn::Result GetSize(int64_t* pOut) override;
            virtual nn::Result Flush() override;
            virtual bool IsValid();

        private:
            const char *m_Path;
        };

        nn::Result ManuFile::Open(const char *path)
        {
            m_Path = path;

            NN_RESULT_SUCCESS;
        }

        void ManuFile::Close()
        {
            m_Path = nullptr;
        }

        nn::Result ManuFile::Write(int64_t offset, const void* buffer, size_t size, bool)
        {
            NN_RESULT_DO(
                nn::manu::WriteToHost(buffer, static_cast<size_t>(size), m_Path, offset, size));

            NN_RESULT_SUCCESS;
        }

        nn::Result ManuFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
        {
            NN_RESULT_DO(
                nn::manu::ReadFromHost(buffer, size, m_Path, offset, size));
            *pOut = size;

            NN_RESULT_SUCCESS;
        }

        nn::Result ManuFile::GetSize(int64_t* pOut)
        {
            size_t outSize;
            NN_RESULT_DO(
                nn::manu::GetFileSize(&outSize, m_Path));

            *pOut = static_cast<int64_t>(outSize);

            NN_RESULT_SUCCESS;
        }

        nn::Result ManuFile::Flush()
        {
            NN_RESULT_SUCCESS;
        }


        bool ManuFile::IsValid()
        {
            return m_Path != nullptr;
        }

        class ManuFileSystem : public FileSystem
        {
        public:
            ManuFileSystem()
            {
                nn::manu::InitializeUfio();
            }

            virtual ~ManuFileSystem()
            {
                nn::manu::FinalizeUfio();
            }

            virtual nn::Result DeleteFile(const char * path) override
            {
                NN_RESULT_THROW(nn::fs::ResultNotImplemented());
            }

            virtual nn::Result CreateFile(const char * path) override
            {
                uint8_t buffer[8];
                NN_RESULT_DO(
                    nn::manu::WriteToHost(buffer, 0, path, 0, 0));

                NN_RESULT_SUCCESS;
            }

            virtual nn::Result OpenFile(std::shared_ptr<IFile> *pOut, const char * path, int mode) override
            {
                auto file = new ManuFile();

                auto result = file->Open(path);
                if (result.IsFailure())
                {
                    delete file;
                    NN_RESULT_THROW(result);
                }

                *pOut = std::shared_ptr<IFile>(file);

                NN_RESULT_SUCCESS;
            }

            virtual nn::Result GetFileSize(int64_t *pOut, const char * path) override
            {
                size_t outSize;
                NN_RESULT_DO(
                    nn::manu::GetFileSize(&outSize, path));

                *pOut = static_cast<int64_t>(outSize);

                NN_RESULT_SUCCESS;
            }

            virtual nn::Result Exists(bool * pOut, const char * path) override
            {
                size_t outSize;
                nn::Result result = nn::manu::GetFileSize(&outSize, path);
                *pOut = result.IsSuccess();

                NN_RESULT_SUCCESS;
            }

        };

        FileSystem* CreateManuFileSystem()
        {
            return new ManuFileSystem();
        }
    }
}
