﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResSkeletalAnim.h>
#include <algorithm>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

void ResBoneAnim::Init(BoneAnimResult* pResult, const ResBone* bone) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    bit32 flag = ref().flag;
    const float* pBaseValue = ref().ofsBaseValueArray.to_ptr<float>();
    NW_G3D_TABLE_FIELD float identityRT[] = { 0.0f, 0.0f, 0.0f, 1.0f };

    if (flag & BASE_SCALE)
    {
        pResult->scale.Set(ref().ofsBaseValueArray.to_ptr<float>());
        pBaseValue += sizeof(Vec3) / sizeof(float);
    }
    else
    {
        NW_G3D_TABLE_FIELD float identityS[] = { 1.0f, 1.0f, 1.0f };
        const float* pScale = bone != NULL ? bone->ref().scale.a : identityS;
        pResult->scale.Set(pScale);
    }

    if (flag & BASE_ROTATE)
    {
        pResult->quat.Set(pBaseValue);
        pBaseValue += sizeof(Quat) / sizeof(float);
    }
    else
    {
        const float *pRotate = bone != NULL ? bone->ref().rotate.quat.a : identityRT;
        pResult->quat.Set(pRotate);
    }

    if (flag & BASE_TRANSLATE)
    {
        pResult->translate.Set(pBaseValue);
    }
    else
    {
        const float *pTranslate = bone != NULL ? bone->ref().translate.a : identityRT;
        pResult->translate.Set(pTranslate);
    }

    pResult->flag = (flag & TRANSFORM_MASK);
}

void ResBoneAnim::Eval(BoneAnimResult* pResult, float frame) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame);
    }
}

void ResBoneAnim::Eval(BoneAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFrameCache, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame, &pFrameCache[idxCurve]);
    }

}

//--------------------------------------------------------------------------------------------------

BindResult ResSkeletalAnim::PreBind(const ResSkeleton* pSkeleton)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    ref().ofsBindSkeleton.set_ptr(pSkeleton->ptr());
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();

    BindResult result;
    for (int idxAnim = 0, numAnim = GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);

        const ResName* pName = pBoneAnim->ref().ofsName.GetResName();
        int idxTarget = pSkeleton->GetBoneIndex(pName);
        if (idxTarget >= 0)
        {
            pBindIndexArray[idxAnim] = static_cast<u16>(idxTarget);
            result |= BindResult::Bound();
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResSkeletalAnim::BindCheck(const ResSkeleton* pSkeleton) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxAnim = 0, numAnim = GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);

        const ResName* pName = pBoneAnim->ref().ofsName.GetResName();
        int idxTarget = pSkeleton->GetBoneIndex(pName);
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

bool ResSkeletalAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxAnim = 0, numAnim = GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        for (int idxCurve = 0, numCurve = pBoneAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = pBoneAnim->GetCurve(idxCurve);
            size_t size = curve->CalcBakedFloatSize();
            curve->BakeFloat(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResSkeletalAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxAnim = 0, numAnim = GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
        {
            ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
            for (int idxCurve = 0, numCurve = pBoneAnim->GetCurveCount(); idxCurve < numCurve; ++idxCurve)
            {
                ResAnimCurve* curve = pBoneAnim->GetCurve(idxCurve);
                bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
                if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_FLOAT)
                {
                    pBuffer = curve->ref().ofsKeyArray.to_ptr();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResSkeletalAnim::Reset()
{
    ref().ofsBindSkeleton.set_ptr(NULL);
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();

    BindResult result;
    for (int idxAnim = 0, numAnim = GetBoneAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
    }

    ResetCurve();
}


}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
