﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_ShapeObj.h>
#include <nw/g3d/math/g3d_Vector2.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/g3d_SkeletonObj.h>

namespace nw { namespace g3d {

void Sphere::Transform(const Sphere& sphere, const Mtx34& mtx)
{
    this->center.Mul(mtx, sphere.center);
    float scaleX = Mtx34::ExtractBaseScaleSq(mtx, 0);
    float scaleY = Mtx34::ExtractBaseScaleSq(mtx, 1);
    float scaleZ = Mtx34::ExtractBaseScaleSq(mtx, 2);
    float scale = Math::Sqrt(Math::Max(Math::Max(scaleX, scaleY), scaleZ));
    this->radius = sphere.radius * scale;
}

void Sphere::Merge(const Sphere& lhs, const Sphere& rhs)
{
    Vec3 diff;
    diff.Sub(rhs.center, lhs.center);
    float distSq = Vec3::LengthSq(diff);
    float radiusDiff = rhs.radius - lhs.radius;
    float epsilon = Math::Max(radiusDiff * radiusDiff, 0.0001f);
    if (distSq < epsilon)
    {
        // 一方が他方を包含する場合、もしくは中心が限りなく近い場合。
        *this = lhs.radius > rhs.radius ? lhs : rhs;
    }
    else
    {
        float dist = distSq * Math::RSqrt(distSq);
        float newRadius = 0.5f * (lhs.radius + rhs.radius + dist);
        this->center.Add(lhs.center, diff.Mul(diff, (newRadius - lhs.radius) / dist));
        this->radius = newRadius;
    }
}

void AABB::Set(const Vec3* pPointArray, int count)
{
    Vec3 minP, maxP;
    minP = maxP = pPointArray[0];
    for (int idxPoint = 0; idxPoint < count; ++idxPoint)
    {
        minP.x = Math::Min(minP.x, pPointArray[idxPoint].x);
        maxP.x = Math::Max(maxP.x, pPointArray[idxPoint].x);
        minP.y = Math::Min(minP.y, pPointArray[idxPoint].y);
        maxP.y = Math::Max(maxP.y, pPointArray[idxPoint].y);
        minP.z = Math::Min(minP.z, pPointArray[idxPoint].z);
        maxP.z = Math::Max(maxP.z, pPointArray[idxPoint].z);
    }
    min = minP;
    max = maxP;
}

void AABB::Transform(const Bounding& aabb, const Mtx34& mtx)
{
    Vec3 center, extent;

    center.Mul(mtx, aabb.center);
    // extent = abs( rotMtx ) * extent;
    Vec3 axis;
    axis.Set(Math::Abs(mtx.m00), Math::Abs(mtx.m10), Math::Abs(mtx.m20));
    extent.Mul(axis, aabb.extent.x);
    axis.Set(Math::Abs(mtx.m01), Math::Abs(mtx.m11), Math::Abs(mtx.m21));
    extent.Mad(axis, aabb.extent.y, extent);
    axis.Set(Math::Abs(mtx.m02), Math::Abs(mtx.m12), Math::Abs(mtx.m22));
    extent.Mad(axis, aabb.extent.z, extent);

    this->min.Sub(center, extent);
    this->max.Add(center, extent);
}

void AABB::Merge(const AABB& lhs, const AABB& rhs)
{
#if defined( __ghs__ )
    this->min.ps[0] = __PS_SEL(__PS_SUB(rhs.min.ps[0], lhs.min.ps[0]), lhs.min.ps[0], rhs.min.ps[0]);
    this->max.ps[0] = __PS_SEL(__PS_SUB(lhs.max.ps[0], rhs.max.ps[0]), lhs.max.ps[0], rhs.max.ps[0]);
#else
    this->min.x = Math::Min(lhs.min.x, rhs.min.x);
    this->max.x = Math::Max(lhs.max.x, rhs.max.x);
    this->min.y = Math::Min(lhs.min.y, rhs.min.y);
    this->max.y = Math::Max(lhs.max.y, rhs.max.y);
#endif
    this->min.z = Math::Min(lhs.min.z, rhs.min.z);
    this->max.z = Math::Max(lhs.max.z, rhs.max.z);
}

void Plane::Set(const Vec3& p0, const Vec3& p1, const Vec3& p2)
{
    // 法線方向から見て p0, p1, p2 の順に時計回り。
    Vec3 v0, v1, v2;
    v0.Sub(p2, p0);
    v1.Sub(p1, p0);
    normal.Normalize(v2.Cross(v0, v1));
    dist = -Vec3::Dot(normal, p0);
}

void ViewVolume::SetPerspective(
    float fovy, float aspect, float zNear, float zFar, const Mtx34& viewToWorld)
{
    float yNear = zNear * Math::Tan(fovy * 0.5f);
    float xNear = yNear * aspect;
    SetFrustum(yNear, -yNear, -xNear, xNear, zNear, zFar, viewToWorld);
}

void ViewVolume::SetPerspectiveOffset(
    float fovy, float aspect, float zNear, float zFar, const Mtx34& viewToWorld, const Vec2& offset)
{
    float yNear = zNear * Math::Tan(fovy * 0.5f);
    float xNear = yNear * aspect;
    float xOffset = xNear * offset.x * 2.0f;
    float yOffset = yNear * offset.y * 2.0f;

    SetFrustum(yNear + yOffset, -yNear + yOffset, -xNear + xOffset, xNear + xOffset,
        zNear, zFar, viewToWorld);
}

void ViewVolume::SetFrustum(
    float top, float bottom, float left, float right, float zNear, float zFar,
    const Mtx34& viewToWorld)
{
    float nearToFar = zFar * Math::Rcp(zNear);
    float topFar = top * nearToFar;
    float bottomFar = bottom * nearToFar;
    float leftFar = left * nearToFar;
    float rightFar = right * nearToFar;

    // 右手系
    Vec3 pt[] = {
        Vec3::Make(left, top, -zNear),          // pt[0] は near の左上。
        Vec3::Make(right, top, -zNear),         // pt[1] は near の右上。
        Vec3::Make(right, bottom, -zNear),      // pt[2] は near の右下。
        Vec3::Make(left, bottom, -zNear),       // pt[3] は near の左下。
        Vec3::Make(leftFar, topFar, -zFar),     // pt[4] は far の左上。
        Vec3::Make(rightFar, topFar, -zFar),    // pt[5] は far の右上。
        Vec3::Make(rightFar, bottomFar, -zFar), // pt[6] は far の右下。
        Vec3::Make(leftFar, bottomFar, -zFar),  // pt[7] は far の左下。
    };

    for (int i = 0; i < 8; ++i)
    {
        pt[i].Mul(viewToWorld, pt[i]);
    }
    aabb.Set(pt, 8);

    Vec3 eye;
    eye.Set(viewToWorld.m03, viewToWorld.m13, viewToWorld.m23);
    planes[0].Set(eye, pt[3], pt[0]); // left
    planes[1].Set(eye, pt[1], pt[2]); // right
    planes[2].Set(pt[0], pt[1], pt[2]); // near
    planes[3].Set(pt[4], pt[7], pt[6]); // far
    planes[4].Set(eye, pt[0], pt[1]); // top
    planes[5].Set(eye, pt[2], pt[3]); // bottom
    numPlanes = 6;
    flag = 0;
}

void ViewVolume::SetOrtho(
    float top, float bottom, float left, float right, float zNear, float zFar,
    const Mtx34& viewToWorld)
{
    // 右手系
    Vec3 pt[] = {
        Vec3::Make(left, top, -zNear),      // pt[0] は near の左上。
        Vec3::Make(right, top, -zNear),     // pt[1] は near の右上。
        Vec3::Make(right, bottom, -zNear),  // pt[2] は near の右下。
        Vec3::Make(left, bottom, -zNear),   // pt[3] は near の左下。
        Vec3::Make(left, top, -zFar),       // pt[4] は far の左上。
        Vec3::Make(right, top, -zFar),      // pt[5] は far の右上。
        Vec3::Make(right, bottom, -zFar),   // pt[6] は far の右下。
        Vec3::Make(left, bottom, -zFar),    // pt[7] は far の左下。
    };

    for (int i = 0; i < 8; ++i)
    {
        pt[i].Mul(viewToWorld, pt[i]);
    }
    aabb.Set(pt, 8);

    planes[0].Set(pt[0], pt[7], pt[4]); // left
    planes[1].Set(pt[1], pt[5], pt[6]); // right
    planes[2].Set(pt[0], pt[1], pt[2]); // near
    planes[3].Set(pt[4], pt[7], pt[6]); // far
    planes[4].Set(pt[0], pt[4], pt[5]); // top
    planes[5].Set(pt[2], pt[6], pt[7]); // bottom
    numPlanes = 6;
    flag = 0;
}

bool ViewVolume::TestIntersection(const Sphere& sphere) const
{
    for (int idxPlane = 0; idxPlane < this->numPlanes; ++idxPlane)
    {
        const Plane& plane = this->planes[idxPlane];
        float dist = Vec3::Dot(plane.normal, sphere.center) + plane.dist;
        if (dist > sphere.radius)
        {
            return false;
        }
    }
    return true;
}

int ViewVolume::TestIntersectionEx(const Sphere& sphere) const
{
    int result = 1; // 内側
    for (int idxPlane = 0; idxPlane < this->numPlanes; ++idxPlane)
    {
        const Plane& plane = this->planes[idxPlane];
        float dist = Vec3::Dot(plane.normal, sphere.center) + plane.dist;
        if (dist > sphere.radius)
        {
            return -1; // 外側
        }
        if (dist >= -sphere.radius)
        {
            result = 0; // 交差
        }
    }
    return result;
}

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

bool ViewVolume::TestIntersection(const AABB& aabb) const
{
    if (this->flag)
    {
        if (this->aabb.min.x > aabb.max.x || aabb.min.x > this->aabb.max.x ||
            this->aabb.min.y > aabb.max.y || aabb.min.y > this->aabb.max.y ||
            this->aabb.min.z > aabb.max.z || aabb.min.z > this->aabb.max.z)
        {
            return false;
        }
    }
    for (int idxPlane = 0; idxPlane < this->numPlanes; ++idxPlane)
    {
        const Plane& plane = this->planes[idxPlane];
        Vec3 pos;
#if defined( __ghs__ )
        pos.ps[0] = __PS_SEL(plane.normal.ps[0], aabb.min.ps[0], aabb.max.ps[0]);
#else
        pos.x = Math::Select(plane.normal.x, aabb.min.x, aabb.max.x);
        pos.y = Math::Select(plane.normal.y, aabb.min.y, aabb.max.y);
#endif
        pos.z = Math::Select(plane.normal.z, aabb.min.z, aabb.max.z);

        if (Vec3::Dot(plane.normal, pos) + plane.dist > 0.0f)
        {
            return false;
        }
    }
    return true;
}

int ViewVolume::TestIntersectionEx(const AABB& aabb) const
{
    if (this->flag)
    {
        if (this->aabb.min.x > aabb.max.x || aabb.min.x > this->aabb.max.x ||
            this->aabb.min.y > aabb.max.y || aabb.min.y > this->aabb.max.y ||
            this->aabb.min.z > aabb.max.z || aabb.min.z > this->aabb.max.z)
        {
            return -1; // 外側
        }
    }
    int result = 1; // 内側
    for (int idxPlane = 0; idxPlane < this->numPlanes; ++idxPlane)
    {
        const Plane& plane = this->planes[idxPlane];
        Vec3 pos, neg;
#if defined( __ghs__ )
        pos.ps[0] = __PS_SEL(plane.normal.ps[0], aabb.min.ps[0], aabb.max.ps[0]);
        neg.ps[0] = __PS_SEL(plane.normal.ps[0], aabb.max.ps[0], aabb.min.ps[0]);
#else
        pos.x = Math::Select(plane.normal.x, aabb.min.x, aabb.max.x);
        neg.x = Math::Select(plane.normal.x, aabb.max.x, aabb.min.x);
        pos.y = Math::Select(plane.normal.y, aabb.min.y, aabb.max.y);
        neg.y = Math::Select(plane.normal.y, aabb.max.y, aabb.min.y);
#endif
        pos.z = Math::Select(plane.normal.z, aabb.min.z, aabb.max.z);
        neg.z = Math::Select(plane.normal.z, aabb.max.z, aabb.min.z);

        if (Vec3::Dot(plane.normal, pos) + plane.dist > 0.0f)
        {
            return -1; // 外側
        }
        if (result && Vec3::Dot(plane.normal, neg) + plane.dist >= 0.0f)
        {
            result = 0; // 交差
        }
    }
    return result;
}

NW_G3D_PRAGMA_POP_WARNINGS

int SubMeshRange::And(SubMeshRange* pDst, const SubMeshRange* pLHS, const SubMeshRange* pRHS)
{
    NW_G3D_ASSERT_NOT_NULL(pDst);
    NW_G3D_ASSERT_NOT_NULL(pLHS);
    NW_G3D_ASSERT_NOT_NULL(pRHS);

    const SubMeshRange* pLower = pLHS;
    const SubMeshRange* pUpper = pRHS;
    if (pLower->index > pUpper->index) // index が小さい方を pLower とします。
    {
        std::swap(pLower, pUpper);
    }

    int numRange = 0;
    while (pLower->count != 0 && pUpper->count != 0)
    {
        int end = pLower->index + pLower->count;
        if (end <= pUpper->index)
        {
            // pUpper の先頭が pLower の終端よりも後ろにあるので pLower を進めて再試行します。
            ++pLower;
            if (pLower->index > pUpper->index)
            {
                std::swap(pLower, pUpper);
            }
        }
        else if (end >= pUpper->index + pUpper->count)
        {
            // pUpper が pLower に包含されるので pUpper を進めて再試行します。
            *pDst = *pUpper;
            pDst->lodLevel = (std::max)(pLower->lodLevel, pUpper->lodLevel);
            ++pDst;
            ++pUpper;
            ++numRange;
        }
        else
        {
            // pLower と pUpper が交差しているので pLower を進めて再試行します。
            pDst->index = pUpper->index;
            pDst->count = static_cast<u16>(end - pUpper->index);
            pDst->lodLevel = (std::max)(pLower->lodLevel, pUpper->lodLevel);
            ++pDst;
            ++numRange;
            ++pLower;
            std::swap(pLower, pUpper); // 交差しているので pLower を進めると必ず逆転します。
        }
    }

    pDst->index = pDst->count = pDst->lodLevel = 0;
    return numRange;
}

//--------------------------------------------------------------------------------------------------

void ShapeObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERTMSG(arg.GetBufferingCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    ResShape* pRes = arg.GetResource();
    bool boundingEnabled = arg.IsBoundingEnabled();
    int totalSubMeshCount = 0;
    for (int index = 0; index < pRes->GetMeshCount(); ++index)
    {
        totalSubMeshCount += pRes->GetSubMeshCount(index);
    }
    // シェイプの AABB は計算しません。
    size_t subboundingSize = boundingEnabled ?
        Align(sizeof(AABB) * totalSubMeshCount, CACHE_BLOCK_SIZE) : 0;
    int numView = arg.GetViewCount();
    int numShpBlock = arg.IsViewDependent() ? numView : 1;
    int numKeyShape = pRes->GetKeyShapeCount();
    size_t userAreaAlignedSize = Align(arg.GetUserAreaSize());
    NW_G3D_ASSERTMSG(userAreaAlignedSize <= ShpBlock::USER_AREA_SIZE, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = subboundingSize;
    chunk[idx++].size = sizeof(GfxBuffer) * numShpBlock;
    chunk[idx++].size = sizeof(float) * numKeyShape;
    chunk[idx++].size = Align(numKeyShape, 32) >> 3;
    chunk[idx++].size = boundingEnabled ? sizeof(Sphere) * NUM_COORD * pRes->GetMeshCount() : 0;
    chunk[idx++].size = userAreaAlignedSize;
    NW_G3D_ASSERTMSG(idx == NUM_CHUNK, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    CalcOffset(chunk, NUM_CHUNK);
}

size_t ShapeObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool ShapeObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pBuffer, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResShape* pRes = arg.GetResource();
    int numView = arg.GetViewCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pRes = pRes;
    m_Flag = BLOCK_BUFFER_SWAP;
    m_NumView = static_cast<u8>(numView);
    m_ViewDependent = arg.IsViewDependent() ? 1 : 0;
    m_NumShpBlock = arg.IsViewDependent() ? m_NumView : 1;
    m_NumBuffering = static_cast<u8>(arg.GetBufferingCount());
    m_pShpBlockArray = sizer.GetPtr<GfxBuffer>(ptr, Sizer::SHAPE_BLOCK_ARRAY);
    m_pBlendWeightArray = sizer.GetPtr<float>(ptr, Sizer::BLEND_WEIGHT_ARRAY);
    m_pBlendWeightValidFlags = sizer.GetPtr<bit32>(ptr, Sizer::BLEND_WEIGHT_FLAGS);
    m_pBounding = sizer.GetPtr<Sphere>(ptr, Sizer::BOUNDING_ARRAY);
    m_pSubMeshBounding = sizer.GetPtr<AABB>(ptr, Sizer::SUBBOUNDING_ARRAY);
    m_pUserArea = sizer.GetPtr(ptr, Sizer::USER_AREA);
    m_UserAreaSize = Align(arg.GetUserAreaSize());
    m_pUserPtr = NULL;
    m_pBufferPtr = pBuffer;
    m_pBlockBuffer = NULL;

    if (m_pBounding)
    {
        for (int meshIndex = 0; meshIndex < m_pRes->GetMeshCount(); ++meshIndex)
        {
            Sphere& sphere = m_pBounding[meshIndex * NUM_COORD + LOCAL_COORD];
            const Bounding& bounding = m_pRes->GetBounding(meshIndex);
            sphere.center = bounding.center;
            sphere.radius = m_pRes->GetRadius(meshIndex);
            m_pBounding[meshIndex * NUM_COORD + WORLD_COORD] = sphere; // 安全のためローカルと同じもので初期化しておく。
        }
    }

    for (int idxShpBlock = 0; idxShpBlock < m_NumShpBlock; ++idxShpBlock)
    {
        new(&m_pShpBlockArray[idxShpBlock]) GfxBuffer();
    }

    ClearBlendWeights();

    return true;
}

size_t ShapeObj::CalcBlockBufferSize() const
{
    // 常に UniformBlock を作成する。
    return sizeof(ShpBlock) * GetShpBlockCount() * m_NumBuffering;
}

bool ShapeObj::SetupBlockBuffer(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pBuffer, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(pBuffer, BLOCK_BUFFER_ALIGNMENT, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG((m_Flag & BLOCK_BUFFER_VALID) == 0, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    size_t size = CalcBlockBufferSize();

    if (size > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBlockBuffer = pBuffer;

    ShpBlock* pShpBufferArray = static_cast<ShpBlock*>(pBuffer);
    Mtx34 worldMtx;
    worldMtx.Identity();
    s32 vtxSkinCount = GetVtxSkinCount();
    const int clearSize = ShpBlock::SYSTEM_AREA_SIZE + Align(m_UserAreaSize, CACHE_BLOCK_SIZE);
    for (int idxShpBlock = 0, numShpBlock = GetShpBlockCount();
        idxShpBlock < numShpBlock; ++idxShpBlock)
    {
        // RigidBody 以外は更新されないので初期化時に値を設定しておく。
        GfxBuffer& shpBlock = m_pShpBlockArray[idxShpBlock];
        // size は複数バッファ分のサイズなので 1 枚分のバッファサイズで SetData します。
        shpBlock.SetData(&pShpBufferArray[idxShpBlock * m_NumBuffering],
            sizeof(ShpBlock), m_NumBuffering);
        shpBlock.Setup();
        for (int idxBuffer = 0; idxBuffer < m_NumBuffering; ++idxBuffer)
        {
            ShpBlock* pShpBuffer = static_cast<ShpBlock*>(shpBlock.GetData(idxBuffer));
            if (CPUCache::IsValid(pShpBuffer, clearSize))
            {
                CPUCache::FillZero(pShpBuffer, clearSize);
            }
            if (IsBlockSwapEnabled())
            {
                Copy32<true>(&pShpBuffer->worldMtx, &worldMtx, sizeof(Mtx34) >> 2);
                Copy32<true>(&pShpBuffer->vtxSkinCount, &vtxSkinCount, sizeof(s32) >> 2);
            }
            else
            {
                Copy32<false>(&pShpBuffer->worldMtx, &worldMtx, sizeof(Mtx34) >> 2);
                Copy32<false>(&pShpBuffer->vtxSkinCount, &vtxSkinCount, sizeof(s32) >> 2);
            }
            shpBlock.DCFlush(idxBuffer);
        }
    }
    m_Flag |= BLOCK_BUFFER_VALID;

    return true;
}

void ShapeObj::CleanupBlockBuffer()
{
    NW_G3D_ASSERTMSG(m_Flag & BLOCK_BUFFER_VALID, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    for (int idxShpBlock = 0, numShpBlock = GetShpBlockCount();
        idxShpBlock < numShpBlock; ++idxShpBlock)
    {
        m_pShpBlockArray[idxShpBlock].Cleanup();
    }
    m_Flag &= ~BLOCK_BUFFER_VALID;
}

void ShapeObj::CalculateBounding(const SkeletonObj* pSkeleton, int meshIndex)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (m_pBounding == NULL)
    {
        return;
    }
    const Mtx34* pWorldMtxArray = pSkeleton->GetWorldMtxArray();
    Sphere& worldBounding = m_pBounding[meshIndex * NUM_COORD + WORLD_COORD];
    Sphere& localBounding = m_pBounding[meshIndex * NUM_COORD + LOCAL_COORD];
    if (IsRigidBody())
    {
        const Mtx34& worldMtx = pWorldMtxArray[GetBoneIndex()];
        worldBounding.Transform(localBounding, worldMtx);
    }
    else
    {
        const u16* pSkinBoneIndexArray = m_pRes->GetSkinBoneIndexArray();
        {
            int idxBone = pSkinBoneIndexArray[0];
            const Mtx34& worldMtx = pWorldMtxArray[idxBone];
            worldBounding.Transform(localBounding, worldMtx);
        }
        for (int idxIndex = 1, numIndex = m_pRes->GetSkinBoneIndexCount();
            idxIndex < numIndex; ++idxIndex)
        {
            Sphere sphere;
            int idxBone = pSkinBoneIndexArray[idxIndex];
            const Mtx34& worldMtx = pWorldMtxArray[idxBone];
            sphere.Transform(localBounding, worldMtx);
            worldBounding.Merge(worldBounding, sphere);
        }
    }
}

void ShapeObj::CalculateSubMeshBounding(const SkeletonObj* pSkeleton, int meshIndex)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (m_pSubMeshBounding == NULL || !IsRigidBody())
    {
        return;
    }

    int startBoundingIndex = 0;
    for (int index = 0; index < meshIndex; ++index)
    {
        startBoundingIndex += m_pRes->GetSubMeshCount(index);
    }

    const Mtx34* pWorldMtxArray = pSkeleton->GetWorldMtxArray();
    const Mtx34& worldMtx = pWorldMtxArray[GetBoneIndex()];
    const Bounding* pResSubMeshBoundingArray = m_pRes->ref().ofsSubMeshBoundingArray.to_ptr<Bounding>();
    Aabb* pSubMeshBounding = &m_pSubMeshBounding[startBoundingIndex];
    pResSubMeshBoundingArray = &pResSubMeshBoundingArray[startBoundingIndex + meshIndex];
#if NW_G3D_IS_HOST_CAFE
    size_t startAddress = Align(reinterpret_cast<size_t>(pSubMeshBounding), CACHE_BLOCK_SIZE);
    size_t endAddress = reinterpret_cast<size_t>(pSubMeshBounding + m_pRes->GetSubMeshCount(meshIndex)) & ~(CACHE_BLOCK_SIZE - 1);
    size_t bufferSize = endAddress - startAddress;
    DCZeroRange(reinterpret_cast<void*>(startAddress), bufferSize);
#endif
    for (int idxSubMesh = 0, numSubMesh = m_pRes->GetSubMeshCount(meshIndex);
        idxSubMesh < numSubMesh; ++idxSubMesh)
    {
        const Bounding& aabb = pResSubMeshBoundingArray[idxSubMesh];
        pSubMeshBounding[idxSubMesh].Transform(aabb, worldMtx);
    }
}

void ShapeObj::CalcShpBlock(int viewIndex, const Mtx34& worldMtx, int bufferIndex /*= 0*/)
{
    GfxBuffer& shpBlock = GetShpBlock(viewIndex);
    ShpBlock* pShpBuffer = static_cast<ShpBlock*>(shpBlock.GetData(bufferIndex));
    s32 vtxSkinCount = GetVtxSkinCount();
    const size_t updateSize = ShpBlock::SYSTEM_AREA_SIZE + Align(m_UserAreaSize, CACHE_BLOCK_SIZE);
    if (CPUCache::IsValid(pShpBuffer, updateSize))
    {
        // HDP 領域はキャッシュレスなので除外する必要がある。
        CPUCache::FillZero(pShpBuffer, updateSize);
    }
    if (IsBlockSwapEnabled())
    {
        Copy32<true>(&pShpBuffer->worldMtx, &worldMtx, sizeof(Mtx34) >> 2);
        Copy32<true>(&pShpBuffer->vtxSkinCount, &vtxSkinCount, sizeof(s32) >> 2);
        Copy32<true>(&pShpBuffer->userFloat, m_pUserArea, m_UserAreaSize >> 2);
    }
    else
    {
        Copy32<false>(&pShpBuffer->worldMtx, &worldMtx, sizeof(Mtx34) >> 2);
        Copy32<false>(&pShpBuffer->vtxSkinCount, &vtxSkinCount, sizeof(s32) >> 2);
        Copy32<false>(&pShpBuffer->userFloat, m_pUserArea, m_UserAreaSize >> 2);
    }
    shpBlock.DCFlush(bufferIndex);
}

void ShapeObj::ClearBlendWeights()
{
    int numKeyShape = GetResource()->GetKeyShapeCount();
    for (int idxKeyShape = 0; idxKeyShape < numKeyShape; ++idxKeyShape)
    {
        m_pBlendWeightArray[idxKeyShape] = 0.0f;
    }
    memset(m_pBlendWeightValidFlags, 0, Align(numKeyShape, 32) >> 3);
    m_Flag &= ~BLEND_WEIGHT_VALID;
}

bool ShapeObj::TestSubMeshIntersection(CullingContext* pCtx, const ViewVolume& volume, int meshIndex) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pCtx, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pSubMeshBounding, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    int startBoundingIndex = 0;
    for (int index = 0; index < meshIndex; ++index)
    {
        startBoundingIndex += m_pRes->GetSubMeshCount(index);
    }

    const Aabb* pSubMeshBounding = &m_pSubMeshBounding[startBoundingIndex];

    int subMeshCount = GetSubMeshCount(meshIndex);

    // 描画すべき最初のサブメッシュを探索
    for (pCtx->submeshIndex = pCtx->nodeIndex;
        pCtx->submeshIndex < subMeshCount; ++pCtx->submeshIndex)
    {
        if (volume.TestIntersection(pSubMeshBounding[pCtx->submeshIndex]))
        {
            break;
        }
    }

    // カリングされるサブメッシュが見つかるまで探索しながらサブメッシュ情報をマージ。
    for (pCtx->nodeIndex = pCtx->submeshIndex + 1; pCtx->nodeIndex < subMeshCount; ++pCtx->nodeIndex)
    {
        if (!volume.TestIntersection(pSubMeshBounding[pCtx->nodeIndex]))
        {
            break;
        }
    }
    pCtx->submeshCount = pCtx->nodeIndex - pCtx->submeshIndex;

    return pCtx->submeshIndex < subMeshCount;
}

bool ShapeObj::TestSubMeshLodIntersection(CullingContext* pCtx, const ViewVolume& volume,
        ICalcLodLevelFunctor& calcLodLevelFunctor) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pCtx, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pSubMeshBounding, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    int numSubMesh = GetSubMeshCount();
    if (pCtx->nodeIndex >= numSubMesh)
    {
        return false;
    }

    const AABB* pBounding = NULL;
    pCtx->submeshLodLevel = pCtx->nodeLodLevel;
    pCtx->submeshIndex = pCtx->nodeIndex;
    // 描画するべき最初のサブメッシュを探索。
    if (pCtx->nodeLodLevel == ICalcLodLevelFunctor::INVALID_LOD_LEVEL)
    {
        for (; pCtx->nodeIndex < numSubMesh; ++pCtx->nodeIndex)
        {
            pBounding = m_pSubMeshBounding + pCtx->nodeIndex;
            if (volume.TestIntersection(*pBounding))
            {
                break;
            }
        }
        if (pCtx->nodeIndex >= numSubMesh)
        {
            return false;
        }
        pCtx->submeshLodLevel = calcLodLevelFunctor(*pBounding, *this);
        NW_G3D_ASSERTMSG(pCtx->submeshLodLevel != ICalcLodLevelFunctor::INVALID_LOD_LEVEL, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        pCtx->submeshIndex = pCtx->nodeIndex;
    }

    // 同時に描画できるサブメッシュをマージ。
    pCtx->nodeLodLevel = ICalcLodLevelFunctor::INVALID_LOD_LEVEL;
    for (++pCtx->nodeIndex; pCtx->nodeIndex < numSubMesh; ++pCtx->nodeIndex)
    {
        pBounding = m_pSubMeshBounding + pCtx->nodeIndex;
        if (!volume.TestIntersection(*pBounding))
        {
            pCtx->nodeLodLevel = ICalcLodLevelFunctor::INVALID_LOD_LEVEL;
            pCtx->submeshCount = pCtx->nodeIndex++ - pCtx->submeshIndex;
            return true;
        }
        pCtx->nodeLodLevel = calcLodLevelFunctor(*pBounding, *this);
        NW_G3D_ASSERTMSG(pCtx->submeshLodLevel != ICalcLodLevelFunctor::INVALID_LOD_LEVEL, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        if (pCtx->submeshLodLevel != pCtx->nodeLodLevel)
        {
            break;
        }
    }

    pCtx->submeshCount = pCtx->nodeIndex - pCtx->submeshIndex;
    return true;
}

int ShapeObj::MakeSubMeshRange(SubMeshRange* pRangeArray, const ViewVolume& volume, int meshIndex) const
{
    CullingContext ctx;
    int idxRange = 0;
    while (TestSubMeshIntersection(&ctx, volume, meshIndex) )
    {
        pRangeArray->index = static_cast<u16>(ctx.submeshIndex);
        pRangeArray->count = static_cast<u16>(ctx.submeshCount);
        pRangeArray->lodLevel = 0;
        ++pRangeArray;
        ++idxRange;
    }
    pRangeArray->index = pRangeArray->count = pRangeArray->lodLevel = 0;
    return idxRange;
}

int ShapeObj::MakeSubMeshLodRange(SubMeshRange* pRangeArray,
                               const ViewVolume& volume, ICalcLodLevelFunctor& calcLodLevelFunctor) const
{
    CullingContext ctx;
    int idxRange = 0;
    while (TestSubMeshLodIntersection(&ctx, volume, calcLodLevelFunctor) )
    {
        pRangeArray->index = static_cast<u16>(ctx.submeshIndex);
        pRangeArray->count = static_cast<u16>(ctx.submeshCount);
        pRangeArray->lodLevel = static_cast<u16>(ctx.submeshLodLevel);
        ++pRangeArray;
        ++idxRange;
    }
    pRangeArray->index = pRangeArray->count = pRangeArray->lodLevel = 0;
    return idxRange;
}

}} // namespace nw::g3d
