﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_ModelObj.h>

namespace nw { namespace g3d {

void ModelObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERTMSG(arg.GetViewCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.GetSkeletonBufferingCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.GetShapeBufferingCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_ASSERTMSG(arg.GetMaterialBufferingCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    ResModel* pRes = arg.GetResource();

    // Skeleton のサイズ計算
    size_t sizeSkeleton = 0;
    ResSkeleton* pResSkeleton = pRes->GetSkeleton();
    SkeletonObj* pSkeleton = arg.GetSkeleton();
    if (pSkeleton)
    {
        // SkeletonObj を渡された場合はそれを使用する。
        NW_G3D_ASSERTMSG(pResSkeleton == pSkeleton->GetResource(), "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    }
    else
    {
        // SkeletonObj も構築する。
        SkeletonObj::InitArg argSkeleton(pResSkeleton);
        argSkeleton.BufferingCount(arg.GetSkeletonBufferingCount());
        size_t size = SkeletonObj::CalcBufferSize(argSkeleton);
        sizeSkeleton = Align(size, SkeletonObj::BUFFER_ALIGNMENT);
    }

    int numShape = pRes->GetShapeCount();
    int numMaterial = pRes->GetMaterialCount();
    int numBone = pResSkeleton->GetBoneCount();
    int numView = arg.GetViewCount();

    // Shape を辿ってサイズ計算
    size_t sizeShape = 0;
    for (int idxShape = 0; idxShape < numShape; ++idxShape)
    {
        ResShape* pResShape = pRes->GetShape(idxShape);
        ShapeObj::InitArg argShape(pResShape);
        argShape.BufferingCount(arg.GetShapeBufferingCount());
        ResBone* pResBone = pResSkeleton->GetBone(pResShape->GetBoneIndex());
        if (pResBone->GetBillboardMode() == ResBone::BILLBOARD_NONE)
        {
            argShape.ViewIndependent();
        }
        else
        {
            argShape.ViewDependent();
        }
        argShape.ViewCount(numView);
        if (arg.IsBoundingEnabled())
        {
            argShape.EnableBounding();
        }
        else
        {
            argShape.DisableBounding();
        }
        argShape.UserAreaSize(arg.GetShapeUserAreaSize());
        size_t size = ShapeObj::CalcBufferSize(argShape);
        sizeShape += Align(size, ShapeObj::BUFFER_ALIGNMENT);
    }

    // Material を辿ってサイズ計算
    size_t sizeMaterial = 0;
    for (int idxMaterial = 0; idxMaterial < numMaterial; ++idxMaterial)
    {
        ResMaterial* pResMaterial = pRes->GetMaterial(idxMaterial);
        MaterialObj::InitArg argMaterial(pResMaterial);
        argMaterial.BufferingCount(arg.GetMaterialBufferingCount());
        size_t size = MaterialObj::CalcBufferSize(argMaterial);
        sizeMaterial += Align(size, MaterialObj::BUFFER_ALIGNMENT);
    }

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = sizeSkeleton;
    chunk[idx++].size = sizeShape;
    chunk[idx++].size = sizeMaterial;
    chunk[idx++].size = sizeof(SkeletonObj);
    chunk[idx++].size = sizeof(ShapeObj) * numShape;
    chunk[idx++].size = sizeof(MaterialObj) * numMaterial;
    chunk[idx++].size = Align(numBone, 32) >> 3; // 1ビジビリティ1ビット
    chunk[idx++].size = Align(numMaterial, 32) >> 3; // 1ビジビリティ1ビット
    chunk[idx++].size = arg.IsBoundingEnabled() ? sizeof(Sphere) * arg.GetLodCount() : 0;
    NW_G3D_ASSERTMSG(idx == NUM_CHUNK, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    CalcOffset(chunk, NUM_CHUNK);
}

//--------------------------------------------------------------------------------------------------

size_t ModelObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool ModelObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pBuffer, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResModel* pRes = arg.GetResource();
    ResSkeleton* pResSkeleton = pRes->GetSkeleton();
    int numShape = pRes->GetShapeCount();
    int numMaterial = pRes->GetMaterialCount();
    int numView = arg.GetViewCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pRes = pRes;
    m_pBoneVisArray = sizer.GetPtr<bit32>(ptr, Sizer::BONE_VIS_ARRAY);
    m_pMatVisArray = sizer.GetPtr<bit32>(ptr, Sizer::MAT_VIS_ARRAY);
    m_NumView = static_cast<u8>(numView);
    m_Flag = 0;
    m_NumShape = static_cast<u16>(numShape);
    m_NumMaterial = static_cast<u16>(numMaterial);
    m_pSkeleton = NULL;
    m_pShapeArray = sizer.GetPtr<ShapeObj>(ptr, Sizer::SHAPE_ARRAY);
    m_pMaterialArray = sizer.GetPtr<MaterialObj>(ptr, Sizer::MATERIAL_ARRAY);
    m_pBounding = sizer.GetPtr<Sphere>(ptr, Sizer::BOUNDING);
    m_pUserPtr = NULL;
    m_pBufferPtr = pBuffer;
    m_pBlockBuffer = NULL;
    m_LodCount = arg.GetLodCount();

    // Skeleton を構築。
    SkeletonObj* pSkeleton = arg.GetSkeleton();
    if (pSkeleton)
    {
        m_pSkeleton = pSkeleton;
    }
    else
    {
        SkeletonObj::InitArg argSkeleton(pResSkeleton);
        argSkeleton.BufferingCount(arg.GetSkeletonBufferingCount());
        size_t size = SkeletonObj::CalcBufferSize(argSkeleton);

        m_pSkeleton = new(sizer.GetPtr(ptr, Sizer::SKELETON)) SkeletonObj();
        m_pSkeleton->Init(argSkeleton, sizer.GetPtr(ptr, Sizer::SKELETON_BUFFER), size);
    }

    // 配下の Shape と Material を構築。

    void* pShapeBuffer = sizer.GetPtr<void>(ptr, Sizer::SHAPE_BUFFER);
    int viewDependent = 0;
    for (int idxShape = 0; idxShape < numShape; ++idxShape)
    {
        ResShape* pResShape = pRes->GetShape(idxShape);

        ShapeObj::InitArg argShape(pResShape);
        argShape.BufferingCount(arg.GetShapeBufferingCount());
        argShape.ViewCount(numView);
        ResBone* pResBone = m_pSkeleton->GetBone(pResShape->GetBoneIndex());
        if (pResBone->GetBillboardMode() == ResBone::BILLBOARD_NONE)
        {
            argShape.ViewIndependent();
        }
        else
        {
            argShape.ViewDependent();
            viewDependent = 1;
        }
        if (arg.IsBoundingEnabled())
        {
            argShape.EnableBounding();
        }
        else
        {
            argShape.DisableBounding();
        }
        argShape.UserAreaSize(arg.GetShapeUserAreaSize());
        size_t size = ShapeObj::CalcBufferSize(argShape);
        size = Align(size, ShapeObj::BUFFER_ALIGNMENT);

        ShapeObj* pShape = new(&m_pShapeArray[idxShape]) ShapeObj();
        pShape->Init(argShape, pShapeBuffer, size);

        pShapeBuffer = AddOffset(pShapeBuffer, size);
    }
    m_ViewDependent = static_cast<u8>(viewDependent);

    void* pMaterialBuffer = sizer.GetPtr<void>(ptr, Sizer::MATERIAL_BUFFER);
    for (int idxMaterial = 0; idxMaterial < numMaterial; ++idxMaterial)
    {
        ResMaterial* pResMaterial = pRes->GetMaterial(idxMaterial);

        MaterialObj::InitArg argMaterial(pResMaterial);
        argMaterial.BufferingCount(arg.GetMaterialBufferingCount());
        size_t size = MaterialObj::CalcBufferSize(argMaterial);
        size = Align(size, MaterialObj::BUFFER_ALIGNMENT);

        MaterialObj* pMaterial = new(&m_pMaterialArray[idxMaterial]) MaterialObj();
        pMaterial->Init(argMaterial, pMaterialBuffer, size);

        pMaterialBuffer = AddOffset(pMaterialBuffer, size);
    }

    ClearBoneVisibility();
    ClearMatVisibility();

    if (m_pBounding)
    {
        memset(m_pBounding, 0, sizeof(*m_pBounding)); // 念のため初期化しておく。
    }

    return true;
}

size_t ModelObj::CalcBlockBufferSize()
{
    size_t size = 0;

    if (m_pSkeleton && !m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に構築されている可能性がある。
        size += Align(m_pSkeleton->CalcBlockBufferSize(), SkeletonObj::BLOCK_BUFFER_ALIGNMENT);
    }

    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (!shape.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size += Align(shape.CalcBlockBufferSize(), ShapeObj::BLOCK_BUFFER_ALIGNMENT);
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (!material.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size += Align(material.CalcBlockBufferSize(), MaterialObj::BLOCK_BUFFER_ALIGNMENT);
        }
    }

    return size;
}

bool ModelObj::SetupBlockBuffer(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERTMSG(bufferSize == 0 || pBuffer, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(pBuffer, BLOCK_BUFFER_ALIGNMENT, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG((m_Flag & BLOCK_BUFFER_VALID) == 0, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (CalcBlockBufferSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBlockBuffer = pBuffer;

    bool success = true;

    if (m_pSkeleton && !m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に構築されている可能性がある。
        size_t size = m_pSkeleton->CalcBlockBufferSize();
        size_t sizeAligned = Align(size, SkeletonObj::BLOCK_BUFFER_ALIGNMENT);
        success = m_pSkeleton->SetupBlockBuffer(pBuffer, size) && success;
        pBuffer = AddOffset(pBuffer, sizeAligned);
    }

    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (!shape.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size_t size = shape.CalcBlockBufferSize();
            size_t sizeAligned = Align(size, ShapeObj::BLOCK_BUFFER_ALIGNMENT);
            success = shape.SetupBlockBuffer(pBuffer, size) && success;
            pBuffer = AddOffset(pBuffer, sizeAligned);
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (!material.IsBlockBufferValid()) // 既に構築されている場合は何もしない。
        {
            size_t size = material.CalcBlockBufferSize();
            size_t sizeAligned = Align(size, MaterialObj::BLOCK_BUFFER_ALIGNMENT);
            success = material.SetupBlockBuffer(pBuffer, size) && success;
            pBuffer = AddOffset(pBuffer, sizeAligned);
        }
    }

    if (success)
    {
        m_Flag |= BLOCK_BUFFER_VALID;
    }

    return success;
}

void ModelObj::CleanupBlockBuffer()
{
    NW_G3D_ASSERTMSG(m_Flag & BLOCK_BUFFER_VALID, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (m_pSkeleton && m_pSkeleton->IsBlockBufferValid())
    {
        // 共有スケルトンは既に破棄されている可能性がある。
        m_pSkeleton->CleanupBlockBuffer();
    }

    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsBlockBufferValid()) // 既に解放されている場合は何もしない。
        {
            shape.CleanupBlockBuffer();
        }
    }

    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        if (material.IsBlockBufferValid()) // 既に解放されている場合は何もしない。
        {
            material.CleanupBlockBuffer();
        }
    }

    m_Flag ^= BLOCK_BUFFER_VALID;
}

void ModelObj::CalcWorld(const Mtx34& baseMtx)
{
    m_pSkeleton->CalcWorldMtx(baseMtx);
}

void ModelObj::CalculateBounding(int lodLevel)
{
    if (m_pBounding == NULL)
    {
        return;
    }
    Sphere& bounding = m_pBounding[lodLevel];
    if (GetShapeCount() >= 1)
    {
        ShapeObj& shape = m_pShapeArray[0];
        if (lodLevel < shape.GetMeshCount())
        {
            shape.CalculateBounding(m_pSkeleton, lodLevel);
            bounding = *shape.GetBounding(lodLevel);
        }
        // 該当のLODレベルを持たないなら、LODレベル0で代用
        else
        {
            shape.CalculateBounding(m_pSkeleton, 0);
            bounding = *shape.GetBounding(0);
        }
    }
    for (int idxShape = 1, numShape = GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (lodLevel < shape.GetMeshCount())
        {
            shape.CalculateBounding(m_pSkeleton, lodLevel);
            bounding.Merge(bounding, *shape.GetBounding(lodLevel));
        }
        // 該当のLODレベルを持たないなら、LODレベル0で代用
        else
        {
            shape.CalculateBounding(m_pSkeleton, 0);
            bounding.Merge(bounding, *shape.GetBounding(0));
        }
    }
}

void ModelObj::CalcMtxBlock(int bufferIndex /*= 0*/)
{
    m_pSkeleton->CalcMtxBlock(bufferIndex);
}

void ModelObj::CalcView(int viewIndex, const Mtx34& cameraMtx, int bufferIndex /*= 0*/)
{
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(viewIndex, m_NumView, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (!IsViewDependent())
    {
        return; // 将来ビューに依存する処理が増えた場合はこの処理はスキップできない可能性がある。
    }

    // 直近のビルボード行列をキャッシュする。
    // キャッシュはシェイプがボーンの階層構造順に並んでいることに依存する。
    int lastBoneIndex = -1;
    Mtx34 bbMtx;

    const Mtx34* pWorldMtxArray = m_pSkeleton->GetWorldMtxArray();
    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsRigidBody() && shape.IsViewDependent())
        {
            // ビューに依存しないシェイプは CalcShape で計算する。
            int idxBone = shape.GetBoneIndex();
            const ResBone* pBone = m_pSkeleton->GetBone(idxBone);

            bit32 bbMode = pBone->GetBillboardMode();
            NW_G3D_ASSERTMSG(bbMode <= ResBone::BILLBOARD_MAX, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
            if (bbMode == ResBone::BILLBOARD_NONE)
            {
                // ビルボードでない場合は通常のワールド行列を書き込む。
                shape.CalcShpBlock(viewIndex, pWorldMtxArray[idxBone], bufferIndex);
                continue;
            }

            int bbBoneIndex = pBone->ref().billboardIndex;
            if (bbBoneIndex == ResBone::BB_INDEX_NONE)
            {
                // 階層ビルボードではないのでワールド行列と乗算されたビルボード行列を取得する。
                if (idxBone != lastBoneIndex) // 非階層ビルボードは自分自身のインデックスを見る。
                {
                    // キャッシュされていない場合はビルボード行列を計算する。
                    m_pSkeleton->CalcBillboardMtx(&bbMtx, cameraMtx, idxBone, true);
                    lastBoneIndex = idxBone;
                }
                shape.CalcShpBlock(viewIndex, bbMtx, bufferIndex);
                continue;
            }

            if (bbBoneIndex != lastBoneIndex)
            {
                // キャッシュされていない場合はビルボード行列を計算する。
                m_pSkeleton->CalcBillboardMtx(&bbMtx, cameraMtx, bbBoneIndex, false);
                lastBoneIndex = bbBoneIndex;
            }

            // ビルボード行列をかけて回転させる。
            Mtx34 worldMtx;
            worldMtx.Mul(bbMtx, pWorldMtxArray[idxBone]);
            shape.CalcShpBlock(viewIndex, worldMtx, bufferIndex);
        }
    }
}

void ModelObj::CalcShape(int bufferIndex /*= 0*/)
{
    const Mtx34* pWorldMtxArray = m_pSkeleton->GetWorldMtxArray();
    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        ShapeObj& shape = m_pShapeArray[idxShape];
        if (shape.IsRigidBody() && !shape.IsViewDependent())
        {
            // ビューに依存するシェイプは CalcView で計算する。
            int idxBone = shape.GetBoneIndex();
            shape.CalcShpBlock(0, pWorldMtxArray[idxBone], bufferIndex);
        }
    }
}

void ModelObj::CalcMaterial(int bufferIndex /*= 0*/)
{
    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
        MaterialObj& material = m_pMaterialArray[idxMaterial];
        material.CalcMatBlock(bufferIndex);
    }
}

void ModelObj::EnableBlockSwapAll()
{
    if (m_pSkeleton)
    {
        m_pSkeleton->EnableBlockSwap();
    }

    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        m_pShapeArray[idxShape].EnableBlockSwap();
    }

    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
         m_pMaterialArray[idxMaterial].EnableBlockSwap();
    }
}

void ModelObj::DisableBlockSwapAll()
{
    if (m_pSkeleton)
    {
        m_pSkeleton->DisableBlockSwap();
    }

    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        m_pShapeArray[idxShape].DisableBlockSwap();
    }

    for (int idxMaterial = 0; idxMaterial < m_NumMaterial; ++idxMaterial)
    {
        m_pMaterialArray[idxMaterial].DisableBlockSwap();
    }
}

void ModelObj::UpdateViewDependency()
{
    int viewDependent = 0;
    for (int idxShape = 0; idxShape < m_NumShape; ++idxShape)
    {
        if (m_pShapeArray[idxShape].IsViewDependent())
        {
            viewDependent = 1;
        }
    }
    m_ViewDependent = static_cast<u8>(viewDependent);
}

void ModelObj::ClearBoneVisibility()
{
    for (int idxBone = 0, numBone = m_pSkeleton->GetBoneCount(); idxBone < numBone; ++idxBone)
    {
        ResBone* pBone = m_pSkeleton->GetBone(idxBone);
        bool visible = pBone->ref().flag & ResBone::VISIBILITY ? true : false;
        SetBoneVisibility(idxBone, visible);
    }
}

void ModelObj::ClearMatVisibility()
{
    for (int idxMat = 0, numMat = GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        ResMaterial* pResMaterial = GetMaterial(idxMat)->GetResource();
        bool visible = pResMaterial->ref().flag & ResMaterial::VISIBILITY ? true : false;
        SetMatVisibility(idxMat, visible);
    }
}

void ModelObj::SetTextureChangeCallback(MaterialObj::TextureChangeCallback pCallback)
{
    for (int idxMat = 0, numMat = GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        GetMaterial(idxMat)->SetTextureChangeCallback(pCallback);
    }
}

}} // namespace nw::g3d
