﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/fnd/g3d_GfxShader.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/ut/g3d_Flag.h>
#if defined(NW_G3D_IS_GL_ES)
#include <nw/g3d/fnd/g3d_EglUtility.h>
#endif

#if NW_G3D_IS_GL
using nw::g3d::fnd::detail::GL;
#endif

namespace nw { namespace g3d { namespace fnd {

void GfxShader::Setup(const SetupArg& arg)
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    NW_G3D_ASSERT((!arg.pComputeShader && arg.pVertexShader) ||
        (arg.pComputeShader && !arg.pVertexShader && !arg.pGeometryShader && !arg.pFragmentShader));
#else
    NW_G3D_ASSERT(arg.pVertexShader);
#endif
    pVertexShader = arg.pVertexShader;
    pGeometryShader = arg.pGeometryShader;
    pFragmentShader = arg.pFragmentShader;
#if NW_G3D_COMPUTE_SHADER_ENABLE
    pComputeShader = arg.pComputeShader;
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    handle = CreateShaderProgram();
#if NW_G3D_GFXSHADER_SETUPARG_CONSTRUCTOR
    if (arg.pBinary)
    {
        glProgramBinary(handle, arg.binaryFormat, arg.pBinary, arg.binarySize);
        NW_G3D_ASSERT(CheckProgramStatus(handle));
    }
    else
#endif
    {
        if(arg.useMultiSource)
        {
            if(arg.ppVertexShader)
            {
                AttachShader(handle, GL_VERTEX_SHADER, arg.ppVertexShader, arg.numVertexShader);
            }
            if(arg.ppGeometryShader)
            {
#if !defined(NW_G3D_IS_GL_ES)
                AttachShader(handle, GL_GEOMETRY_SHADER, arg.ppGeometryShader, arg.numGeometryShader);
#else
                NW_G3D_ASSERTMSG(false, "NW: geometry shader is not supported.\n");
#endif
            }
            if(arg.ppFragmentShader)
            {
                AttachShader(handle, GL_FRAGMENT_SHADER, arg.ppFragmentShader, arg.numFragmentShader);
            }
            if(arg.ppComputeShader)
            {
#if !defined(NW_G3D_IS_GL_ES)
                AttachShader(handle, GL_COMPUTE_SHADER, arg.ppComputeShader, arg.numComputeShader);
#else
                NW_G3D_ASSERTMSG(false, "NW: compute shader is not supported.\n");
#endif
            }
        }
        else
        {
            if(arg.pVertexShader)
            {
                AttachShader(handle, GL_VERTEX_SHADER, arg.pVertexShader);
            }
            if(arg.pGeometryShader)
            {
#if !defined(NW_G3D_IS_GL_ES)
                AttachShader(handle, GL_GEOMETRY_SHADER, arg.pGeometryShader);
#else
                NW_G3D_ASSERTMSG(false, "NW: geometry shader is not supported.\n");
#endif
            }
            if(arg.pFragmentShader)
            {
                AttachShader(handle, GL_FRAGMENT_SHADER, arg.pFragmentShader);
            }
            if(arg.pComputeShader)
            {
#if !defined(NW_G3D_IS_GL_ES)
                AttachShader(handle, GL_COMPUTE_SHADER, arg.pComputeShader);
#else
                NW_G3D_ASSERTMSG(false, "NW: compute shader is not supported.\n");
#endif
            }
        }
        if (arg.numVaryings > 0 && arg.ppTransformFeedbackVaryings)
        {
            glTransformFeedbackVaryings(
                handle, arg.numVaryings, arg.ppTransformFeedbackVaryings, GL_INTERLEAVED_ATTRIBS);
        }
        LinkShaderProgram(handle);
    }
#if NW_G3D_GL_PORTABILITY >= NW_G3D_GL_LEVEL2
    glUseProgram(handle);
#endif // NW_G3D_GL_PORTABILITY

    // sampler uniform に対して texture unit を指定してしまう。
    int numUniform = 0;
    glGetProgramiv(handle, GL_ACTIVE_UNIFORMS, &numUniform);
    if (numUniform > 0)
    {
        int nameBufSize = 0;
        glGetProgramiv(handle, GL_ACTIVE_UNIFORM_MAX_LENGTH, &nameBufSize);
#if NW_G3D_IS_HOST_WIN
        char* name = static_cast<char*>(_malloca(nameBufSize));
#else
        char* name = static_cast<char*>(alloca(nameBufSize));
#endif
        int numSampler = 0;
        for (int idxUniform = 0; idxUniform < numUniform; ++idxUniform)
        {
            GLenum type = GL_NONE;
            int size = 0;
            glGetActiveUniform(handle, idxUniform, nameBufSize, NULL, &size, &type, name);
            if (IsSamplerType(type))
            {
                int location = glGetUniformLocation(handle, name);
                GL::ProgramUniform(handle, location, numSampler++);
            }
        }
#if NW_G3D_IS_HOST_WIN
        _freea(name);
#endif
    }

    // uniform block に対して binding を指定してしまう。
    int numBlock = 0;
    glGetProgramiv(handle, GL_ACTIVE_UNIFORM_BLOCKS, &numBlock);
    GLint shaderBindingPoint;
    bit32 usedBindingPoints[(maxBindingPointNumber + 31) >> 5] = {0};
    // シェーダーで設定済みのbinding pointをチェックし、使用中にする
    for (int idxBlock = 0; idxBlock < numBlock; ++idxBlock)
    {
        glGetActiveUniformBlockiv(handle, idxBlock, GL_UNIFORM_BLOCK_BINDING, &shaderBindingPoint);
        if (shaderBindingPoint != 0 && shaderBindingPoint < maxBindingPointNumber)
        {
            SetBit(usedBindingPoints, shaderBindingPoint, 1);
        }
    }
    // binding pointを設定
    GLint bindingPoint = 0;
    for (int idxBlock = 0; idxBlock < numBlock; ++idxBlock)
    {
        glGetActiveUniformBlockiv(handle, idxBlock, GL_UNIFORM_BLOCK_BINDING, &shaderBindingPoint);
        // 未設定のuniform blockのみ設定する
        if (shaderBindingPoint == 0)
        {
            // 次の使用可能binding pointを探す
            for (; bindingPoint < maxBindingPointNumber; ++bindingPoint)
            {
                if ( IsBitOn<bool>(usedBindingPoints, bindingPoint) == false )
                {
                    break;
                }
            }

            NW_G3D_ASSERTMSG(bindingPoint < maxBindingPointNumber,
                             "Failed to assign binding point automatically. There are over %d uniform blocks.", maxBindingPointNumber);

            glUniformBlockBinding(handle, idxBlock, bindingPoint);
            ++bindingPoint;
        }
    }

    // shader storage buffer object に対して binding を指定してしまう。
#if !defined(NW_G3D_IS_GL_ES)
    int numSSBO = 0;
    glGetProgramInterfaceiv(handle, GL_SHADER_STORAGE_BLOCK,
        GL_ACTIVE_RESOURCES, &numSSBO);
    for (int idxSSBO = 0; idxSSBO < numSSBO; ++idxSSBO)
    {
        glShaderStorageBlockBinding(handle, idxSSBO, idxSSBO);
    }
#endif

#if NW_G3D_GL_PORTABILITY >= NW_G3D_GL_LEVEL2
    glUseProgram(0);
#endif // NW_G3D_GL_PORTABILITY
#else
    NW_G3D_UNUSED(arg);
#endif
}

bool GfxShader::IsGLBinaryAvailable(const SetupArg& arg)
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(arg.pBinary);
    GLint formatCount;
    glGetIntegerv(GL_NUM_PROGRAM_BINARY_FORMATS, &formatCount);
    if (formatCount < 1)
    {
        return false;
    }
    // 動的な数のフォーマット取得のため一時的にメモリを確保
    GLint* formats = static_cast<GLint*>(malloc(formatCount * sizeof(GLint)));
    glGetIntegerv(GL_PROGRAM_BINARY_FORMATS, formats);
    bool compatibleFormat = false;
    for (int idxFormat = 0; idxFormat < formatCount; ++idxFormat)
    {
        if (static_cast<uint>(formats[idxFormat]) == arg.binaryFormat)
        {
            compatibleFormat = true;
            break;
        }
    }
    free(formats);
    if (!compatibleFormat)
    {
        return false;
    }
    handle = CreateShaderProgram();
    glProgramBinary(handle, arg.binaryFormat, arg.pBinary, arg.binarySize);
    bool ret = CheckProgramStatus(handle);
    DestroyShaderProgram(handle);
    return ret;
#else
    NW_G3D_UNUSED(arg);
    return false;
#endif
}

void GfxShader::Cleanup()
{
#if NW_G3D_IS_GX2
    pVertexShader = NULL;
    pGeometryShader = NULL;
    pFragmentShader = NULL;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    DestroyShaderProgram(handle);
    handle = 0;
#endif
}

void GfxShader::Load() const
{
#if NW_G3D_IS_GX2
    if (pVertexShader)
    {
        GX2SetVertexShader(pVertexShader);
    }
    if (pGeometryShader)
    {
        GX2SetGeometryShader(pGeometryShader);
    }
    if (pFragmentShader)
    {
        GX2SetPixelShader(pFragmentShader);
    }
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (pComputeShader)
    {
        GX2SetComputeShader(pComputeShader);
    }
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    glUseProgram(handle);
    NW_G3D_GL_ASSERT();
#endif
}

//--------------------------------------------------------------------------------------------------

s8 GfxShader::GetVertexAttribLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    return pVertexShader ? GX2GetVertexAttribVarLocation(pVertexShader, name) : -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    return static_cast<s8>(glGetAttribLocation(handle, name));
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

//--------------------------------------------------------------------------------------------------

s8 GfxShader::GetVertexSamplerLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    return pVertexShader ? GX2GetVertexSamplerVarLocation(pVertexShader, name) : -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    int location = glGetUniformLocation(handle, name);
    if (location != -1)
    {
        glGetUniformiv(handle, location, &location);
    }
    return static_cast<s8>(location);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetGeometrySamplerLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    return pGeometryShader ? GX2GetGeometrySamplerVarLocation(pGeometryShader, name) : -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexSamplerLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetFragmentSamplerLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    return pFragmentShader ? GX2GetPixelSamplerVarLocation(pFragmentShader, name) : -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexSamplerLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetComputeSamplerLocation(const char* name) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    return pComputeShader ? GX2GetComputeSamplerVarLocation(pComputeShader, name) : -1;
#else
    return -1;
#endif
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexSamplerLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

//--------------------------------------------------------------------------------------------------

s8 GfxShader::GetVertexBlockLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pVertexShader)
    {
        GX2UniformBlock* pBlock = GX2GetVertexUniformBlock(pVertexShader, name);
        if (pBlock)
        {
            return static_cast<s8>(pBlock->location);
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    u32 blockIndex = glGetUniformBlockIndex(handle, name);
    if (blockIndex == GL_INVALID_INDEX)
    {
        return -1;
    }
    GLint bindingPoint = GL_INVALID_INDEX;
    glGetActiveUniformBlockiv(handle, blockIndex, GL_UNIFORM_BLOCK_BINDING, &bindingPoint);
    return static_cast<s8>(bindingPoint);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetGeometryBlockLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pGeometryShader)
    {
        GX2UniformBlock* pBlock = GX2GetGeometryUniformBlock(pGeometryShader, name);
        if (pBlock)
        {
            return static_cast<s8>(pBlock->location);
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetFragmentBlockLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pFragmentShader)
    {
        GX2UniformBlock* pBlock = GX2GetPixelUniformBlock(pFragmentShader, name);
        if (pBlock)
        {
            return static_cast<s8>(pBlock->location);
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s8 GfxShader::GetComputeBlockLocation(const char* name) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (pComputeShader)
    {
        GX2UniformBlock* pBlock = GX2GetComputeUniformBlock(pComputeShader, name);
        if (pBlock)
        {
            return static_cast<s8>(pBlock->location);
        }
    }
#endif
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

u32 GfxShader::GetVertexBlockSize(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pVertexShader)
    {
        GX2UniformBlock* pBlock = GX2GetVertexUniformBlock(pVertexShader, name);
        if (pBlock)
        {
            return pBlock->size;
        }
    }
    return 0;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    s32 size = 0;
    u32 blockIndex = glGetUniformBlockIndex(handle, name);
    if (blockIndex != GL_INVALID_INDEX)
    {
        glGetActiveUniformBlockiv(handle, blockIndex, GL_UNIFORM_BLOCK_DATA_SIZE, &size);
    }
    return static_cast<u32>(size);
#else
    NW_G3D_UNUSED(name);
    return 0;
#endif
}

u32 GfxShader::GetGeometryBlockSize(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pGeometryShader)
    {
        GX2UniformBlock* pBlock = GX2GetGeometryUniformBlock(pGeometryShader, name);
        if (pBlock)
        {
            return pBlock->size;
        }
    }
    return 0;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockSize(name);
#else
    NW_G3D_UNUSED(name);
    return 0;
#endif
}

u32 GfxShader::GetFragmentBlockSize(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pFragmentShader)
    {
        GX2UniformBlock* pBlock = GX2GetPixelUniformBlock(pFragmentShader, name);
        if (pBlock)
        {
            return pBlock->size;
        }
    }
    return 0;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockSize(name);
#else
    NW_G3D_UNUSED(name);
    return 0;
#endif
}

u32 GfxShader::GetComputeBlockSize(const char* name) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (pComputeShader)
    {
        GX2UniformBlock* pBlock = GX2GetComputeUniformBlock(pComputeShader, name);
        if (pBlock)
        {
            return pBlock->size;
        }
    }
#endif
    return 0;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexBlockSize(name);
#else
    NW_G3D_UNUSED(name);
    return 0;
#endif
}

s32 GfxShader::GetVertexUniformOffset(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pVertexShader)
    {
        GX2UniformVar* pUniform = GX2GetVertexUniformVar(pVertexShader, name);
        if (pUniform && pUniform->blockIndex != GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return sizeof(float) * pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    s32 offset = -1;
    u32 uniformIndex = GL_INVALID_INDEX;
    glGetUniformIndices(handle, 1, &name, &uniformIndex);
    if (uniformIndex != GL_INVALID_INDEX)
    {
        glGetActiveUniformsiv(handle, 1, &uniformIndex, GL_UNIFORM_OFFSET, &offset);
    }
    return offset;
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetGeometryUniformOffset(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pGeometryShader)
    {
        GX2UniformVar* pUniform = GX2GetGeometryUniformVar(pGeometryShader, name);
        if (pUniform && pUniform->blockIndex != GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return sizeof(float) * pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformOffset(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetFragmentUniformOffset(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pFragmentShader)
    {
        GX2UniformVar* pUniform = GX2GetPixelUniformVar(pFragmentShader, name);
        if (pUniform && pUniform->blockIndex != GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return sizeof(float) * pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformOffset(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetComputeUniformOffset(const char* name) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (pComputeShader)
    {
        GX2UniformVar* pUniform = GX2GetComputeUniformVar(pComputeShader, name);
        if (pUniform && pUniform->blockIndex != GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return sizeof(float) * pUniform->offset;
        }
    }
#endif
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformOffset(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

//--------------------------------------------------------------------------------------------------

s32 GfxShader::GetVertexUniformLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pVertexShader)
    {
        GX2UniformVar* pUniform = GX2GetVertexUniformVar(pVertexShader, name);
        if (pUniform && pUniform->blockIndex == GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    NW_G3D_ASSERT(handle != 0);
    return glGetUniformLocation(handle, name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetGeometryUniformLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pGeometryShader)
    {
        GX2UniformVar* pUniform = GX2GetGeometryUniformVar(pGeometryShader, name);
        if (pUniform && pUniform->blockIndex == GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetFragmentUniformLocation(const char* name) const
{
#if NW_G3D_IS_GX2
    if (pFragmentShader)
    {
        GX2UniformVar* pUniform = GX2GetPixelUniformVar(pFragmentShader, name);
        if (pUniform && pUniform->blockIndex == GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return pUniform->offset;
        }
    }
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

s32 GfxShader::GetComputeUniformLocation(const char* name) const
{
#if NW_G3D_IS_GX2
#if NW_G3D_COMPUTE_SHADER_ENABLE
    if (pComputeShader)
    {
        GX2UniformVar* pUniform = GX2GetComputeUniformVar(pComputeShader, name);
        if (pUniform && pUniform->blockIndex == GX2_UNIFORM_BLOCK_INDEX_INVALID)
        {
            return pUniform->offset;
        }
    }
#endif
    return -1;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    return GetVertexUniformLocation(name);
#else
    NW_G3D_UNUSED(name);
    return -1;
#endif
}

//--------------------------------------------------------------------------------------------------

void SetShaderMode(GX2ShaderMode mode)
{
#if NW_G3D_IS_GX2
    GX2SetShaderMode(mode);
#else
    NW_G3D_UNUSED(mode);
#endif
}

void SetShaderModeEx(GX2ShaderMode mode,
                      u32 vsGprs,
                      u32 vsStackSize,
                      u32 gsGprs,
                      u32 gsStackSize,
                      u32 psGprs,
                      u32 psStackSize)
{
#if NW_G3D_IS_GX2
    GX2SetShaderModeEx(mode, vsGprs, vsStackSize, gsGprs, gsStackSize, psGprs, psStackSize);
#else
    NW_G3D_UNUSED(mode);
    NW_G3D_UNUSED(vsGprs);
    NW_G3D_UNUSED(vsStackSize);
    NW_G3D_UNUSED(gsGprs);
    NW_G3D_UNUSED(gsStackSize);
    NW_G3D_UNUSED(psGprs);
    NW_G3D_UNUSED(psStackSize);
#endif
}

size_t GfxRingBuffer::CalcInputBufferSize(u32 ringItem)
{
#if NW_G3D_IS_GX2
    return GX2CalcGeometryShaderInputRingBufferSize(ringItem);
#else
    NW_G3D_UNUSED(ringItem);
    return 0;
#endif
}

size_t GfxRingBuffer::CalcOutputBufferSize(u32 ringItem)
{
#if NW_G3D_IS_GX2
    return GX2CalcGeometryShaderOutputRingBufferSize(ringItem);
#else
    NW_G3D_UNUSED(ringItem);
    return 0;
#endif
}

void GfxRingBuffer::SetInputBuffer(void* ptr, size_t size)
{
    NW_G3D_ASSERT(ptr != NULL || size == 0);
    m_pInputBuffer = ptr;
    m_InputSize = size;
}

void GfxRingBuffer::SetOutputBuffer(void* ptr, size_t size)
{
    NW_G3D_ASSERT(ptr != NULL || size == 0);
    m_pOutputBuffer = ptr;
    m_OutputSize = size;
}

void GfxRingBuffer::Load()
{
#if NW_G3D_IS_GX2
    GX2SetGeometryShaderInputRingBuffer(m_pInputBuffer, m_InputSize);
    GX2SetGeometryShaderOutputRingBuffer(m_pOutputBuffer, m_OutputSize);
#endif
}

}}} // namespace nw::g3d::fnd
