﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/fnd/g3d_GfxManage.h>

#include <algorithm>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/fnd/g3d_GfxState.h>
#include <nw/g3d/fnd/g3d_WinUtility.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#if defined(ANDROID)
#include <nw/g3d/fnd/g3d_EglUtility.h>
#endif

#if NW_G3D_IS_HOST_WIN
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#elif NW_G3D_IS_HOST_CAFE
#include <cafe/os/OSTime.h>
#endif // NW_G3D_PLATFORM

namespace nw { namespace g3d { namespace fnd {

#if NW_G3D_IS_HOST_WIN || defined(ANDROID) || defined(__APPLE__)

namespace {

using detail::GL;
using detail::SystemShader;
using detail::SystemContext;

#if NW_G3D_IS_HOST_WIN
HWND s_hWnd;
HDC s_hDC;
#endif

SystemContext s_System;

#if NW_G3D_IS_HOST_WIN
inline HGLRC GetRC() { return static_cast<HGLRC>(s_System.ctx.hRC); }
inline void SetRC(void* hRC) { s_System.ctx.hRC = static_cast<HGLRC>(hRC); }
#endif

const char* vshCopy =
#if defined(NW_G3D_IS_GL_ES)
"#version 300 es\n"
#else
"#version 330\n"
#endif
"layout(location = 0) in vec2 pos;\n"
"out vec2 uv;\n"
"void main()\n"
"{\n"
"    gl_Position = vec4(pos, 0, 1);\n"
"    uv.x = pos.x * 0.5 + 0.5;\n"
"    uv.y = 1.0 - (pos.y * 0.5 + 0.5);\n"
"}\n";

const char* const vshSwapCopy =
#if defined(NW_G3D_IS_GL_ES)
"#version 300 es\n"
#else
"#version 330\n"
#endif
"layout(location = 0) in vec2 pos;\n"
"out vec2 uv;\n"
"void main()\n"
"{\n"
"    gl_Position = vec4(pos, 0, 1);\n"
"    uv = pos.xy * 0.5 + 0.5;\n"
"}\n";

const char* fshCopy =
#if defined(NW_G3D_IS_GL_ES)
"#version 300 es\n"
#else
"#version 330\n"
#endif
"precision highp float;\n"
"in vec2 uv;\n"
"layout(location = 0) out vec4 o_Color;\n"
"uniform sampler2D tex;\n"
"uniform float gamma;\n"
"void main()\n"
"{\n"
"    vec4 color = texture(tex, uv);\n"
"    o_Color = pow(color, vec4(gamma, gamma, gamma, 1));\n"
"}\n";

const char* const fshResolve =
"#version 330\n"
"in vec2 uv;\n"
"layout(location = 0) out vec4 o_Color;\n"
"uniform sampler2DMS tex;\n"
"uniform float gamma;\n"
"void main()\n"
"{\n"
"    vec4 color = vec4(0.0, 0.0, 0.0, 0.0);\n"
"    ivec2 texel = ivec2(uv * textureSize(tex));\n"
"    for (int sample = 0; sample < 8; ++sample)\n"
"    {\n"
"        color += texelFetch(tex, texel, sample);\n"
"    }\n"
"    color *= 0.125;\n"
"    o_Color = pow(color, vec4(gamma, gamma, gamma, 1));\n"
"}\n";

const char* const fshCopy3D =
#if defined(NW_G3D_IS_GL_ES)
"#version 300 es\n"
#else
"#version 330\n"
#endif
"precision highp float;\n"
"precision highp sampler3D;\n"
"in vec2 uv;\n"
"layout(location = 0) out vec4 o_Color;\n"
"uniform sampler3D tex;\n"
"uniform float gamma;\n"
"uniform int slice;\n"
"void main()\n"
"{\n"
"    vec4 color = texture(tex, vec3(uv, float(slice)));\n"
"    o_Color = pow(color, vec4(gamma, gamma, gamma, 1));\n"
"}\n";

const char* const fshCopy2DArray =
#if defined(NW_G3D_IS_GL_ES)
"#version 300 es\n"
#else
"#version 330\n"
#endif
"precision highp float;\n"
"precision highp sampler2DArray;\n"
"in vec2 uv;\n"
"layout(location = 0) out vec4 o_Color;\n"
"uniform sampler2DArray tex;\n"
"uniform float gamma;\n"
"uniform int slice;\n"
"void main()\n"
"{\n"
"    vec4 color = texture(tex, vec3(uv, float(slice)));\n"
"    o_Color = pow(color, vec4(gamma, gamma, gamma, 1));\n"
"}\n";

const char* const fshResolveArray =
"#version 330\n"
"in vec2 uv;\n"
"layout(location = 0) out vec4 o_Color;\n"
"uniform sampler2DMSArray tex;\n"
"uniform float gamma;\n"
"uniform int slice;\n"
"void main()\n"
"{\n"
"    vec4 color = vec4(0.0, 0.0, 0.0, 0.0);\n"
"    ivec3 texel = ivec3(uv * textureSize(tex).xy, slice);\n"
"    for (int sample = 0; sample < 8; ++sample)\n"
"    {\n"
"        color += texelFetch(tex, texel, sample);\n"
"    }\n"
"    color *= 0.125;\n"
"    o_Color = pow(color, vec4(gamma, gamma, gamma, 1));\n"
"}\n";

u32 CreateShader(u32* locations, const char* vsh, const char* fsh)
{
    u32 handle = CreateShaderProgram();
    AttachShader(handle, GL_VERTEX_SHADER, vsh);
    AttachShader(handle, GL_FRAGMENT_SHADER, fsh);
    LinkShaderProgram(handle);
    u32 locTex = glGetUniformLocation(handle, "tex");
    u32 locGamma = glGetUniformLocation(handle, "gamma");
    u32 locSlice = glGetUniformLocation(handle, "slice");

    GL::ProgramUniform(handle, locTex, SystemShader::SAMPLER_TEX);
    GL::ProgramUniform(handle, locGamma, 1.0f);

    NW_G3D_GL_ASSERT();

    locations[SystemShader::LOCATION_GAMMA] = locGamma;
    locations[SystemShader::LOCATION_SLICE] = locSlice;

    return handle;
}


#if !defined(NW_G3D_IS_GL_ES)
void APIENTRY GLDebugCallback(GLenum source,
                              GLenum type,
                              GLuint id,
                              GLenum severity,
                              GLsizei /*length*/,
                              const GLchar* message,
#if defined(GL_VERSION_4_5)
                              const GLvoid* /*userParam*/)
#else
                              GLvoid* /*userParam*/)
#endif
{
    nw::g3d::DebugPrint(
        "%s\n\tSource: %s, Type: %s, ID: %d, Severity: %s\n",
        message,
        GetGLDebugString(source),
        GetGLDebugString(type),
        id,
        GetGLDebugString(severity));
}
#endif

} // anonymous namespace

namespace detail {

SystemContext::SystemContext()
{
#if !defined(NW_G3D_IS_GL_ES)
    s_hWnd = detail::CreateSystemWnd();
    NW_G3D_ASSERT_NOT_NULL(s_hWnd);
    s_hDC = GetDC(s_hWnd);
    NW_G3D_ASSERT_NOT_NULL(s_hDC);
#endif
    ctx.hRC = NULL;
#if !defined(NW_G3D_IS_GL_ES)
    Setup();
#endif
}

SystemContext::~SystemContext()
{
    Cleanup();
#if !defined(NW_G3D_IS_GL_ES)
    ReleaseDC(s_hWnd, s_hDC);
    s_hDC = NULL;
    detail::DestroyWnd(s_hWnd);
    s_hWnd = NULL;
#endif
}

void SystemContext::Setup()
{
#if !defined(NW_G3D_IS_GL_ES)
    ctx.Setup();
    ctx.Prepare();
    glewInit();
#endif

    // framebuffer
    glGenFramebuffers(1, &hFrameBuffer);

    // sampler
    glGenSamplers(1, &hSampler);
    glSamplerParameteri(hSampler, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(hSampler, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(hSampler, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(hSampler, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glSamplerParameteri(hSampler, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
    glSamplerParameteri(hSampler, GL_TEXTURE_COMPARE_MODE, GL_NONE);

    copyShader.hShader             = CreateShader(
        copyShader.locations, vshCopy, fshCopy);
    copyShader.texUnit             = SystemShader::SAMPLER_TEX;

#if !defined(NW_G3D_IS_GL_ES)
    resolveShader.hShader          = CreateShader(
        resolveShader.locations, vshCopy, fshResolve);
    resolveShader.texUnit          = SystemShader::SAMPLER_TEX;
#endif

    copy3DShader.hShader           = CreateShader(
        copy3DShader.locations, vshCopy, fshCopy3D);
    copy3DShader.texUnit           = SystemShader::SAMPLER_TEX;

    copy2DArrayShader.hShader      = CreateShader(
        copy2DArrayShader.locations, vshCopy, fshCopy2DArray);
    copy2DArrayShader.texUnit      = SystemShader::SAMPLER_TEX;

#if !defined(NW_G3D_IS_GL_ES)
    resolveArrayShader.hShader     = CreateShader(
        resolveArrayShader.locations, vshCopy, fshResolveArray);
    resolveArrayShader.texUnit     = SystemShader::SAMPLER_TEX;
#endif

    swapCopyShader.hShader         = CreateShader(
        swapCopyShader.locations, vshSwapCopy, fshCopy);
    swapCopyShader.texUnit         = SystemShader::SAMPLER_TEX;

#if !defined(NW_G3D_IS_GL_ES)
    swapResolveShader.hShader      = CreateShader(
        swapResolveShader.locations, vshSwapCopy, fshResolve);
    swapResolveShader.texUnit      = SystemShader::SAMPLER_TEX;
#endif

    swapCopy3DShader.hShader       = CreateShader(
        swapCopy3DShader.locations, vshSwapCopy, fshCopy3D);
    swapCopy3DShader.texUnit       = SystemShader::SAMPLER_TEX;

    swapCopy2DArrayShader.hShader  = CreateShader(
        swapCopy2DArrayShader.locations, vshSwapCopy, fshCopy2DArray);
    swapCopy2DArrayShader.texUnit  = SystemShader::SAMPLER_TEX;

#if !defined(NW_G3D_IS_GL_ES)
    swapResolveArrayShader.hShader = CreateShader(
        swapResolveArrayShader.locations, vshSwapCopy, fshResolveArray);
    swapResolveArrayShader.texUnit = SystemShader::SAMPLER_TEX;
#endif
    NW_G3D_GL_ASSERT();
}

void SystemContext::Cleanup()
{
#if !defined(NW_G3D_IS_GL_ES)
    ctx.Prepare();

    NW_G3D_GL_ASSERT();
#endif

    glDeleteFramebuffers(1, &hFrameBuffer);
    glDeleteSamplers(1, &hSampler);

    DestroyShaderProgram(copyShader.hShader);
#if !defined(NW_G3D_IS_GL_ES)
    DestroyShaderProgram(resolveShader.hShader);
#endif
    DestroyShaderProgram(copy3DShader.hShader);
    DestroyShaderProgram(copy2DArrayShader.hShader);
#if !defined(NW_G3D_IS_GL_ES)
    DestroyShaderProgram(resolveArrayShader.hShader);
#endif
    DestroyShaderProgram(swapCopyShader.hShader);
#if !defined(NW_G3D_IS_GL_ES)
    DestroyShaderProgram(swapResolveShader.hShader);
#endif
    DestroyShaderProgram(swapCopy3DShader.hShader);
    DestroyShaderProgram(swapCopy2DArrayShader.hShader);
#if !defined(NW_G3D_IS_GL_ES)
    DestroyShaderProgram(swapResolveArrayShader.hShader);
#endif

    NW_G3D_GL_ASSERT();

#if !defined(NW_G3D_IS_GL_ES)
    GfxContext::Invalidate();
    ctx.Cleanup();
#endif
}

void ResetViewportScissor(::GX2Surface& surface, u32 mipLevel)
{
    // GX2 の仕様に合わせるため Viewport と Scissor をサーフェイス全体に設定する。
    int width = std::max(1, static_cast<int>(surface.width >> mipLevel));
    int height = std::max(1, static_cast<int>(surface.height >> mipLevel));
    glViewport(0, 0, width, height);
    glScissor(0, 0, width, height);
}

void SystemContext::BindFrameBuffer(GfxTexture* pTexture, u32 mipLevel, u32 slice)
{
    NW_G3D_ASSERT_NOT_NULL(pTexture);

    ResetViewportScissor(pTexture->GetGX2Texture()->surface, mipLevel);

    glBindFramebuffer(GL_FRAMEBUFFER, hFrameBuffer);

    GLenum drawBuffer = GL_COLOR_ATTACHMENT0;
    GL::FrameBufferDrawBuffers(hFrameBuffer, 1, &drawBuffer);
    GL::FrameBufferTexture(hFrameBuffer, GL_DEPTH_ATTACHMENT, 0, 0);
    if (pTexture->GetDepth() > 1)
    {
        GL::FrameBufferTexture(
            hFrameBuffer, GL_COLOR_ATTACHMENT0, pTexture->handle, mipLevel, slice);
    }
    else
    {
        GL::FrameBufferTexture(hFrameBuffer, GL_COLOR_ATTACHMENT0, pTexture->handle, mipLevel);
    }

    bool result = CheckFramebufferStatus(hFrameBuffer);
    NW_G3D_UNUSED(result);
    NW_G3D_ASSERT(result);

    NW_G3D_GL_ASSERT();
}

void SystemContext::BindFrameBuffer(GfxColorBuffer* pColorBuffer, int layer)
{
    NW_G3D_ASSERT_NOT_NULL(pColorBuffer);

    u32 mipLevel = pColorBuffer->GetGX2ColorBuffer()->viewMip;
    ResetViewportScissor(pColorBuffer->GetGX2ColorBuffer()->surface, mipLevel);

    glBindFramebuffer(GL_FRAMEBUFFER, hFrameBuffer);

    GLenum drawBuffer = GL_COLOR_ATTACHMENT0;
    GL::FrameBufferDrawBuffers(hFrameBuffer, 1, &drawBuffer);
    GL::FrameBufferTexture(hFrameBuffer, GL_DEPTH_ATTACHMENT, 0, 0);
    if (layer >= 0)
    {
        GL::FrameBufferTexture(hFrameBuffer, GL_COLOR_ATTACHMENT0, pColorBuffer->handle, mipLevel, layer);
    }
    else
    {
        GL::FrameBufferTexture(hFrameBuffer, GL_COLOR_ATTACHMENT0, pColorBuffer->handle, mipLevel);
    }

    bool result = CheckFramebufferStatus(hFrameBuffer);
    NW_G3D_UNUSED(result);
    NW_G3D_ASSERT(result);

    NW_G3D_GL_ASSERT();
}

void SystemContext::BindFrameBuffer(GfxDepthBuffer* pDepthBuffer, int layer)
{
    NW_G3D_ASSERT_NOT_NULL(pDepthBuffer);

    u32 mipLevel = pDepthBuffer->GetGX2DepthBuffer()->viewMip;
    ResetViewportScissor(pDepthBuffer->GetGX2DepthBuffer()->surface, mipLevel);

    glBindFramebuffer(GL_FRAMEBUFFER, hFrameBuffer);

    GLenum drawBuffer = GL_NONE;
    GX2SurfaceFormat format = pDepthBuffer->GetGX2DepthBuffer()->surface.format;
    GLenum attachment =
        format == GX2_SURFACE_FORMAT_D_D24_S8_UNORM ||
        format == GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24 ?
            GL_DEPTH_STENCIL_ATTACHMENT : GL_DEPTH_ATTACHMENT;
    GL::FrameBufferDrawBuffers(hFrameBuffer, 1, &drawBuffer);
    if (layer >= 0)
    {
        GL::FrameBufferTexture(hFrameBuffer, attachment, pDepthBuffer->handle, mipLevel, layer);
    }
    else
    {
        GL::FrameBufferTexture(hFrameBuffer, attachment, pDepthBuffer->handle, mipLevel);
    }
    GL::FrameBufferTexture(hFrameBuffer, GL_COLOR_ATTACHMENT0, 0, 0);

    bool result = CheckFramebufferStatus(hFrameBuffer);
    NW_G3D_UNUSED(result);
    NW_G3D_ASSERT(result);

    NW_G3D_GL_ASSERT();
}

void SystemContext::BindTexture(u32 handle, u32 unit, u32 mipLevel, GX2SurfaceDim dim)
{
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_TABLE_FIELD GLenum s_tblTarget[] = {
        GL_TEXTURE_1D,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_1D_ARRAY,
        GL_TEXTURE_2D_ARRAY,
        GL_TEXTURE_2D_MULTISAMPLE,
        GL_TEXTURE_2D_MULTISAMPLE_ARRAY
    };
#else
    NW_G3D_TABLE_FIELD GLenum s_tblTarget[] = {
        GL_INVALID_INDEX,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_INVALID_INDEX,
        GL_TEXTURE_2D_ARRAY,
        GL_INVALID_INDEX,
        GL_INVALID_INDEX
    };

    NW_G3D_ASSERTMSG(s_tblTarget[dim] != GL_INVALID_INDEX, "NW: Unsupported texture target.");
#endif

    GL::BindTexture(unit, s_tblTarget[dim], handle);
    glSamplerParameteri(hSampler, GL_TEXTURE_MIN_LOD, mipLevel);
    glSamplerParameteri(hSampler, GL_TEXTURE_MAX_LOD, mipLevel);
    glBindSampler(unit, hSampler);
    NW_G3D_GL_ASSERT();
}

void SystemContext::DrawScreenQuad()
{
    static const float pos[][2] = {
        { -1.0f, 1.0f }, { 1.0f, 1.0f }, { -1.0f, -1.0f }, { 1.0f, -1.0f }
    };
    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, &pos[0][0]);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    NW_G3D_GL_ASSERT();
}

const SystemShader& SystemContext::SelectCopyShader(GX2SurfaceDim dim, bool swap) const
{
    switch(dim)
    {
    case GX2_SURFACE_DIM_2D_MSAA:
#if !defined(NW_G3D_IS_GL_ES)
        return swap ? swapResolveShader : resolveShader;
#else
        NW_G3D_ASSERTMSG(false, "NW: Can't select copy shader for GX2_SURFACE_DIM_2D_MSAA.");
        return swap ? swapResolveShader : resolveShader;
#endif
        break;
    case GX2_SURFACE_DIM_3D:
        return swap ? swapCopy3DShader : copy3DShader;
        break;
    case GX2_SURFACE_DIM_2D_ARRAY:
        return swap ? swapCopy2DArrayShader : copy2DArrayShader;
        break;
    case GX2_SURFACE_DIM_2D_MSAA_ARRAY:
#if !defined(NW_G3D_IS_GL_ES)
        return swap ? swapResolveArrayShader : resolveArrayShader;
#else
        NW_G3D_ASSERTMSG(false, "NW: Can't select copy shader for GX2_SURFACE_DIM_2D_MSAA_ARRAY.");
        return swap ? swapResolveArrayShader : resolveArrayShader;
#endif
        break;
    default:
        return swap ? swapCopyShader : copyShader;
        break;
    }
}

SystemContext& GetSystemContext()
{
    return s_System;
}

} // namespace detail

void GfxContext::Reinit(void* hRC)
{
#if defined(NW_G3D_IS_GL_ES)
    NW_G3D_WARNING(false, "GfxContext doesn't do anything except InitEGL method.");
#else
    s_System.Cleanup();
    SetRC(hRC);
    s_System.Setup();
#endif
}

#endif // NW_G3D_IS_HOST_WIN

#if defined(NW_G3D_IS_GL_ES)
#if defined(ANDROID)

// EGLの初期化およびコンテキスト作成を行います。
// Android向け対応では作成されたコンテキストはアプリ、システムの両方から使用されます。
bool GfxContext::InitEGL(const struct android_app* app)
{
    bool ret = detail::CreateGLContext(app);
    s_System.Setup();
    return ret;
}

#endif
#endif

//--------------------------------------------------------------------------------------------------

void GfxContext::Setup()
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT_ADDR_ALIGNMENT(this, GX2_CONTEXT_STATE_ALIGNMENT);
    GX2SetupContextState(&gx2ContextState);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    hRC = detail::CreateGLContext(s_hDC, GetRC());
    NW_G3D_ASSERT_NOT_NULL(hRC);
    wglMakeCurrent(s_hDC, static_cast<HGLRC>(hRC));
    if (glDebugMessageCallbackARB)
    {
        glDebugMessageCallbackARB(GLDebugCallback, NULL);
    }
    wglMakeCurrent(NULL, NULL);
#else
    NW_G3D_WARNING(false, "GfxContext doesn't do anything except InitEGL method.");
#endif
#endif
}

void GfxContext::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT_NOT_NULL(hRC);
    detail::DeleteGLContext(static_cast<HGLRC>(hRC));
    hRC = NULL;
#endif
#endif
}


void GfxContext::Activate()
{
#if NW_G3D_IS_GX2
    GX2SetContextState(&gx2ContextState);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT_NOT_NULL(hRC);
    wglMakeCurrent(s_hDC, static_cast<HGLRC>(hRC));
#endif
#endif
}

void GfxContext::Prepare()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    wglMakeCurrent(s_hDC, GetRC());
#endif
#endif
}

void GfxContext::TempPrepare()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT_NOT_NULL(hRC);
    wglMakeCurrent(s_hDC, static_cast<HGLRC>(hRC));
#endif
#endif
}

void GfxContext::Invalidate()
{
#if NW_G3D_IS_GX2
    GX2SetContextState(NULL);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    wglMakeCurrent(NULL, NULL);
#endif
#endif
}

//--------------------------------------------------------------------------------------------------

void GfxManage::DrawDone()
{
#if NW_G3D_IS_GX2
    GX2DrawDone();
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    HDC hDC = wglGetCurrentDC();
    HGLRC hRC = wglGetCurrentContext();
    detail::SystemContext& system = detail::GetSystemContext();
    system.ctx.Activate();
    glFinish();
    NW_G3D_GL_ASSERT();
    wglMakeCurrent(hDC, hRC);
#else
    glFinish();
    NW_G3D_GL_ASSERT();
#endif
#endif
}

void GfxManage::FlushCommands()
{
#if NW_G3D_IS_GX2
    GX2Flush();
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL )
#if !defined(NW_G3D_IS_GL_ES)
    HDC hDC = wglGetCurrentDC();
    HGLRC hRC = wglGetCurrentContext();
    detail::SystemContext& system = detail::GetSystemContext();
    system.ctx.Activate();
#endif
    glFlush();
    NW_G3D_GL_ASSERT();
#if !defined(NW_G3D_IS_GL_ES)
    wglMakeCurrent(hDC, hRC);
#endif
#endif
}

//--------------------------------------------------------------------------------------------------

#if NW_G3D_IS_GX2
GPUClock::Tick GPUClock::s_Freq = OS_TIMER_CLOCK;
#elif NW_G3D_IS_GL
GPUClock::Tick GPUClock::s_Freq = 1000 * 1000 * 1000;
#endif

void GPUClock::Setup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle == 0)
    {
        glGenQueries(1, &handle);
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GPUClock::Cleanup()
{
#if NW_G3D_IS_GL && !defined( NW_STRIP_GL )
    if (handle)
    {
        glDeleteQueries(1, &handle);
        handle = 0;
        NW_G3D_GL_ASSERT();
    }
#endif
}

void GPUClock::Query()
{
#if NW_G3D_IS_GX2
    GX2SampleBottomGPUCycle(&timestamp);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    NW_G3D_ASSERT(handle != 0);
    glQueryCounter(handle, GL_TIMESTAMP);
    NW_G3D_GL_ASSERT();
#endif
}

bool GPUClock::IsReady() const
{
#if NW_G3D_IS_GX2
    return GX2_FALSE != GX2GetGPUCycleReady(&timestamp);
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    GLint available = 0;
    NW_G3D_ASSERT(handle != 0);
    glGetQueryObjectiv(handle, GL_QUERY_RESULT_AVAILABLE, &available);
    NW_G3D_GL_ASSERT();
    return 0 != available;
#else
    return false;
#endif
}

GPUClock::Tick GPUClock::GetTimeStamp() const
{
#if NW_G3D_IS_GX2
    NW_G3D_ASSERT(timestamp != GX2_INVALID_COUNTER_VALUE_U64);
    return static_cast<Tick>(GX2GPUTimeToCPUTime(timestamp));
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    GLuint64 ticks = 0;
    glGetQueryObjectui64v(handle, GL_QUERY_RESULT, &ticks);
    NW_G3D_GL_ASSERT();
    return static_cast<Tick>(ticks);
#else
    return 0;
#endif
}

GPUClock::Tick GPUClock::Now()
{
#if NW_G3D_IS_GX2
    return static_cast<Tick>(OSGetSystemTime());
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    GLint64 ticks = 0;
    glGetInteger64v(GL_TIMESTAMP, &ticks);
    NW_G3D_GL_ASSERT();
    return static_cast<Tick>(ticks);
#else
    return 0;
#endif
}

//--------------------------------------------------------------------------------------------------

CPUClock::Freq CPUClock::s_Freq;

void CPUClock::Freq::Init()
{
#if NW_G3D_IS_GX2
    m_Value = OS_TIMER_CLOCK;
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    BOOL success = FALSE;
    success = QueryPerformanceFrequency(reinterpret_cast<LARGE_INTEGER*>(&m_Value));
    NW_G3D_ASSERT(success);
#else
    m_Value = 0;
#endif
}

CPUClock::Tick CPUClock::Now()
{
#if NW_G3D_IS_GX2
    return OSGetSystemTime();
#elif NW_G3D_IS_GL && !defined( NW_STRIP_GL ) && !defined(NW_G3D_IS_GL_ES)
    s64 ticks = 0;
    QueryPerformanceCounter(reinterpret_cast<LARGE_INTEGER*>(&ticks));
    return ticks;
#else
    return 0;
#endif
}

//--------------------------------------------------------------------------------------------------

void GPUCache::InvalidateAll()
{
#if NW_G3D_IS_GX2
    GX2Invalidate(static_cast<GX2InvalidateType>(
        GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE |
        GX2_INVALIDATE_UNIFORM_BLOCK | GX2_INVALIDATE_SHADER),
        NULL, 0xFFFFFFFF);
#endif
}

void GPUCache::FlushAll()
{
#if NW_G3D_IS_GX2
    GX2Invalidate(static_cast<GX2InvalidateType>(
        GX2_INVALIDATE_COLOR_BUFFER | GX2_INVALIDATE_DEPTH_BUFFER |
        GX2_INVALIDATE_STREAMOUT_BUFFER
#if NW_G3D_COMPUTE_SHADER_ENABLE
        | GX2_INVALIDATE_EXPORT_BUFFER
#endif
        ),
        NULL, 0xFFFFFFFF);
#endif
}

//--------------------------------------------------------------------------------------------------

void CPUCache::Flush(const void* addr, size_t size)
{
#if NW_G3D_IS_HOST_CAFE
#if NW_G3D_FORCE_PPC_SYNC
    DCFlushRange(addr, size);
#else
    DCFlushRangeNoSync(addr, size);
#endif // NW_G3D_FORCE_PPC_SYNC
#else
    NW_G3D_UNUSED(addr);
    NW_G3D_UNUSED(size);
#endif
}

void CPUCache::Store(const void* addr, size_t size)
{
#if NW_G3D_IS_HOST_CAFE
#if NW_G3D_FORCE_PPC_SYNC
    DCStoreRange(addr, size);
#else
    DCStoreRangeNoSync(addr, size);
#endif // NW_G3D_FORCE_PPC_SYNC
#else
    NW_G3D_UNUSED(addr);
    NW_G3D_UNUSED(size);
#endif
}

void CPUCache::Invalidate(void* addr, size_t size)
{
    NW_G3D_ASSERT_ADDR_ALIGNMENT(addr, CACHE_BLOCK_SIZE);
    NW_G3D_ASSERT_ALIGNMENT(size, CACHE_BLOCK_SIZE);
#if NW_G3D_IS_HOST_CAFE
    DCInvalidateRange(addr, size);
#else
    NW_G3D_UNUSED(addr);
    NW_G3D_UNUSED(size);
#endif
}

void CPUCache::Sync()
{
#if NW_G3D_IS_HOST_CAFE
    OSMemoryBarrier();
#endif
}

bool CPUCache::IsValid(const void* addr, size_t size)
{
    NW_G3D_ASSERT(size > 0);
#if NW_G3D_IS_HOST_CAFE
    return FALSE != OSIsAddressRangeDCValid(addr, size);
#else
    NW_G3D_UNUSED(addr);
    NW_G3D_UNUSED(size);
    return true;
#endif
}

void CPUCache::FillZero(void* addr, size_t size)
{
    NW_G3D_ASSERT_ADDR_ALIGNMENT(addr, CACHE_BLOCK_SIZE);
    NW_G3D_ASSERT_ALIGNMENT(size, CACHE_BLOCK_SIZE);
    NW_G3D_ASSERT(IsValid(addr, size));
#if NW_G3D_IS_HOST_CAFE
    DCZeroRange(addr, size);
#else
    memset(addr, 0, size);
#endif
}

//--------------------------------------------------------------------------------------------------

size_t LockedCache::GetAllocatableSize()
{
#if NW_G3D_IS_HOST_CAFE
    return LCGetUnallocated();
#else
    return 0;
#endif
}

void* LockedCache::Alloc(size_t size)
{
#if NW_G3D_IS_HOST_CAFE
    return LCAlloc(size);
#else
    NW_G3D_UNUSED(size);
    return NULL;
#endif
}

void LockedCache::Free(void* ptr)
{
#if NW_G3D_IS_HOST_CAFE
    LCDealloc(ptr);
#else
    NW_G3D_UNUSED(ptr);
#endif
}

bool LockedCache::EnableDMA()
{
#if NW_G3D_IS_HOST_CAFE
    return LCEnableDMA();
#else
    return false;
#endif
}

void LockedCache::DisableDMA()
{
#if NW_G3D_IS_HOST_CAFE
    LCDisableDMA();
#endif
}

bool LockedCache::IsDMAEnabled()
{
#if NW_G3D_IS_HOST_CAFE
    return LCIsDMAEnabled();
#else
    return false;
#endif
}

u32 LockedCache::LoadDMABlocks(void* pDstLC, const void* pSrcMem, u32 numBlocks)
{
#if NW_G3D_IS_HOST_CAFE
    u32 numTransactions = (numBlocks + LC_MAX_DMA_BLOCKS - 1) / LC_MAX_DMA_BLOCKS;
    while (numBlocks > 0)
    {
        if (numBlocks < LC_MAX_DMA_BLOCKS)
        {
            LCLoadDMABlocks(pDstLC, const_cast<void*>(pSrcMem), numBlocks);
            numBlocks = 0;
        }
        else
        {
            LCLoadDMABlocks(pDstLC, const_cast<void*>(pSrcMem), 0);
            numBlocks  -= LC_MAX_DMA_BLOCKS;
            pDstLC = AddOffset(pDstLC, LC_MAX_DMA_BYTES);
            pSrcMem = AddOffset(pSrcMem, LC_MAX_DMA_BYTES);
        }
    }
    return numTransactions;
#else
    NW_G3D_UNUSED(pDstLC);
    NW_G3D_UNUSED(pSrcMem);
    NW_G3D_UNUSED(numBlocks);
    return 0;
#endif
}

u32 LockedCache::StoreDMABlocks(void* pDstMem, const void* pSrcLC, u32 numBlocks)
{
#if NW_G3D_IS_HOST_CAFE
    u32 numTransactions = (numBlocks + LC_MAX_DMA_BLOCKS - 1) / LC_MAX_DMA_BLOCKS;
    while (numBlocks > 0)
    {
        if (numBlocks < LC_MAX_DMA_BLOCKS)
        {
            LCStoreDMABlocks(pDstMem, const_cast<void*>(pSrcLC), numBlocks);
            numBlocks = 0;
        }
        else
        {
            LCStoreDMABlocks(pDstMem, const_cast<void*>(pSrcLC), 0);
            numBlocks  -= LC_MAX_DMA_BLOCKS;
            pDstMem = AddOffset(pDstMem, LC_MAX_DMA_BYTES);
            pSrcLC = AddOffset(pSrcLC, LC_MAX_DMA_BYTES);
        }
    }
    return numTransactions;
#else
    NW_G3D_UNUSED(pDstMem);
    NW_G3D_UNUSED(pSrcLC);
    NW_G3D_UNUSED(numBlocks);
    return 0;
#endif
}

void LockedCache::WaitDMAQueue(u32 length)
{
#if NW_G3D_IS_HOST_CAFE
    LCWaitDMAQueue(length);
#else
    NW_G3D_UNUSED(length);
#endif
}

}}} // namespace nw::g3d::fnd
