﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/fnd/g3d_EglUtility.h>
#include <cstdlib>

#if defined(NW_G3D_IS_GL_ES) && defined(ANDROID)

namespace nw { namespace g3d { namespace fnd { namespace detail {

namespace {
    struct EglInfo {
        EGLDisplay display;
        EGLSurface surface;
        EGLContext context;
        int32_t width;
        int32_t height;

        EglInfo()
        : display(EGL_NO_DISPLAY), context(EGL_NO_CONTEXT),
          surface(EGL_NO_SURFACE), width(0), height(0) { }
    } eglInfo;
}

bool CreateGLContext(const struct android_app* app)
{
    NW_G3D_ASSERT_NOT_NULL(app);

    // OpenGL ES and EGLの初期化

    // カラー8bitのアトリビュート
    const EGLint attribs[] = {
            EGL_BLUE_SIZE, 8,
            EGL_GREEN_SIZE, 8,
            EGL_RED_SIZE, 8,
            EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
            EGL_NONE
    };
    const EGLint contextAttribs[] = {
            EGL_CONTEXT_CLIENT_VERSION, 2,
            EGL_NONE
    };

    EGLSurface surface;
    EGLContext context;

    EGLDisplay display = eglGetDisplay(EGL_DEFAULT_DISPLAY);
    if (display == EGL_NO_DISPLAY)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglGetDisplay");
        return false;
    }

    EGLint major = 0, minor = 0;
    if (eglInitialize(display, &major, &minor) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglInitialize");
        return false;
    }

    // 初めに見つかったコンフィギュレーションを使用する
    EGLint numConfigs;
    EGLConfig config;
    if (eglChooseConfig(display, attribs, &config, 1, &numConfigs) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglChooseConfig");
        return false;
    }

    // ANativeWindow_setBuffersGeometryで受け付けられるフォーマットを取得
    EGLint format;
    if (eglGetConfigAttrib(display, config, EGL_NATIVE_VISUAL_ID, &format) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglGetConfigAttrib");
        return false;
    }

    ANativeWindow_setBuffersGeometry(app->window, 0, 0, format);

    surface = eglCreateWindowSurface(display, config, app->window, NULL);
    if (surface == EGL_NO_SURFACE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglCreateWindowSurface");
        return false;
    }

    context = eglCreateContext(display, config, EGL_NO_CONTEXT, contextAttribs);
    if (context == EGL_NO_CONTEXT)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglCreateContext");
        return false;
    }

    if (eglMakeCurrent(display, surface, surface, context) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglMakeCurrent");
        return false;
    }

    EGLInt width;
    if (eglQuerySurface(display, surface, EGL_WIDTH, &width) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglQuerySurface for EGL_WIDTH");
        return false;
    }
    EGLInt height
    if (eglQuerySurface(display, surface, EGL_HEIGHT, &height) == EGL_FALSE)
    {
        NW_G3D_ASSERTMSG(false, "Failed to eglQuerySurface for EGL_HEIGHT");
        return false;
    }

    eglInfo.display = display;
    eglInfo.context = context;
    eglInfo.surface = surface;
    eglInfo.width = width;
    eglInfo.height = height;

    return true;
}

void DeleteGLContext()
{
    if (eglInfo.display != EGL_NO_DISPLAY) {
        eglMakeCurrent(eglInfo.display, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
        if (eglInfo.context != EGL_NO_CONTEXT) {
            eglDestroyContext(eglInfo.display, eglInfo.context);
        }
        if (eglInfo.surface != EGL_NO_SURFACE) {
            eglDestroySurface(eglInfo.display, eglInfo.surface);
        }
        eglTerminate(eglInfo.display);
    }
    eglInfo.display = EGL_NO_DISPLAY;
    eglInfo.context = EGL_NO_CONTEXT;
    eglInfo.surface = EGL_NO_SURFACE;
}

EGLDisplay GetEGLDisplay()
{
    return eglInfo.display;
}

void SwapBuffer()
{
    eglSwapBuffers(eglInfo.display, eglInfo.surface);
}

int32_t GetWidth()
{
    return eglInfo.width;
}

int32_t GetHeight()
{
    return eglInfo.height;
}

}}}} // namespace nw::g3d::fnd::detail

#endif
