﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/edit/g3d_EditServer.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/edit/g3d_IAllocator.h>
#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>
#include "detail/g3d_EditServerImpl.h"
#include "detail/g3d_EditManager.h"
#include "detail/g3d_EditRenderInfo.h"
#include "detail/g3d_EditPickup.h"
#include "detail/g3d_EditAnimControl.h"

namespace nw { namespace g3d { namespace edit {

using namespace nw::g3d::edit::detail;

EditServer EditServer::s_Instance;

/*static*/bool
EditServer::CreateInstance(const CreateArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL(arg.allocator);
    if (!s_Instance.InitializeImpl(arg))
    {
        s_Instance.DeleteInstance();
        return false;
    }

    return true;
}

bool
EditServer::InitializeImpl(const CreateArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL(arg.allocator);

    m_pAllocator = arg.allocator;

    {
        void* pBuffer = m_pAllocator->Alloc(sizeof(Impl), DEFAULT_ALIGNMENT);
        if (pBuffer == NULL)
        {
            return false;
        }

        m_pImpl = new (pBuffer) Impl();
        if (!m_pImpl->Initialize(arg))
        {
            return false;
        }
    }

    return true;
}

bool EditServer::IsInitialized()
{
    return Instance().m_pImpl != NULL;
}

EditServer::EditServer()
    : m_pImpl(NULL)
    , m_pAllocator(NULL)
{
}

/*static*/void
EditServer::DeleteInstance()
{
    if (Instance().m_pAllocator)
    {
        if (Instance().m_pImpl)
        {
            Instance().m_pImpl->~Impl();
            Instance().m_pAllocator->Free(Instance().m_pImpl);
            Instance().m_pImpl = NULL;
        }

        Instance().m_pAllocator = NULL;
    }
}

bool EditServer::IsConnected() const
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->IsConnected();
}

void EditServer::CalcAnimations()
{
    if (m_pImpl == NULL)
    {
        return;
    }

    if (m_pImpl->GetEditManager()->HasEditAnim() ||
        m_pImpl->GetEditManager()->HasEditSceneAnim())
    {
        m_pImpl->GetEditManager()->CalcAnimations();
    }
}

void EditServer::CalcSkeletalAnimations(const nw::g3d::ModelObj* modelObj)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    if (m_pImpl->GetEditManager()->HasEditAnim())
    {
        m_pImpl->GetEditManager()->CalcSkeletalAnimations(modelObj);
    }
}

bool EditServer::GetAnimFlag() const
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->GetAnimFlag();
}

bool EditServer::IsLoopAnim() const
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->IsLoopAnim();
}

float EditServer::GetFrame() const
{
    if (m_pImpl == NULL)
    {
        return 0.0f;
    }

    return m_pImpl->GetEditManager()->GetFrame();
}

float EditServer::GetFrameStep() const
{
    if (m_pImpl == NULL)
    {
        return 0.0f;
    }

    return m_pImpl->GetEditManager()->GetFrameStep();
}

float EditServer::GetPreviewFrameStepRate() const
{
    if (m_pImpl == NULL)
    {
        return 0.0f;
    }

    return m_pImpl->GetEditManager()->GetPreviewFrameStepRate();
}

float EditServer::GetFrameCount() const
{
    if (m_pImpl == NULL)
    {
        return 0.0f;
    }

    return m_pImpl->GetEditManager()->GetFrameCount();
}

int EditServer::GetModelAnimCount() const
{
    if (m_pImpl == NULL)
    {
        return 0;
    }

    return m_pImpl->GetEditManager()->GetEditAnimCount();
}

int EditServer::GetModelAnimBoundCount(const ModelObj* modelObj) const
{
    if (m_pImpl == NULL)
    {
        return 0;
    }

    return m_pImpl->GetEditManager()->GetModelAnimBoundCount(modelObj);
}

int EditServer::GetActiveModelAnimIndex(const ModelObj* modelObj, int boundAnimIndex) const
{
    if (m_pImpl == NULL)
    {
        return 0;
    }

    return m_pImpl->GetEditManager()->GetActiveEditAnimIndex(modelObj, boundAnimIndex);
}

float EditServer::GetModelAnimFrameCount(int modelAnimIndex) const
{
    if (m_pImpl == NULL)
    {
        return 0.0f;
    }

    return m_pImpl->GetEditManager()->GetEditAnimFrameCount(modelAnimIndex);
}

const char* EditServer::GetModelAnimName(int modelAnimindex) const
{
    if (m_pImpl == NULL)
    {
        return NULL;
    }

    return m_pImpl->GetEditManager()->GetEditAnimName(modelAnimindex);
}

EditAnimKind EditServer::GetModelAnimKind(int modelAnimIndex) const
{
    if (m_pImpl == NULL)
    {
        return static_cast<EditAnimKind>(-1);
    }

    return m_pImpl->GetEditManager()->GetEditAnimKind(modelAnimIndex);
}

void EditServer::PollDataCommunication()
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->PollDataCommunication();
}

void EditServer::PollDataEdit()
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->PollDataEdit();
    m_pImpl->GetEditManager()->UpdateBlink();
}

void
EditServer::Poll()
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->PollDataCommunication();

    m_pImpl->PollDataEdit();
    m_pImpl->GetEditManager()->UpdateBlink();
}

bool
EditServer::Open()
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    bool failed = !m_pImpl->GetCommandMonitor()->Open();
    failed |= !m_pImpl->GetPingMonitor()->Open();
    return !failed;
}

void
EditServer::Close()
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->GetCommandMonitor()->Close();
    m_pImpl->GetPingMonitor()->Close();
}

bool
EditServer::AttachModel(const AttachModelArg& arg)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(arg.modelObj);
    if (!IsConnected()) // 接続されていない場合は失敗
    {
        return false;
    }

    if (HasModelObj(arg.modelObj))
    {
        return false;
    }

    return m_pImpl->GetEditManager()->QueueAttachModel(arg.modelObj, arg.attachFileName);;
}

bool
EditServer::DetachModel(const DetachModelArg& arg)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(arg.modelObj);
    return m_pImpl->GetEditManager()->QueueDetachModel(arg.modelObj);
}

bool
EditServer::AttachShaderArchive(const AttachShaderArchiveArg& arg)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(arg.resShaderArchive);

    if (!IsConnected()) // 接続されていない場合は失敗
    {
        return false;
    }

    if (HasShaderArchive(arg.resShaderArchive))
    {
        return false;
    }

    // 強制バリエーションされていると .fsdb とブランチ情報が異なるため許可しない
    if (arg.resShaderArchive->ref().flag & ResShaderArchive::FORCE_VARIATION)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->QueueAttachShaderArchive(arg.resShaderArchive, arg.attachFileName);
}

bool
EditServer::DetachShaderArchive(const DetachShaderArchiveArg& arg)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(arg.resShaderArchive);
    return m_pImpl->GetEditManager()->QueueDetachShaderArchive(arg.resShaderArchive);
}

bool
EditServer::HasModelObj(const nw::g3d::ModelObj* modelObj) const
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->HasModelObj(modelObj);
}

bool
EditServer::HasShaderArchive(const nw::g3d::res::ResShaderArchive* resShaderArchive) const
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->HasShaderArchive(resShaderArchive);
}

bool
EditServer::PushRenderInfoChoice(const char* labelName, const char* itemName, const char* aliasItemName)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(itemName);
    NW_G3D_ASSERT_NOT_NULL(m_pImpl->GetEditRenderInfo());
    return m_pImpl->GetEditRenderInfo()->PushChoice(labelName, itemName, aliasItemName);
}

bool
EditServer::PushRenderInfoChoice(const char* labelName, s32 minValue, s32 maxValue)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(m_pImpl->GetEditRenderInfo());
    return m_pImpl->GetEditRenderInfo()->PushChoice(labelName, minValue, maxValue);
}

bool
EditServer::PushRenderInfoChoice(const char* labelName, f32 minValue, f32 maxValue)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(m_pImpl->GetEditRenderInfo());
    return m_pImpl->GetEditRenderInfo()->PushChoice(labelName, minValue, maxValue);
}

bool
EditServer::PushPickupMaterial(const ModelObj* modelObj, int materialIndex)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(modelObj);
    NW_G3D_ASSERTMSG(materialIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(modelObj->GetResource(), GetName()));
    return m_pImpl->GetEditPickup()->PushMaterialPickup(modelObj, materialIndex);
}

bool EditServer::PushRenderInfoDefault(const char* labelName, const char* value)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    NW_G3D_ASSERT_NOT_NULL(value);
    return m_pImpl->GetEditRenderInfo()->PushDefault(labelName, value);
}

bool EditServer::PushRenderInfoDefault(const char* labelName, s32 value)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    return m_pImpl->GetEditRenderInfo()->PushDefault(labelName, value);
}

bool EditServer::PushRenderInfoDefault(const char* labelName, f32 value)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    NW_G3D_ASSERT_NOT_NULL(labelName);
    return m_pImpl->GetEditRenderInfo()->PushDefault(labelName, value);
}

bool
EditServer::ClearPickupMaterial()
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditPickup()->PushMaterialPickup(NULL, -1);
}

bool
EditServer::PushModelLayout(const ModelLayoutArg& arg)
{
    if (m_pImpl == NULL)
    {
        return false;
    }

    return m_pImpl->GetEditManager()->AddModelLayoutQueue(reinterpret_cast<u32>(arg.modelObj), arg.scale, arg.rotate, arg.translate);
}

void
EditServer::Clear(bool isCallbackEnabled)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->GetEditManager()->ClearEditManager(isCallbackEnabled);
    m_pImpl->ClearFileBuffer();
    m_pImpl->GetCommandMonitor()->ClearWorkBuffer();
}

void
EditServer::SetAnimFlag(bool enable)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->GetEditManager()->SetAnimFlag(enable);

    m_pImpl->GetEditAnimControl()->SetAnimFlag(enable, m_pImpl->GetEditManager()->GetFrame());
    m_pImpl->GetEditAnimControl()->SendAnimControl(&m_pImpl->GetCommandMonitor()->GetSocket());
}

void
EditServer::SetFrame(float value)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->GetEditManager()->SetFrame(value);

    m_pImpl->GetEditAnimControl()->SetFrame(value);
    m_pImpl->GetEditAnimControl()->SendAnimControl(&m_pImpl->GetCommandMonitor()->GetSocket());
}

void
EditServer::SetFrameStep(float value)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    static const f32 TOOL_DELTA = 0.01f;
    if (value <= TOOL_DELTA)
    {
        value = TOOL_DELTA;
    }

    m_pImpl->GetEditManager()->SetFrameStep(value);

    m_pImpl->GetEditAnimControl()->SetFrameStep(value);
    m_pImpl->GetEditAnimControl()->SendAnimControl(&m_pImpl->GetCommandMonitor()->GetSocket());
}

void
EditServer::SetPreviewFrameStepRate(float value)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    m_pImpl->GetEditManager()->SetPreviewFrameStepRate(value);
}

void
EditServer::MoveNextModelAnim(const ModelObj* modelObj)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    NW_G3D_ASSERT_NOT_NULL(modelObj);

    m_pImpl->GetEditAnimControl()->SetNextModelAnim(modelObj);
    m_pImpl->GetEditAnimControl()->SendAnimControl(&m_pImpl->GetCommandMonitor()->GetSocket());
}

void
EditServer::MovePrevModelAnim(const ModelObj* modelObj)
{
    if (m_pImpl == NULL)
    {
        return;
    }

    NW_G3D_ASSERT_NOT_NULL(modelObj);

    m_pImpl->GetEditAnimControl()->SetPrevModelAnim(modelObj);
    m_pImpl->GetEditAnimControl()->SendAnimControl(&m_pImpl->GetCommandMonitor()->GetSocket());
}

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
