﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace ut {

namespace detail {

template <typename T>
class RingBuffer
{
public:
    typedef int (*CompareCallback)(const T* a, const T* b);

    explicit RingBuffer(T* pBuffer, int numMax)
        : m_Buffer(pBuffer)
        , m_Start(0)
        , m_Num(0)
        , m_NumMax(numMax)
    {
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        NW_G3D_ASSERT(numMax > 0);
    }

    void Clear()
    {
        m_Start = 0;
        m_Num = 0;
    }

    T* Get(int index)
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Num))
        {
            return &m_Buffer[GetIndex(index)];
        }
        return NULL;
    }

    const T* Get(int index) const
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_Num))
        {
            return &m_Buffer[GetIndex(index)];
        }
        return NULL;
    }

    T* UnsafeGet(int index)
    {
        return &m_Buffer[GetIndex(index)];
    }

    const T* UnsafeGet(int index) const
    {
        return &m_Buffer[GetIndex(index)];
    }

    bool PushBack(const T& value)
    {
        if (m_Num >= m_NumMax)
        {
            return false;
        }
        int index = m_Num;
        ++m_Num;
        m_Buffer[GetIndex(index)] = value;
        return true;
    }

    void PopBack()
    {
        if (m_Num > 0)
        {
            --m_Num;
        }
    }

    int Size() const
    {
        return m_Num;
    }
    int MaxSize() const
    {
        returnm_NumMax;
    }

    void EraseAt(int index)
    {
        NW_G3D_ASSERT(m_Num > 0 && index >= 0 && index < m_Num);

        for (int i = index; i < m_Num - 1; ++i)
        {
            m_Buffer[GetIndex(i)] = m_Buffer[GetIndex(i + 1)];
        }
        --m_Num;
    }

    T* Find(const T* ptr, CompareCallback compareCallback) const
    {
        return static_cast<T*>(FindImpl(ptr, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

    template<typename TKey>
    T* Find(const TKey* key, int (*compareCallback)(const T* a, const TKey* b)) const
    {
        return static_cast<T*>(FindImpl(key, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

private:
    int GetIndex(int i) const
    {
        int index = m_Start + i;
        if (index >= m_NumMax)
        {
            index -= m_NumMax;
        }
        return index;
    }

    typedef int (*CompareCallbackImpl)(const void* a, const void* b);
    void* FindImpl(const void* ptr, CompareCallbackImpl compareCallback) const
    {
        for (int i = 0; i < m_Num; ++i)
        {
            void* target = &m_Buffer[GetIndex(i)];
            if (compareCallback(target, ptr) == 0)
            {
                return target;
            }
        }
        return NULL;
    }
private:
    T*          m_Buffer;
    int         m_Start;
    int         m_Num;
    const int   m_NumMax;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(RingBuffer<T>);
};

} // namespace detail

}}}} // namespace nw::g3d::edit::ut

#endif // NW_G3D_CONFIG_USE_HOSTIO
