﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace ut {

namespace detail {

class DynamicPtrArrayImpl
{
public:
    void SetAllocator(nw::g3d::edit::IAllocator* allocator)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        m_pAllocator = allocator;
        m_PtrArray = static_cast<void**>(m_pAllocator->Alloc(sizeof(void*) * m_PtrCountMax, m_Alignment));
        NW_G3D_ASSERT_NOT_NULL(m_PtrArray);
    }

    int Size() const
    {
        return m_PtrCount;
    }

    void Clear()
    {
        m_PtrCount = 0;
    }

    void Destroy()
    {
        if (m_pAllocator == NULL)
        {
            return;
        }

        m_pAllocator->Free(m_PtrArray);
        m_PtrArray = NULL;
        m_PtrCount = 0;
        m_pAllocator = NULL;
    }

    void Erase(int pos)
    {
        Erase(pos, 1);
    }
protected:
    typedef int (*CompareCallbackImpl)(const void* a, const void* b);

    explicit DynamicPtrArrayImpl(nw::g3d::edit::IAllocator* allocator, size_t alignment)
        : m_pAllocator(allocator)
        , m_Alignment(alignment)
        , m_PtrCount(0)
        , m_PtrCountMax(DEFAULT_ARRAY_SIZE)
        , m_PtrArray(NULL)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        m_PtrArray = static_cast<void**>(allocator->Alloc(sizeof(void*) * m_PtrCountMax, alignment));
        NW_G3D_ASSERT_NOT_NULL(m_PtrArray);
    }

    explicit DynamicPtrArrayImpl(size_t alignment)
        : m_pAllocator(NULL)
        , m_Alignment(alignment)
        , m_PtrCount(0)
        , m_PtrCountMax(DEFAULT_ARRAY_SIZE)
        , m_PtrArray(NULL)
    {
    }

    ~DynamicPtrArrayImpl()
    {
        Destroy();
    }

    void Erase(int pos, int num)
    {
        NW_G3D_ASSERT(pos >= 0);
        NW_G3D_ASSERT(num >= 0);
        NW_G3D_ASSERT((pos + num) <= m_PtrCount);

        int movePos = pos + num;
        if (movePos < m_PtrCount)
        {
            CopyOverlap(&m_PtrArray[pos], &m_PtrArray[movePos], (m_PtrCount - movePos) * sizeof(void*));
        }
        m_PtrCount = m_PtrCount - num;
    }

    void* At(int index) const
    {
        if (static_cast<u32>(index) < static_cast<u32>(m_PtrCount))
        {
            return m_PtrArray[index];
        }
        else
        {
            NW_G3D_ASSERT_INDEX_BOUNDS(index, m_PtrCount);
            return NULL;
        }
    }

    void* UnsafeAt(int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_PtrCount);
        return m_PtrArray[index];
    }

    void* Front() const
    {
        return At(0);
    }

    void* Back() const
    {
        return At(m_PtrCount - 1);
    }

    bool PushBack(void* ptr)
    {
        if (m_PtrCount < m_PtrCountMax)
        {
            m_PtrArray[m_PtrCount] = ptr;
            ++m_PtrCount;
            return true;
        }

        if (Resize(m_PtrCountMax + 1))
        {
            m_PtrArray[m_PtrCount] = ptr;
            ++m_PtrCount;
            return true;
        }
        return false;
    }

    void* PopBack()
    {
        if (0 < m_PtrCount)
        {
            m_PtrCount--;
            return m_PtrArray[m_PtrCount];
        }
        else
        {
            return NULL;
        }
    }

    void* Find(const void* ptr, CompareCallbackImpl compareCallback) const
    {
        for (int i = 0; i < m_PtrCount; ++i)
        {
            if (compareCallback(m_PtrArray[i], ptr) == 0)
            {
                return m_PtrArray[i];
            }
        }
        return NULL;
    }

    int Search(const void* ptr, CompareCallbackImpl compareCallback) const
    {
        for (int i = 0; i < m_PtrCount; ++i)
        {
            if (compareCallback(m_PtrArray[i], ptr) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    int IndexOf(const void* ptr) const
    {
        for (int i = 0; i < m_PtrCount; ++i)
        {
            if (m_PtrArray[i] == ptr)
            {
                return i;
            }
        }
        return -1;
    }

    bool Resize(size_t size)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pAllocator);

        size_t currentSize = static_cast<size_t>(m_PtrCountMax);
        if (currentSize >= size)
        {
            return true;
        }

        int bufferCount = m_PtrCountMax << 2;

        void** newPtrArray = static_cast<void**>(m_pAllocator->Alloc(sizeof(void*) * bufferCount, m_Alignment));
        if (newPtrArray == NULL)
        {
            return false;
        }

        m_PtrCountMax = bufferCount;

        for (int i = 0; i < m_PtrCount; ++i)
        {
            newPtrArray[i] = m_PtrArray[i];
        }

        m_pAllocator->Free(m_PtrArray);
        m_PtrArray = newPtrArray;
        return true;
    }

protected:
    static const int DEFAULT_ARRAY_SIZE = 8;

private:
    void* CopyOverlap(void* dst, const void* src, u32 size)
    {
#if NW_G3D_IS_HOST_CAFE
        return OSBlockMove(dst, src, size, FALSE);
#else
        return std::memmove(dst, src, size);
#endif
    }

private:
    nw::g3d::edit::IAllocator* m_pAllocator;
    size_t m_Alignment;
    int m_PtrCount;
    int m_PtrCountMax;
    void** m_PtrArray;
};

template<typename T>
class DynamicPtrArray : public DynamicPtrArrayImpl
{
public:
    typedef int (*CompareCallback)(const T* a, const T* b);

    explicit DynamicPtrArray(nw::g3d::edit::IAllocator* allocator, size_t alignment)
        : DynamicPtrArrayImpl(allocator, alignment)
    {}

    explicit DynamicPtrArray(size_t alignment)
        : DynamicPtrArrayImpl(alignment)
    {}

    T* At(int index) const
    {
        return reinterpret_cast<T*>(DynamicPtrArrayImpl::At(index));
    }

    T* UnsafeAt(int index) const
    {
        return reinterpret_cast<T*>(DynamicPtrArrayImpl::UnsafeAt(index));
    }

    T* Front() const
    {
        return reinterpret_cast<T*>(DynamicPtrArrayImpl::Front());
    }

    T* Back() const
    {
        return reinterpret_cast<T*>(DynamicPtrArrayImpl::Back());
    }

    bool PushBack(T* ptr)
    {
        return DynamicPtrArrayImpl::PushBack(ptr);
    }

    T* PopBack()
    {
        return reinterpret_cast<T*>(DynamicPtrArrayImpl::PopBack());
    }

    T* Find(const T* ptr, CompareCallbackImpl compareCallback) const
    {
        return static_cast<T*>(DynamicPtrArrayImpl::Find(ptr, reinterpret_cast<CompareCallbackImpl>(compareCallback)));
    }

    template<typename TKey>
    T* Find(const TKey* key, int (*compareCallback)(const T* a, const TKey* b)) const
    {
        return (T*)DynamicPtrArrayImpl::Find(key, reinterpret_cast<CompareCallbackImpl>(compareCallback));
    }

    template<typename TKey>
    int Search(const TKey* key, int (*compareCallback)(const T* a, const TKey* b)) const
    {
        return DynamicPtrArrayImpl::Search(key, reinterpret_cast<CompareCallbackImpl>(compareCallback));
    }

    int IndexOf(const T* ptr) const
    {
        return DynamicPtrArrayImpl::IndexOf(static_cast<const void*>(ptr));
    }
};

} // namespace detail

}}}} // namespace nw::g3d::edit::ut

#endif // NW_G3D_CONFIG_USE_HOSTIO
