﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderParamAnimObj.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

class EditShaderParamAnimObj::EditShaderParamMatAnim
{
public:
    explicit EditShaderParamMatAnim(IAllocator* allocator, ResShaderParamMatAnim* resMatAnim)
        : m_pAllocator(allocator)
        , m_pResMatAnim(resMatAnim)
        , m_pEditAnimCurveArray(NULL)
        , m_EditAnimCurveCount(0)
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NW_G3D_RES_GET_NAME(m_pResMatAnim, GetName()));
        NW_G3D_ASSERT_NOT_NULL(resMatAnim);
    }

    bool Init()
    {
        int curveCount = m_pResMatAnim->GetCurveCount();
        size_t bufferSize = sizeof(EditAnimCurve) * curveCount;

        if (bufferSize > 0)
        {
            void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
            if (buffer == NULL)
            {
                return false;
            }
            m_pEditAnimCurveArray = static_cast<EditAnimCurve*>(buffer);
        }

        m_EditAnimCurveCount = curveCount;

        for(int i = 0; i < m_EditAnimCurveCount; ++i)
        {
            nw::g3d::res::ResAnimCurve* resAnimCurve = m_pResMatAnim->GetCurve(i);

            EditAnimCurve* editAnimCurve = new(&m_pEditAnimCurveArray[i]) EditAnimCurve(i, m_pAllocator, resAnimCurve);
            editAnimCurve->Setup();
        }
        return true;
    }

    void Destroy()
    {
        for(int i = 0; i < m_EditAnimCurveCount; ++i)
        {
            EditAnimCurve* editAnimCurve = &m_pEditAnimCurveArray[i];
            editAnimCurve->Cleanup();
        }

        if (m_pEditAnimCurveArray)
        {
            m_pAllocator->Free(m_pEditAnimCurveArray);
            m_pEditAnimCurveArray = NULL;
        }
    }

    EditAnimCurve* GetEditAnimCurve(int index)
    {
        return &m_pEditAnimCurveArray[index];
    }

    int GetEditAnimCurveCount() const
    {
        return m_EditAnimCurveCount;
    }

private:
    IAllocator*             m_pAllocator;
    ResShaderParamMatAnim*  m_pResMatAnim;
    EditAnimCurve* m_pEditAnimCurveArray;
    int m_EditAnimCurveCount;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(EditShaderParamMatAnim);
};

/*virtual*/void
EditShaderParamAnimObj::ResetToOriginalValue(ModelObj* pBoundModelObj)
{
    if (!IsModelBound(pBoundModelObj))
    {
        return;
    }

    int materialCount = pBoundModelObj->GetMaterialCount();
    for (int i = 0; i < materialCount; ++i)
    {
        MaterialObj* materialObj = pBoundModelObj->GetMaterial(i);
        materialObj->ClearShaderParam();
    }
}

/*virtual*/bool
EditShaderParamAnimObj::CreateDataForEditingAnimCurve()
{
    ResShaderParamAnim* pResAnim = static_cast<ResShaderParamAnim*>(GetResAnim());
    int matAnimCount = pResAnim->GetMatAnimCount();
    size_t bufferSize = sizeof(EditShaderParamAnimObj::EditShaderParamMatAnim) * matAnimCount;

    void* buffer = NULL;
    if (bufferSize > 0)
    {
        buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL)
        {
            return false;
        }
    }

    m_pMatAnimArray = static_cast<EditShaderParamAnimObj::EditShaderParamMatAnim*>(buffer);
    m_MatAnimCount = matAnimCount;

    bool isFailed = false;
    for(int i = 0; i < m_MatAnimCount; ++i)
    {
        nw::g3d::res::ResShaderParamMatAnim* resMatAnim = pResAnim->GetMatAnim(i);

        EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = new(&m_pMatAnimArray[i]) EditShaderParamAnimObj::EditShaderParamMatAnim(m_pAllocator, resMatAnim);
        isFailed = !editMatAnim->Init();
    }

    // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
    if (isFailed)
    {
        DestroyDataForEditingAnimCurve();
        return false;
    }

    return true;
}

/*virtual*/void
EditShaderParamAnimObj::DestroyDataForEditingAnimCurve()
{
    for(int i = 0; i < m_MatAnimCount; ++i)
    {
        EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = &m_pMatAnimArray[i];
        editMatAnim->Destroy();
    }

    m_MatAnimCount = 0;

    if (m_pMatAnimArray != NULL)
    {
        m_pAllocator->Free(m_pMatAnimArray);
        m_pMatAnimArray = NULL;
    }
}

/*virtual*/bool
EditShaderParamAnimObj::SetupInternal(ModelObj* pBindTargetModelObj, ModelAnimObj* pBoundAnimObj)
{
    BindResult result;

    ResModel* pResModel = pBindTargetModelObj->GetResource();
    ShaderParamAnimObj* pAnimObj = static_cast<ShaderParamAnimObj*>(pBoundAnimObj);
    ResShaderParamAnim* pResAnim = static_cast<ResShaderParamAnim*>(GetResAnim());

    ShaderParamAnimObj::InitArg initArg;
    initArg.DisableContext();
    initArg.Reserve(pBindTargetModelObj->GetResource());
    initArg.Reserve(pResAnim);

    size_t bufferSize = ShaderParamAnimObj::CalcBufferSize(initArg);
    void* buffer = m_pAllocator->Alloc(bufferSize, ShaderParamAnimObj::BUFFER_ALIGNMENT);
    NW_G3D_ASSERT_NOT_NULL_DETAIL(buffer, "%s\n", NW_G3D_RES_GET_NAME(pResModel, GetName()));//今は止める
    pAnimObj->Init(initArg, buffer, bufferSize);
    pAnimObj->SetResource(pResAnim);
    result = pAnimObj->Bind(pBindTargetModelObj);

    // 何もない場合もＯＫにするか後で考える。
    return true;
}

void
EditShaderParamAnimObj::EditCurve(
    int matAnimIndex,
    int curveIndex,
    const nw::g3d::res::ResAnimCurve* resAnimCurve,
    u32 resAnimCurveSize)
{
    ResTexPatternAnim* pResAnim = static_cast<ResTexPatternAnim*>(GetResAnim());
    NW_G3D_ASSERTMSG(curveIndex != -1, "%s\n", NW_G3D_RES_GET_NAME(pResAnim, GetName()));

    if (matAnimIndex >= m_MatAnimCount)
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(matAnimIndex, m_MatAnimCount);
        return;
    }

    EditShaderParamAnimObj::EditShaderParamMatAnim* editMatAnim = &m_pMatAnimArray[matAnimIndex];

    int curveCount = editMatAnim->GetEditAnimCurveCount();
    if (curveIndex >= curveCount)
    {
        NW_G3D_EDIT_WARNING_INDEX_BOUNDS(curveIndex, curveCount);
        return;
    }

    EditAnimCurve* editAnimCurve = editMatAnim->GetEditAnimCurve(curveIndex);
    editAnimCurve->Edit(resAnimCurve->ptr(), resAnimCurveSize);
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
