﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditHostFileIO.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/g3d_math.h>

#include <cstdio>

namespace {

const size_t FILE_DIV_SIZE = 4 * 1024 * 1024;

} // anonymous namespace

#if NW_G3D_IS_HOST_WIN

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <windows.h>
#include <string>

namespace {

bool ConvertFSPathToWinPath(char* dst, u32 dstSize, const char* src, u32 srcSize)
{
    if (dstSize <= (srcSize + 1) || dstSize < 3) // コンバート元の長さに1文字追加分と3文字以下の場合も失敗
    {
        return false;
    }

    const char* mountPath = src;
    if (mountPath[0] != '/') // 先頭文字は、/ でない場合は失敗
    {
        return false;
    }

    const char* exceptFirstSlashPath = src + 1;

    dst[0] = exceptFirstSlashPath[0];
    dst[1] = ':';
    dst[2] = '\\';

    char* exceptDrivePath = dst + 3;
    u32 copySize = srcSize - 3;
    strncpy_s(exceptDrivePath, MAX_PATH, exceptFirstSlashPath + 2, copySize);

    dst[copySize + 3] = '\0';

    u32 dstPos = 2;

    while (dstPos + 1 < dstSize)
    {
        if (dst[dstPos] == '/')
        {
            dst[dstPos] = '\\';
        }
        else if (dst[dstPos] == '\0')
        {
            break;
        }
        ++dstPos;
    }

    return true;
}

} // anonymous namespace

#endif

namespace nw { namespace g3d { namespace edit { namespace detail {

#if NW_G3D_IS_HOST_WIN

struct HostFileDevice::Impl
{
    HANDLE              handle;
    mutable OVERLAPPED  overlapped;
    DWORD               readingSize;
};

#endif

bool
HostFileDevice::Open(const char* fileName, OpenFlag flag)
{
#if NW_G3D_IS_HOST_WIN

    if (m_Impl->handle != INVALID_HANDLE_VALUE)
    {
        return false;
    }

    char winPathBuffer[MAX_PATH * 2];
    bool convertResult = ConvertFSPathToWinPath(winPathBuffer, MAX_PATH * 2, fileName, strlen(fileName));
    if (!convertResult)
    {
        return false;
    }

    DWORD access = 0;
    DWORD create = 0;
    switch(flag)
    {
    case READ_ONLY:
        {
            access = GENERIC_READ;
            create = OPEN_EXISTING;
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    HANDLE fileHandle = CreateFileA(winPathBuffer, access, FILE_SHARE_READ, NULL, create, FILE_FLAG_OVERLAPPED, NULL);
    if (fileHandle == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    m_Impl->handle = fileHandle;
    ZeroMemory(&m_Impl->overlapped, sizeof(OVERLAPPED));
    m_Impl->readingSize = 0;

    return true;

#else

    NW_G3D_ASSERT_NOT_NULL(m_ClientPtr);
    NW_G3D_EDIT_ASSERT(m_Handle < 0);
    (void)flag;

    // READ_ONLY でしかないので以下に設定
    const char* flagStr = "r";

    FSCmdBlock cmdBlock;
    FSInitCmdBlock(&cmdBlock);

    snprintf(m_FileFullPath, FS_MAX_FULLPATH_SIZE - 1, "%s%s", m_MountPath, fileName);
    m_FileFullPath[FS_MAX_FULLPATH_SIZE - 1] = '\0';
    FSStatus status = FSOpenFile(m_ClientPtr, &cmdBlock, m_FileFullPath, flagStr, &m_Handle, FS_RET_NOT_FOUND);
    if (status != FS_STATUS_OK)
    {
        NW_G3D_WARNING(0, "Failed FSOpenFile %s: Result code %d\n", fileName, status);
        return false;
    }

    m_ReadStatus = -1;

    return true;;

#endif
}

void
HostFileDevice::Close()
{
#if NW_G3D_IS_HOST_WIN

    bool success = CloseHandle(m_Impl->handle) == TRUE;
    NW_G3D_ASSERT(success);
    m_Impl->handle = INVALID_HANDLE_VALUE;

#else
    FSCmdBlock cmdBlock;
    FSInitCmdBlock(&cmdBlock);
    FSStatus status = FSCloseFile(m_ClientPtr, &cmdBlock, m_Handle, FS_RET_NO_ERROR);
    NW_G3D_ASSERT(status == FS_STATUS_OK);
    m_Handle = -1;
    m_ReadStatus = -1;
#endif
}

bool
HostFileDevice::ReadASync(void* buf, u32 size)
{
#if NW_G3D_IS_HOST_WIN

    if (m_Impl->handle == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    if (!ReadFile(m_Impl->handle, buf, size, &m_Impl->readingSize, &m_Impl->overlapped))
    {
        if (GetLastError() != ERROR_IO_PENDING)
        {
            return false;
        }
    }

    return true;

#else

    if (m_Handle < 0)
    {
        NW_G3D_WARNING(0, "Failed ReadASync\n");
        return false;
    }

    u8* bufferPtr = static_cast<u8*>(buf);
    u32 totalReadSize = 0;
    do
    {
        u32 bufferSize = size < FILE_DIV_SIZE ? size : FILE_DIV_SIZE;
        u32 tmpReadSize = 0;
        FSInitCmdBlock(&m_CmdBlock);
        FSStatus status = FSReadFile(m_ClientPtr, &m_CmdBlock, bufferPtr, 1, bufferSize, m_Handle, 0, FS_RET_NO_ERROR);
        if (status < FS_STATUS_OK)
        {
            return false;
        }
        tmpReadSize = status;
        totalReadSize += tmpReadSize;
        if (tmpReadSize < bufferSize)
        {
            break;
        }
        bufferPtr += tmpReadSize;
        size -= bufferSize;
    } while(size > 0);

    m_ReadStatus = totalReadSize;
    return m_ReadStatus > 0;

#endif
}

bool HostFileDevice::IsReading() const
{
#if NW_G3D_IS_HOST_WIN

    return !HasOverlappedIoCompleted(&m_Impl->overlapped) == TRUE;

#else

    if (m_Handle < 0)
    {
        return false;
    }
    return m_ReadStatus < 0;

#endif
}

bool HostFileDevice::IsValidHandle() const
{
#if NW_G3D_IS_HOST_WIN

    if (m_Impl->handle != INVALID_HANDLE_VALUE)
    {
        return true;
    }

#else

    if (m_Handle >= 0)
    {
        return true;
    }

#endif

    return false;
}

bool
HostFileDevice::Initialize(const InitArg& arg)
{
#if NW_G3D_IS_HOST_WIN
    (void)arg;
#else

    if (arg.hostFileIOHandle == NULL)
    {
        FSInit();
        FSStatus status = FSAddClient(&m_Client, FS_RET_NO_ERROR);
        if (status != FS_STATUS_OK)
        {
            NW_G3D_WARNING(false, "Failed FSAddClient.\n");
            return false;
        }

        FSStateChangeParams stateChangeParams;
        stateChangeParams.userCallback = StateChangeCallback;
        stateChangeParams.userContext = NULL;
        stateChangeParams.ioMsgQueue = NULL;
        FSSetStateChangeNotification(&m_Client, &stateChangeParams);

        FSCmdBlock cmdBlock;
        FSInitCmdBlock(&cmdBlock);
        FSMountSource mountSrc;
        status = FSGetMountSource(&m_Client, &cmdBlock, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);
        if (status != FS_STATUS_OK)
        {
            NW_G3D_WARNING(false, "Failed FSGetMountSource %d.\n", status);
            status = FSDelClient(&m_Client, FS_RET_NO_ERROR);
            NW_G3D_ASSERT(status == FS_STATUS_OK);
            return false;
        }

        status = FSMount(&m_Client, &cmdBlock, &mountSrc, m_MountPath, FS_MAX_MOUNTPATH_SIZE, FS_RET_NO_ERROR);
        if (status != FS_STATUS_OK)
        {
            NW_G3D_WARNING(false, "Failed FSMount %d.\n", status);
            status = FSDelClient(&m_Client, FS_RET_NO_ERROR);
            NW_G3D_ASSERT(status == FS_STATUS_OK);
            return false;
        }
        m_ClientPtr = &m_Client;
    }
    else
    {
        m_IsExternalClientHandle = true;
        m_ClientPtr = arg.hostFileIOHandle;
    }

#endif

    return true;
}

HostFileDevice::~HostFileDevice()
{
#if NW_G3D_IS_HOST_CAFE
    if (m_ClientPtr != NULL)
    {
        if (!m_IsExternalClientHandle)
        {
            FSCmdBlock cmdBlock;
            FSInitCmdBlock(&cmdBlock);
            FSStatus status = FSUnmount(m_ClientPtr, &cmdBlock, m_MountPath, FS_RET_NO_ERROR);
            NW_G3D_ASSERT(status == FS_STATUS_OK);

            status = FSDelClient(m_ClientPtr, FS_RET_NO_ERROR);
            NW_G3D_ASSERT(status == FS_STATUS_OK);
        }

        m_ClientPtr = NULL;
    }
#endif
}

HostFileDevice::HostFileDevice()
#if NW_G3D_IS_HOST_WIN
    : m_Impl(NULL)
#else
    : m_Handle(-1)
    , m_ClientPtr(NULL)
    , m_ReadStatus(-1)
    , m_IsExternalClientHandle(false)
#endif
{
#if NW_G3D_IS_HOST_WIN
    NW_G3D_ASSERT(sizeof(m_FileIOMemberBuffer) >= sizeof(Impl));
    memset(&m_FileIOMemberBuffer, 0, sizeof(m_FileIOMemberBuffer));
    m_Impl = reinterpret_cast<Impl*>(&m_FileIOMemberBuffer);
    m_Impl->handle = INVALID_HANDLE_VALUE;
#else
    memset(&m_Client, 0, sizeof(m_Client));
#endif
}

#if NW_G3D_IS_HOST_CAFE

/*static*/ void
HostFileDevice::ReadCallback(
    FSClient*       client,
    FSCmdBlock*     block,
    FSStatus        result,
    void*           context)
{
    HostFileDevice* device = static_cast<HostFileDevice*>(context);
    device->m_ReadStatus = result;
}

/*static*/void
HostFileDevice::StateChangeCallback(FSClient* client, FSVolumeState state, void* context)
{
    FSError lastError = FSGetLastError(client);
    NW_G3D_EDIT_PRINT("Volume state of client 0x%08x changed to %d, last error is %d\n", client, state, lastError);
}

#endif

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
