﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditCommandExecutor.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/edit/g3d_IAllocator.h>
#include <nw/g3d/g3d_fnd.h>
#include "ut/g3d_ScopedAllocator.h"
#include "g3d_EditManager.h"
#include "g3d_EditModelObj.h"
#include "g3d_EditShaderArchive.h"

using namespace nw::g3d;
using namespace nw::g3d::ut;
using namespace nw::g3d::edit;
using namespace nw::g3d::edit::detail;

EditCommandExecutor::EditCommandExecutor(IAllocator* pAllocator, EditCallback* pCallback, EditManager* pEditManager)
    : m_pAllocator(pAllocator)
    , m_pEditManager(pEditManager)
    , m_pCallback(pCallback)
{
}

void EditCommandExecutor::ExecuteEditRenderInfo(RenderInfoEditInfo* pInfo)
{
    const char* labelName = nw::g3d::ut::AddOffset<const char>(pInfo, pInfo->labelOffset - sizeof(PacketHeader));

    detail::EditModelObj* editModelObj =
        m_pEditManager->FindEditModelObj(pInfo->modelKey);
    bool isValid = true;
    if (editModelObj == NULL)
    {
        isValid = false;
    }

    if (isValid)
    {
        ModelObj* modelObj = reinterpret_cast<ModelObj*>(pInfo->modelKey);

        RenderInfoArg arg;
        arg.modelObj = modelObj;
        arg.materialIndex = pInfo->materialIndex;
        arg.name = labelName;

        MaterialObj* materialObj = modelObj->GetMaterial(arg.materialIndex);
        NW_G3D_ASSERT_NOT_NULL_DETAIL(materialObj, "%s\n", NW_G3D_RES_GET_NAME(materialObj->GetResource(), GetName()));

        nw::g3d::res::ResRenderInfo* resRenderInfo = materialObj->GetResource()->GetRenderInfo(arg.name);
        if (resRenderInfo)
        {
            NW_G3D_ASSERTMSG(resRenderInfo->GetType() == static_cast<ResRenderInfo::Type>(pInfo->type), NW_G3D_RES_GET_NAME(materialObj->GetResource(), GetName()));

            if (pInfo->slotIndex >= 0 && pInfo->slotIndex < resRenderInfo->GetArrayLength())
            {
                // 数値型はここで直接 ResRenderInfo を書き換えます。
                switch(static_cast<ResRenderInfo::Type>(pInfo->type))
                {
                case ResRenderInfo::INT:
                    resRenderInfo->ref().iValue[pInfo->slotIndex] = pInfo->iValue;
                    break;
                case ResRenderInfo::FLOAT:
                    resRenderInfo->ref().fValue[pInfo->slotIndex] = pInfo->fValue;
                    break;
                case ResRenderInfo::STRING:
                    break;
                default:
                    NW_G3D_EDIT_UNEXPECTED_DEFAULT;
                }
            }
        }

        if (m_pCallback)
        {
            m_pCallback->UpdateRenderInfo(arg);
        }
    }
}

void EditCommandExecutor::ExecuteUnloadFile(FileDataBlock* pBlock)
{
    switch(pBlock->kind)
    {
    case FILEDATA_MODEL:
        {
            NW_G3D_EDIT_PRINT("Unload Target Model Key: %d\n", pBlock->key);

            m_pEditManager->DeleteEditModelObj(pBlock->key);
            NW_G3D_ASSERT(!m_pEditManager->HasModelObj(GetModelObjFromKey(pBlock->key)));

            m_pEditManager->DeleteUnusedResFiles();
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
        {
            NW_G3D_EDIT_PRINT("Unload Target Anim Key: %d\n", pBlock->key);
            m_pEditManager->DeleteEditAnimObj(pBlock->key);
            m_pEditManager->DeleteAnimResFile(pBlock->key);
        }
        break;
    case FILEDATA_SHADER_ARCHIVE:
        {
            m_pEditManager->DeleteEditShaderArchive(pBlock->key);
            m_pEditManager->DeleteUnusedResFiles();
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            NW_G3D_EDIT_PRINT("Unload Target Scene Anim Key: %d\n", pBlock->key);
            m_pEditManager->DeleteEditSceneAnimObj(pBlock->key);
            m_pEditManager->DeleteUnusedResFiles();
        }
        break;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteEditMaterial(EditValueInfoBlock* pBlock)
{
    switch(pBlock->editTargetKind)
    {
    case EDIT_TARGET_MATERIAL_VISIBILITY:
    case EDIT_TARGET_MATERIAL_DISPLAYFACE:
    case EDIT_TARGET_MATERIAL_RENDER_STATE_MODE:
    case EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE:
    case EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC:
    case EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE:
    case EDIT_TARGET_MATERIAL_ALPHATEST_FUNC:
    case EDIT_TARGET_MATERIAL_ALPHATEST_VALUE:
    case EDIT_TARGET_MATERIAL_COLOR_COMBINE:
    case EDIT_TARGET_MATERIAL_ALPHA_COMBINE:
    case EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND:
    case EDIT_TARGET_MATERIAL_COLOR_DST_BLEND:
    case EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND:
    case EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND:
    case EDIT_TARGET_MATERIAL_LOGIC_OP:
        {
            EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;

            m_pEditManager->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_CONSTANT_COLOR:
        {
            EditVector4ValueBlock* valueBlock = reinterpret_cast<EditVector4ValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;

            m_pEditManager->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
        {
            EditSamplerValueBlock* valueBlock = reinterpret_cast<EditSamplerValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;

            m_pEditManager->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
        {
            EditShaderParamVectorValueBlock* valueBlock = reinterpret_cast<EditShaderParamVectorValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;

            m_pEditManager->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
        {
            EditShaderParamMatrixValueBlock* valueBlock = reinterpret_cast<EditShaderParamMatrixValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;

            m_pEditManager->EditMaterials(arg);
        }
        break;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteEditAnimationCurve(CommandFlag command, AnimCurveBlock* pBlock)
{
    EditAnimCurveArg arg;
    arg.animationKey = pBlock->info.animationKey;
    arg.animationKind = pBlock->info.animationKind;
    arg.curveData = NULL;
    arg.curveDataSize = 0;
    arg.value = 0;

    switch(arg.animationKind)
    {
    case EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_CAMERA_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_LIGHT_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_FOG_CURVE:
        {
            arg.value = &pBlock->value;
            arg.curveData = pBlock->curveData;
            arg.curveDataSize = pBlock->value.curveSize;
        }
        break;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
    switch(command)
    {
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        m_pEditManager->EditAnimCurve(arg);
        break;
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        m_pEditManager->EditSceneAnimCurve(arg);
        break;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteBindAnimation(CommandFlag command, BindAnimInfoBlock* block)
{
    BindAnimArg arg;
    arg.modelKey = block->info.modelKey;
    arg.animationKeys = block->animationKeys;
    arg.animationKeySize = block->info.animationKeySize;

    switch(command)
    {
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
        {
            m_pEditManager->BindAnimations(arg);

            NW_G3D_EDIT_DEBUG_PRINT("------------------------------------\nListing Bound Animations\n");
            for (int i = 0, end = arg.animationKeySize; i < end; ++i)
            {
                u32 resFileKey = arg.animationKeys[i];
                NW_G3D_EDIT_DEBUG_PRINT("key: %d bound at model %d\n", resFileKey, arg.modelKey);
                m_pEditManager->PrintEditAnimObjs(resFileKey, arg.modelKey);
            }
        }
        break;
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            NW_G3D_EDIT_DEBUG_PRINT("------------------------------------\nListing Unbind Target Animations\n");
            for (int i = 0, end = arg.animationKeySize; i < end; ++i)
            {
                u32 resFileKey = arg.animationKeys[i];
                NW_G3D_EDIT_DEBUG_PRINT("key: %d bound at model %d\n", resFileKey, arg.modelKey);
                m_pEditManager->PrintEditAnimObjs(resFileKey, arg.modelKey);
            }

            m_pEditManager->UnbindAnimations(arg);
        }
        break;
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
        {
            m_pEditManager->BindSceneAnimations(arg);
        }
        break;
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            m_pEditManager->UnbindSceneAnimations(arg);
        }
        break;
    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    NW_G3D_EDIT_DEBUG_PRINT("------------------------------------\nListing All Animations\n");
    m_pEditManager->PrintAllEditAnimObjs();

    NW_G3D_EDIT_DEBUG_PRINT("------------------------------------\nListing All Resource Files\n");
    m_pEditManager->PrintAllResFiles();
}

void EditCommandExecutor::ExecutePickup(CommandFlag command, EditValueInfoBlock* pBlock)
{
    EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(pBlock + 1);
    SelectTargetArg arg;
    arg.modelObj = reinterpret_cast<ModelObj*>(pBlock->key);

    switch (command)
    {
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
        {
            arg.targetKind = SelectTargetArg::TARGET_MODEL;
            arg.index = NULL;
            arg.indexSize = 0;
            m_pCallback->SelectTarget(arg);

            m_pEditManager->StartBlinking(arg);
        }
        break;

    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
        {
            arg.targetKind = SelectTargetArg::TARGET_MATERIAL;
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;
            m_pCallback->SelectTarget(arg);

            m_pEditManager->StartBlinking(arg);
        }
        break;

    case PICK_RUNTIME_BONE_COMMAND_FLAG:
        {
            arg.targetKind = SelectTargetArg::TARGET_BONE;
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;
            m_pCallback->SelectTarget(arg);
        }
        break;

    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        {
            arg.targetKind = SelectTargetArg::TARGET_SHAPE;
            arg.index = valueBlock->index;
            arg.indexSize = pBlock->indexSize;
            m_pCallback->SelectTarget(arg);

            m_pEditManager->StartBlinking(arg);
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteUserScript(UserScriptBlock* pBlock)
{
    if (m_pCallback == NULL)
    {
        return;
    }

    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pAllocator, "Internal error: Allocator must be set before receive commands");

    ExecuteModelUserScriptArg arg;
    ut::detail::ScopedAllocator commandTextAllocator(*m_pAllocator, pBlock->scriptTextLength + 1, DEFAULT_ALIGNMENT);
    ut::detail::ScopedAllocator targetNameAllocator(*m_pAllocator, pBlock->targetNameLength + 1, DEFAULT_ALIGNMENT);

    ut::detail::ScopedAllocator boneIndicesAllocator(*m_pAllocator, pBlock->selectedBoneCount * sizeof(int), DEFAULT_ALIGNMENT);
    ut::detail::ScopedAllocator shapeIndicesAllocator(*m_pAllocator, pBlock->selectedShapeCount * sizeof(int), DEFAULT_ALIGNMENT);
    ut::detail::ScopedAllocator materialIndicesAllocator(*m_pAllocator, pBlock->selectedMaterialCount * sizeof(int), DEFAULT_ALIGNMENT);

    s32* pInputSelectedBoneIndices = reinterpret_cast<s32*>(pBlock->dynamicData);
    s32* pInputSelectedShapeIndices = pInputSelectedBoneIndices + pBlock->selectedBoneCount;
    s32* pInputSelectedMaterialIndices = pInputSelectedShapeIndices + pBlock->selectedShapeCount;

    s8* pInputScriptTextData = reinterpret_cast<s8*>(pInputSelectedMaterialIndices + pBlock->selectedMaterialCount);
    s8* pInputTargetNameData = pInputScriptTextData + pBlock->scriptTextLength + 1;

    {
        char* scriptText = reinterpret_cast<char*>(commandTextAllocator.GetAllocatedBuffer());
        memset(scriptText, 0, commandTextAllocator.GetAllocatedSize());
        memcpy(scriptText, pInputScriptTextData, pBlock->scriptTextLength);
        arg.scriptText = scriptText;
    }

    // モデル以外に適用するケースが出てきたらargにtargetNameのようなメンバを追加して、targetName、targetKindを渡す
    {
        char* targetName = reinterpret_cast<char*>(targetNameAllocator.GetAllocatedBuffer());
        memset(targetName, 0, targetNameAllocator.GetAllocatedSize());
        memcpy(targetName, pInputTargetNameData, pBlock->targetNameLength);
    }

    {
        EditModelObj* pTargetEditModelObj = m_pEditManager->FindEditModelObj(pBlock->targetModelObjKey);
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pTargetEditModelObj, "ModelObj at 0x%08x is not attached\n", pBlock->targetModelObjKey);

        arg.targetModelObj = pTargetEditModelObj->GetTargetModelObj();
    }

    {
        int selectedBoneCount = pBlock->selectedBoneCount;
        int* selectedBoneIndices = reinterpret_cast<int*>(boneIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedBoneCount; ++index, ++pInputSelectedBoneIndices)
        {
            selectedBoneIndices[index] = *pInputSelectedBoneIndices;
        }

        arg.selectedBoneCount = selectedBoneCount;
        arg.selectedBoneIndices = selectedBoneIndices;
    }

    {
        int selectedShapeCount = pBlock->selectedShapeCount;
        int* selectedShapeIndices = reinterpret_cast<int*>(shapeIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedShapeCount; ++index, ++pInputSelectedShapeIndices)
        {
            selectedShapeIndices[index] = *pInputSelectedShapeIndices;
        }

        arg.selectedShapeCount = selectedShapeCount;
        arg.selectedShapeIndices = selectedShapeIndices;
    }

    {
        int selectedMaterialCount = pBlock->selectedMaterialCount;
        int* selectedMaterialIndices = reinterpret_cast<int*>(materialIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedMaterialCount; ++index, ++pInputSelectedMaterialIndices)
        {
            selectedMaterialIndices[index] = *pInputSelectedMaterialIndices;
        }

        arg.selectedMaterialCount = selectedMaterialCount;
        arg.selectedMaterialIndices = selectedMaterialIndices;
    }

    m_pCallback->ExecuteModelUserScript(arg);
}

#endif
