﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <nn/ndd/detail/ndd_Handler.h>
#include <nn/ndd/detail/ndd_InternalTypes.h>
#include <nn/ndd/detail/ndd_Queue.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/detail/ndd_Context.h>
#include <nn/ndd/detail/ndd_HandlerSubmodule.h>
#include <nn/ndd/detail/ndd_Packet.h>
#include <nn/ndd/detail/ndd_Receive.h>
#include <nn/ndd/detail/ndd_Scan.h>
#include <nn/ndd/detail/ndd_Wlan.h>
#include <nn/ndd/detail/ndd_Handle.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/ndd_PrivateResult.h>

namespace nn { namespace ndd { namespace handler {

namespace {

ContextContainer g_ContextContainer;
HandleProvider   g_HandleProvider;
StateConfig      g_StateConfig;
ScanConfig       g_ScanConfig;

Packet g_Packet;

//[todo]グローバルである必要は無い
NetworkRequest g_NetworkRequest;
PowerRequest   g_PowerRequest;
MultiWait      g_MultiWait;
}

bool IsNetworkActive()
{
    return g_StateConfig.IsNetworkEnabled();
}

bool IsAutoCommunicationEnabled()
{
    return g_StateConfig.IsAutoCommunicationEnabled();
}

int GetSendData(SendDataDescription* pBuffer, int offset, int count)
{
    if(g_StateConfig.IsSendDataExist())
    {
        *pBuffer = *g_StateConfig.GetSendDataPtr();
        return 1;
    }
    else
    {
        return 0;
    }
}

void HandleStateStepFromStop(State nextState)
{
    if(nextState == State_SleepStop)
    {
        //nifmがwlanのスリープ処理を行う
    }
    else if(nextState == State_Idle)
    {
        wlan::driver::Initialize();
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromIdle(State nextState)
{
    if(nextState == State_Stop)
    {
        wlan::driver::Finalize();
    }
    else if(nextState == State_SleepIdle)
    {
        wlan::driver::Sleep();
    }
    else if(nextState == State_Receive)
    {
        wlan::driver::EnableNetwork();
        wlan::driver::CreateReceiveEntry();
        wlan::receive::Start();
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromReceive(State nextState)
{
    if(nextState == State_Idle)
    {
        wlan::receive::Stop();
        wlan::driver::DeleteReceiveEntry();
        wlan::driver::DisableNetwork();
    }
    else if(nextState == State_SleepReceive)
    {
        //受信スリープはWLANの仕様に無いため、SleepIdleと同等とする
        wlan::receive::Stop();
        wlan::driver::DeleteReceiveEntry();
        wlan::driver::DisableNetwork();
        wlan::driver::Sleep();
    }
    else if(nextState == State_ReceiveSend)
    {
        g_Packet.SetFromSendDataDescription(*g_StateConfig.GetSendDataPtr());
        wlan::driver::EnableSend(g_Packet.GetPacketPtr(), g_Packet.GetPacketSize(), g_StateConfig.GetSendDataPtr()->dataId);
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromReceiveSend(State nextState)
{
    if(nextState == State_Receive)
    {
        wlan::driver::DisableSend();
    }
    else if(nextState == State_SleepReceiveSend)
    {
        wlan::driver::DisableSend();
        wlan::receive::Stop();
        wlan::driver::DisableNetwork();
        g_Packet.SetFromSendDataDescription(*g_StateConfig.GetSendDataPtr());
        wlan::driver::DetectiveSleep(g_Packet.GetPacketPtr(), g_Packet.GetPacketSize(), g_StateConfig.GetSendDataPtr()->dataId);
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromSleepStop(State nextState)
{
    if(nextState == State_Stop)
    {
        //nifmがwlanのアウェイク処理を行う
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromSleepIdle(State nextState)
{
    if(nextState == State_Idle)
    {
        wlan::driver::Awake();
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromSleepReceive(State nextState)
{
    if(nextState == State_Receive)
    {
        wlan::driver::Awake();
        wlan::driver::EnableNetwork();
        wlan::driver::CreateReceiveEntry();
        wlan::receive::Start();
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

void HandleStateStepFromSleepReceiveSend(State nextState)
{
    if(nextState == State_ReceiveSend)
    {
        wlan::driver::DetectiveAwake();
        wlan::driver::EnableNetwork();
        wlan::receive::Start();
        g_Packet.SetFromSendDataDescription(*g_StateConfig.GetSendDataPtr());
        wlan::driver::EnableSend(g_Packet.GetPacketPtr(), g_Packet.GetPacketSize(), g_StateConfig.GetSendDataPtr()->dataId);
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
}

//[todo]ステートチェックと動作決定の順を逆にすることを検討。ステートに影響を受ける機能の割合次第
void HandleStateTransition(State startState, State targetState)
{
    NN_NDD_LOG("HandleStateTransition: start = %d, target = %d\n", startState, targetState);
    State currentState;
    currentState = startState;

    //遷移しない
    if(startState == targetState)
    {
        if(startState == State_ReceiveSend)
        {
            wlan::driver::DisableSend();
            g_Packet.SetFromSendDataDescription(*g_StateConfig.GetSendDataPtr());
            wlan::driver::EnableSend(g_Packet.GetPacketPtr(), g_Packet.GetPacketSize(), g_StateConfig.GetSendDataPtr()->dataId);
            NN_NDD_LOG("same state. update SendData\n");
            return;
        }
        else
        {
            NN_NDD_LOG("same state\n");
            return;
        }
    }

    //遷移する
    for(;;)
    {
        State nextState;
        nextState = StateConfig::GetNext(currentState, targetState);
        switch(currentState)
        {
            case State_Stop:
                HandleStateStepFromStop(nextState);
                break;
            case State_Idle:
                HandleStateStepFromIdle(nextState);
                break;
            case State_Receive:
                HandleStateStepFromReceive(nextState);
                break;
            case State_ReceiveSend:
                HandleStateStepFromReceiveSend(nextState);
                break;
            case State_SleepStop:
                HandleStateStepFromSleepStop(nextState);
                break;
            case State_SleepIdle:
                HandleStateStepFromSleepIdle(nextState);
                break;
            case State_SleepReceive:
                HandleStateStepFromSleepReceive(nextState);
                break;
            case State_SleepReceiveSend:
                HandleStateStepFromSleepReceiveSend(nextState);
                break;
            default:
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
        }
        currentState = nextState;
        if(currentState == targetState)
        {
            break;
        }
    }
}

void HandleApi(const Message& message)
{
    NN_NDD_LOG("HandleApi: id = %d\n", message.GetId());
    switch(message.GetId())
    {
        case ApiId_Initialize:
        {
            const auto* pParam = reinterpret_cast<const ApiParam_Initialize*>(message.GetDataPtr());
            auto* pContext = pParam->pContext;
            auto handle = g_HandleProvider.Acquire();
            g_ContextContainer.Add(pContext, handle);
            g_ContextContainer.Reply(handle);
            break;
        }
        case ApiId_Finalize:
        {
            const auto* pParam = reinterpret_cast<const ApiParam_Finalize*>(message.GetDataPtr());
            auto handle = pParam->handle;
            g_ContextContainer.Remove(handle);
            g_HandleProvider.Delete(handle);
            break;
        }
        case ApiId_EnableAutoCommunication:
        {
            State currentState = g_StateConfig.Get();
            g_StateConfig.EnableAutoCommunication();
            HandleStateTransition(currentState, g_StateConfig.Get());
            break;
        }
        case ApiId_DisableAutoCommunication:
        {
            State currentState = g_StateConfig.Get();
            g_StateConfig.DisableAutoCommunication();
            HandleStateTransition(currentState, g_StateConfig.Get());
            break;
        }
        case ApiId_AddSendData:
        {
            State currentState = g_StateConfig.Get();
            const auto* pParam = reinterpret_cast<const ApiParam_AddSendData*>(message.GetDataPtr());
            g_StateConfig.AddSendData(pParam->sdd);
            g_ContextContainer.Signal(pParam->handle, Context::EventId_SendDataUpdate);
            //nddに登録できたタイミングでシグナルする。
            //nddはスリープ時にも送信データの再設定を行うため、上位層は上記シグナルを待つことで、スリープにおける送信データ更新を確実にできる。（フライトモード等は除く）
            HandleStateTransition(currentState, g_StateConfig.Get());
            break;
        }
        case ApiId_ClearSendData:
        {
            State currentState = g_StateConfig.Get();
            const auto* pParam = reinterpret_cast<const ApiParam_ClearSendData*>(message.GetDataPtr());
            g_StateConfig.ClearSendData();
            g_ContextContainer.Signal(pParam->handle, Context::EventId_SendDataUpdate);
            //nddに登録できたタイミングでシグナルする。
            //nddはスリープ時にも送信データの再設定を行うため、上位層は上記シグナルを待つことで、スリープにおける送信データ更新を確実にできる。（フライトモード等は除く）
            HandleStateTransition(currentState, g_StateConfig.Get());
            break;
        }
        case ApiId_AddReceiveData:
        {
            const auto* pParam = reinterpret_cast<const ApiParam_AddReceiveData*>(message.GetDataPtr());
            wlan::receive::Add(pParam->rdd);
            g_ContextContainer.Signal(Context::EventId_ReceiveData);
            break;
        }
        case ApiId_ClearReceiveData:
        {
            wlan::receive::Clear();
            break;
        }
        case ApiId_ClearDataIdFilter:
        {
            wlan::receive::ClearDataIdFilter();
            break;
        }
        case ApiId_StartDeviceScan:
        {
            const auto* pParam = reinterpret_cast<const ApiParam_StartDeviceScan*>(message.GetDataPtr());
            auto isStartTrigger = g_ScanConfig.Enable(pParam->handle);
            if(isStartTrigger)
            {
                wlan::receive::ClearDeviceScanResult();
                wlan::scan::Start();
            }
            else
            {
                //既に動作中なので何もしない
                //イベントシグナルは、動作中のScan完了時に行われる
            }
            break;
        }
        case ApiId_CancelDeviceScan:
        {
            const auto* pParam = reinterpret_cast<const ApiParam_CancelDeviceScan*>(message.GetDataPtr());
            auto isEnabled = g_ScanConfig.IsEnabled(pParam->handle);
            if(isEnabled)
            {
                wlan::scan::Cancel();
            }
            else
            {
                //動作していないため何もしない
            }
            break;
        }
        case ApiId_System_Received:
        {
            g_ContextContainer.Signal(Context::EventId_ReceiveData);
            break;
        }
        case ApiId_System_DeviceScanFinished:
        {
            NN_ABORT_UNLESS(g_ScanConfig.GetRequesterCount() > 0);
            auto handleCount = g_ScanConfig.GetRequesterCount();
            for(int i=0;i<handleCount;++i)
            {
                auto handle = g_ScanConfig.GetRequester(i);
                g_ContextContainer.Signal(handle, Context::EventId_DeviceScan);
            }
            g_ScanConfig.Disable();
            break;
        }
        default:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
        }
    }
}//NOLINT(impl/function_size)

void HandleNetworkRequest()
{
    NN_NDD_LOG("HandleNetworkRequest\n");
    g_NetworkRequest.ClearEvent();//MultiWaitのため明示クリア

    switch(g_NetworkRequest.GetState())
    {
        case nn::nifm::RequestState_Accepted:
        {
            NN_NDD_LOG("RequestState_Accepted\n");
            auto startState = g_StateConfig.Get();
            g_StateConfig.EnableNetwork();
            HandleStateTransition(startState, g_StateConfig.Get());
            break;
        }
        case nn::nifm::RequestState_Blocking:
        {
            NN_NDD_LOG("RequestState_Blocking\n");
            auto startState = g_StateConfig.Get();
            g_StateConfig.DisableNetwork();
            HandleStateTransition(startState, g_StateConfig.Get());
            g_NetworkRequest.Cancel();
            break;
        }
        case nn::nifm::RequestState_Free:
        {
            NN_NDD_LOG("RequestState_Free\n");
            g_NetworkRequest.Submit();
            break;
        }
        case nn::nifm::RequestState_OnHold:
        {
            NN_NDD_LOG("RequestState_OnHold\n");
            //何もしない
            break;
        }
        default:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
    }
    NN_NDD_LOG("\n\n");
}

void HandlePowerRequest()
{
    NN_NDD_LOG("HandlePowerRequest\n");
    g_PowerRequest.ClearEvent();//MultiWaitのため明示クリア
    auto state = g_PowerRequest.GetState();
    switch(state)
    {
        case nn::psc::PmState_FullAwake:
        {
            NN_NDD_LOG("PmState_FullAwake\n");
            //スリープ境界はminimumAwake-SleepReadyのため、何もしない
            g_PowerRequest.Acknowledge(state);
            break;
        }
        case nn::psc::PmState_MinimumAwake:
        {
            NN_NDD_LOG("PmState_MinimumAwake\n");
            auto startState = g_StateConfig.Get();
            g_StateConfig.Awake();
            HandleStateTransition(startState, g_StateConfig.Get());
            g_PowerRequest.Acknowledge(state);
            break;
        }
        case nn::psc::PmState_SleepReady:
        {
            NN_NDD_LOG("PmState_SleepReady\n");
            auto startState = g_StateConfig.Get();
            g_StateConfig.Sleep();
            HandleStateTransition(startState, g_StateConfig.Get());
            g_PowerRequest.Acknowledge(state);
            break;
        }
        case nn::psc::PmState_ShutdownReady:
        {
            NN_NDD_LOG("PmState_ShutdownReady\n");
            //[todo]fs系の処理
            g_PowerRequest.Acknowledge(state);
            break;
        }
        default:
        {
            NN_NDD_LOG("Ignorable PSC event.\n");//Essential系は無視する
            g_PowerRequest.Acknowledge(state);
            break;
        }
    }
    NN_NDD_LOG("\n\n");
}

void HandleQueue()
{
    NN_NDD_LOG("HandleQueue\n");
    static Message g_Message;
    queue::DeQueue(&g_Message);
    HandleApi(g_Message);
    NN_NDD_LOG("\n\n");
}

void InitializeWorker()
{
    wlan::receive::Initialize();
    wlan::scan::Initialize();

    g_NetworkRequest.Initialize();
    g_PowerRequest.Initialize();
    queue::Initialize();
}

MultiWaitId MultiWait()
{
    if(g_StateConfig.IsAwake())
    {
        g_MultiWait.Link(g_NetworkRequest.GetMultiWaitHolderPtr());
        g_MultiWait.Link(g_PowerRequest.GetMultiWaitHolderPtr());
        g_MultiWait.Link(queue::GetMultiWaitHolderPtr());
        auto multiWaitId = g_MultiWait.Wait();
        g_MultiWait.Unlink(g_NetworkRequest.GetMultiWaitHolderPtr());
        g_MultiWait.Unlink(g_PowerRequest.GetMultiWaitHolderPtr());
        g_MultiWait.Unlink(queue::GetMultiWaitHolderPtr());
        return multiWaitId;
    }
    else
    {
        g_MultiWait.Link(g_PowerRequest.GetMultiWaitHolderPtr());
        auto multiWaitId = g_MultiWait.Wait();
        g_MultiWait.Unlink(g_PowerRequest.GetMultiWaitHolderPtr());
        return multiWaitId;
    }
}

void Work(void* arg)
{
    for(;;)
    {
        switch(MultiWait())
        {
            case MultiWaitId_NetworkRequest:
            {
                HandleNetworkRequest();
                break;
            }
            case MultiWaitId_PowerRequest:
            {
                HandlePowerRequest();
                break;
            }
            case MultiWaitId_Queue:
            {
                HandleQueue();
                break;
            }
            default:
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ndd::ResultFaultyDesign());
        }
    }
}

}}}
