﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_SdkLog.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator

#include <nn/ndd/ndd_Result.h>
#include <nn/ndd/detail/Service/ndd_Impl.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/detail/ndd_InternalTypes.h>
#include <nn/ndd/detail/ndd_Handler.h>
#include <nn/ndd/detail/ndd_Receive.h>
#include <nn/ndd/detail/ndd_Utility.h>

namespace nn { namespace ndd {

namespace {
struct  NddServiceByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, NddServiceByHipcTag>   NddServiceAllocator;
} // namespace

// NddServiceAllocator を静的コンストラクタで初期化するためのヘルパー
class NddServiceAllocatorInitializer
{
public:

    NddServiceAllocatorInitializer() NN_NOEXCEPT
    {
        NddServiceAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_NddServiceAllocatorInitializer;


ServiceImpl::ServiceImpl() NN_NOEXCEPT
{
    m_Context.Initialize();

    ApiParam_Initialize param;
    param.pContext = &m_Context;
    m_Message.Set(ApiId_Initialize, &param, sizeof(param));
    queue::EnQueue(m_Message);

    m_Context.WaitReply();
}

ServiceImpl::~ServiceImpl() NN_NOEXCEPT
{
    ApiParam_Finalize param;
    param.handle = m_Context.GetHandle();
    m_Message.Set(ApiId_Finalize, &param, sizeof(param));
    queue::EnQueue(m_Message);
    m_Context.WaitReply();
    m_Context.Finalize();
}

nn::Result ServiceImpl::EnableAutoCommunication() NN_NOEXCEPT
{
    m_Message.Set(ApiId_EnableAutoCommunication);
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::DisableAutoCommunication() NN_NOEXCEPT
{
    m_Message.Set(ApiId_DisableAutoCommunication);
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::IsAutoCommunicationEnabled(nn::sf::Out<bool> pIsEnabled) NN_NOEXCEPT
{
    *pIsEnabled = nn::ndd::handler::IsAutoCommunicationEnabled();
    return ResultSuccess();
}

nn::Result ServiceImpl::IsNetworkActive(nn::sf::Out<bool> pIsActive) NN_NOEXCEPT
{
    *pIsActive = nn::ndd::handler::IsNetworkActive();
    return ResultSuccess();
}

nn::Result ServiceImpl::AcquireSendDataUpdateEvent(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    auto eventHandle = m_Context.AcquireHandle(Context::EventId_SendDataUpdate);
    handle.Set(nn::sf::NativeHandle(eventHandle, false));
    return ResultSuccess();
}

nn::Result ServiceImpl::AddSendData(const nn::ndd::SendDataDescription& data) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS_EQUAL(data.dataSize, SendDataSizeMax);
    ApiParam_AddSendData param;
    param.handle = m_Context.GetHandle();
    param.sdd = data;
    m_Message.Set(ApiId_AddSendData, &param, sizeof(param));
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::ClearSendData() NN_NOEXCEPT
{
    ApiParam_ClearSendData param;
    param.handle = m_Context.GetHandle();
    m_Message.Set(ApiId_ClearSendData, &param, sizeof(param));
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::GetSendData(nn::sf::Out<SendDataDescription> pBuffer, nn::sf::Out<std::int32_t> pCount, int offset, int count) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(count > 0);
    //[todo]コピー回数削減
    if(offset != 0)
    {
        *pCount = 0;
        return ResultSuccess();
    }

    SendDataDescription sdd;
    *pCount = nn::ndd::handler::GetSendData(&sdd, offset, count);
    *pBuffer = sdd;
    return ResultSuccess();
}

nn::Result ServiceImpl::AcquireReceiveDataEvent(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    auto eventHandle = m_Context.AcquireHandle(Context::EventId_ReceiveData);
    handle.Set(nn::sf::NativeHandle(eventHandle, false));
    return ResultSuccess();
}

nn::Result ServiceImpl::GetCurrentReceiveDataCounter(nn::sf::Out<std::uint32_t> pCounter) NN_NOEXCEPT
{
    *pCounter = nn::ndd::wlan::receive::GetCurrentCounter();
    return ResultSuccess();
}

nn::Result ServiceImpl::GetOldestReceiveDataCounter(nn::sf::Out<std::uint32_t> pCounter) NN_NOEXCEPT
{
    *pCounter = nn::ndd::wlan::receive::GetOldestCounter();
    return ResultSuccess();
}

nn::Result ServiceImpl::GetNextReceiveDataCounter(nn::sf::Out<std::uint32_t> pCounter) NN_NOEXCEPT
{
    *pCounter = nn::ndd::wlan::receive::GetNextCounter();
    return ResultSuccess();
}

nn::Result ServiceImpl::GetRecentReceiveDataCounter(nn::sf::Out<std::uint32_t> pCounter, std::uint32_t counter, int count) NN_NOEXCEPT
{
    *pCounter = nn::ndd::wlan::receive::GetRecentCounter(counter, count);
    return ResultSuccess();
}

nn::Result ServiceImpl::GetAvailableReceiveDataCount(nn::sf::Out<std::int32_t> pCount, std::uint32_t counter) NN_NOEXCEPT
{
    *pCount = nn::ndd::wlan::receive::GetAvailableCount(counter);
    return ResultSuccess();
}

nn::Result ServiceImpl::GetReceiveData(const nn::sf::OutArray<nn::ndd::ReceiveDataDescription>& pBuffer, nn::sf::Out<std::int32_t> pCount, nn::sf::Out<std::uint32_t> pNextCounter, std::uint32_t counter) NN_NOEXCEPT
{
    if(pBuffer.GetLength() == 0)
    {
        *pCount = 0;
        return ResultSuccess();
    }
    else
    {
        uint32_t nextCounter;
        *pCount = nn::ndd::wlan::receive::Get(pBuffer.GetData(), &nextCounter, pBuffer.GetLength(), counter);
        *pNextCounter = nextCounter;
        return ResultSuccess();
    }
}

nn::Result ServiceImpl::AddReceiveData(const nn::ndd::ReceiveDataDescription& data) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS_EQUAL(data.dataSize, ReceiveDataSizeMax);
    ApiParam_AddReceiveData param;
    param.rdd = data;
    m_Message.Set(ApiId_AddReceiveData, &param, sizeof(param));
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::ClearReceiveData() NN_NOEXCEPT
{
    m_Message.Set(ApiId_ClearReceiveData);
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::ClearDataIdFilter() NN_NOEXCEPT
{
    m_Message.Set(ApiId_ClearDataIdFilter);
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::AcquireDeviceScanEvent(nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
{
    auto eventHandle = m_Context.AcquireHandle(Context::EventId_DeviceScan);
    handle.Set(nn::sf::NativeHandle(eventHandle, false));
    return ResultSuccess();
}

nn::Result ServiceImpl::StartDeviceScan() NN_NOEXCEPT
{
    ApiParam_StartDeviceScan param;
    param.handle = m_Context.GetHandle();;
    m_Message.Set(ApiId_StartDeviceScan, &param, sizeof(param));
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::CancelDeviceScan() NN_NOEXCEPT
{
    ApiParam_CancelDeviceScan param;
    param.handle = m_Context.GetHandle();;
    m_Message.Set(ApiId_CancelDeviceScan, &param, sizeof(param));
    queue::EnQueue(m_Message);
    return ResultSuccess();
}

nn::Result ServiceImpl::GetDeviceScanResult(const nn::sf::OutArray<nn::ndd::ReceiveDataDescription>& pBuffer, nn::sf::Out<std::int32_t> pCount) NN_NOEXCEPT
{
    if(pBuffer.GetLength() == 0)
    {
        *pCount = 0;
        return ResultSuccess();
    }
    else
    {
        *pCount = nn::ndd::wlan::receive::GetDeviceScanResult(pBuffer.GetData(), pBuffer.GetLength());
        return ResultSuccess();
    }
}

nn::Result ServiceCreatorImpl::CreateNddService(nn::sf::Out<nn::sf::SharedPointer<nn::ndd::IService>> pOutService) NN_NOEXCEPT
{
    *pOutService = nn::sf::ObjectFactory<NddServiceAllocator::Policy>::CreateSharedEmplaced<nn::ndd::IService, nn::ndd::ServiceImpl>();
    return ResultSuccess();
}

}}  // namespace nn::ndd
