﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nd/service/nd_IService.sfdl.h>
#include <nn/nd/service/nd_ServiceNames.h>
#include <nn/nd/service/nd_ServiceResource.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>

namespace nn { namespace nd { namespace detail {

class ObjectHolder
{
public:
    static const int ProxyObjectCountMax = 4;

    explicit ObjectHolder(sf::ProxyObjectAllocator<ProxyObjectCountMax>& allocator) NN_NOEXCEPT;
    ~ObjectHolder() NN_NOEXCEPT;

    void Release() NN_NOEXCEPT;

    Result AcquireForApplicatoion() NN_NOEXCEPT;
    Result AcquireForSystem() NN_NOEXCEPT;

    service::IServiceBase& GetBase() const NN_NOEXCEPT;
    service::IServiceForApplication& GetForApplication() const NN_NOEXCEPT;
    service::IServiceForSystem& GetForSystem() const NN_NOEXCEPT;

private:
    sf::ProxyObjectAllocator<ProxyObjectCountMax>& m_ProxyObjectAllocator;
    service::IServiceBase* m_pBase;
    service::IServiceForApplication* m_pForApplication;
    service::IServiceForSystem* m_pForSystem;

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    static const int ClientSessionConcurrency = 1;
    sf::HipcSimpleClientSessionManager m_HipcDomain;
#endif

    template <typename Interface>
    Result AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT;
};

}}} // ~namespace nn::nd::detail

// 実装

namespace nn { namespace nd { namespace detail {

inline ObjectHolder::ObjectHolder(sf::ProxyObjectAllocator<ProxyObjectCountMax>& allocator) NN_NOEXCEPT
    : m_ProxyObjectAllocator(allocator)
    , m_pBase(nullptr)
    , m_pForApplication(nullptr)
    , m_pForSystem(nullptr)
{
    m_ProxyObjectAllocator.Initialize();
}

inline ObjectHolder::~ObjectHolder() NN_NOEXCEPT
{
    Release();
    m_ProxyObjectAllocator.Finalize();
}

inline void ObjectHolder::Release() NN_NOEXCEPT
{
    if( m_pBase != nullptr )
    {
        sf::ReleaseSharedObject(m_pBase);
        m_pBase = nullptr;
        m_pForApplication = nullptr;
        m_pForSystem = nullptr;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        m_HipcDomain.Finalize();
#endif
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)

service::ServiceResource& GetServiceResourceRef() NN_NOEXCEPT;

template <>
inline Result ObjectHolder::AcquireImpl(service::IServiceForSystem** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    NN_UNUSED(serviceName);
    *ppOutService = GetServiceResourceRef().GetSystemServicePointer().Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nd] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}

template <>
inline Result ObjectHolder::AcquireImpl(service::IServiceForApplication** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    NN_UNUSED(serviceName);
    *ppOutService = GetServiceResourceRef().GetApplicationServicePointer().Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nd] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}

#else

template <typename Interface>
inline Result ObjectHolder::AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<Interface> ptr;
    NN_RESULT_DO((m_HipcDomain.InitializeByName<Interface, sf::MemoryResourceAllocationPolicy>(&ptr, m_ProxyObjectAllocator.GetMemoryResource() ,serviceName)));
    NN_RESULT_DO(m_HipcDomain.SetSessionCount(ClientSessionConcurrency));
    *ppOutService = ptr.Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nd] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}

#endif

inline Result ObjectHolder::AcquireForApplicatoion() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    NN_RESULT_DO(AcquireImpl(&m_pForApplication, service::ServiceNameForApplication));
    m_pBase = m_pForApplication;
    NN_RESULT_SUCCESS;
}

inline Result ObjectHolder::AcquireForSystem() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    NN_RESULT_DO(AcquireImpl(&m_pForSystem, service::ServiceNameForSystem));
    m_pBase = m_pForSystem;
    NN_RESULT_SUCCESS;
}

inline service::IServiceBase& ObjectHolder::GetBase() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pBase);
    return *m_pBase;
}

inline service::IServiceForApplication& ObjectHolder::GetForApplication() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pForApplication);
    return *m_pForApplication;
}

inline service::IServiceForSystem& ObjectHolder::GetForSystem() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pForSystem);
    return *m_pForSystem;
}

}}} // ~namespace nn::nd::detail
