﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_Utility.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_MigrationList.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/result/result_HandlingUtility.h>

#include "../detail/migration_BufferedWriter.h"
#include "../detail/migration_FsUtil.h"

namespace nn { namespace migration { namespace user {

class SaveDataAggregationOperator
{
private:
    const account::Uid m_UserId;

    detail::BufferedWriter<detail::DataInfo> m_Writer {};
    size_t m_Count;

public:
    SaveDataAggregationOperator(
        const account::Uid& uid,
        const detail::AbstractStorage& storage, const char* path, size_t initialOffset,
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_UserId(uid)
        , m_Count(0)
    {
        m_Writer.Reset(storage, path, initialOffset, buffer, bufferSize);
    }
    size_t Complete() NN_NOEXCEPT
    {
        m_Writer.Flush();
        NN_ABORT_UNLESS_LESS_EQUAL(m_Count, UserMigrationSaveDataCountMax);
        return m_Count;
    }
    const fs::SaveDataSpaceId* GetTargetSpaceIds(int* pOut) const NN_NOEXCEPT
    {
        static const fs::SaveDataSpaceId TargetSpaceIds[] = {
            fs::SaveDataSpaceId::System,
            fs::SaveDataSpaceId::User,
        };

        *pOut = std::extent<decltype(TargetSpaceIds)>::value;
        return TargetSpaceIds;
    }
    bool Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT;
    Result Process(const fs::SaveDataInfo& info) NN_NOEXCEPT;
};

class SaveDataExistenceCheckOperator
{
private:
    const detail::DataInfo m_DataInfo;
    bool m_Found;

public:
    explicit SaveDataExistenceCheckOperator(const detail::DataInfo& dataInfo) NN_NOEXCEPT
        : m_DataInfo(dataInfo)
        , m_Found(false)
    {
        detail::PrintDataInfo(dataInfo);
    }
    bool Exists() const NN_NOEXCEPT
    {
        return m_Found;
    }
    const fs::SaveDataSpaceId* GetTargetSpaceIds(int* pOut) const NN_NOEXCEPT
    {
        *pOut = 1;
        return &m_DataInfo.spaceId;
    }
    bool Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT;
    Result Process(const fs::SaveDataInfo& info) NN_NOEXCEPT;
};

}}} // ~namespace migration::user
