﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "migration_UserSaveDataAggregationOperator.h"

#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>

namespace nn { namespace migration { namespace user {
namespace {
detail::DataInfo MakeDataInfoFromAccountSystemSaveDataInfo(const account::Uid& uid, const fs::SaveDataInfo& info) NN_NOEXCEPT
{
    NN_SDK_ASSERT(info.saveDataType == fs::SaveDataType::System);
    fs::ScopedAutoAbortDisabler antiAbort;

    detail::DataInfo data;
    data.type = detail::DataInfo::Type_System;
    data.spaceId = info.saveDataSpaceId;
    data.id = info.saveDataId;
    data.rawSize = fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
    data.codedSize = fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info) + fs::SaveDataExporter::InitialDataSize;
    data.attribute.system.Set(
        info.systemSaveDataId,
        uid);
    return data;
}

detail::DataInfo MakeDataInfoFromAccountSaveDataInfo(const account::Uid& uid, const fs::SaveDataInfo& info) NN_NOEXCEPT
{
    NN_SDK_ASSERT(info.saveDataType == fs::SaveDataType::Account);
    fs::ScopedAutoAbortDisabler antiAbort;

    detail::DataInfo data;
    data.type = detail::DataInfo::Type_User;
    data.spaceId = info.saveDataSpaceId;
    data.id = info.saveDataId;
    data.rawSize = fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
    data.codedSize = fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info) + fs::SaveDataExporter::InitialDataSize;
    data.attribute.user.Set(
        info.applicationId,
        ns::GetLaunchRequiredVersion(info.applicationId),
        uid);
    return data;
}
} // ~namespace nn::migration::user::<anonymous>

bool SaveDataAggregationOperator::Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT
{
    static const fs::SystemSaveDataId TargetSystemSaveDataIds[] = {
        /* System Processes */
        0x8000000000000080llu, // friends
        0x80000000000000f0llu, // pdm
        /* Library Applets */
        0x8000000000001020llu, // swkbd
        0x8000000000001061llu, // shop
        0x8000000000001071llu, // web
        0x8000000000001091llu, // loginShare
        0x80000000000010e1llu, // migration
    };

    switch (info.saveDataType)
    {
    case fs::SaveDataType::System:
        if (info.saveDataUserId == fs::ConvertAccountUidToFsUserId(m_UserId))
        {
            for (const auto& id : TargetSystemSaveDataIds)
            {
                if (info.systemSaveDataId == id)
                {
                    return true;
                }
            }
        }
        return false;
    case fs::SaveDataType::Account:
        return info.saveDataUserId == fs::ConvertAccountUidToFsUserId(m_UserId);
    default:
        // NOP
        break;
    }
    return false;
}

Result SaveDataAggregationOperator::Process(const fs::SaveDataInfo& info) NN_NOEXCEPT
{
    detail::PrintFsSaveDataInfo(info);

    switch (info.saveDataType)
    {
    case fs::SaveDataType::System:
        NN_RESULT_THROW_UNLESS(m_Count < UserMigrationSaveDataCountMax, ResultSaveDataCountExceedingLimit());
        m_Writer.Update(MakeDataInfoFromAccountSystemSaveDataInfo(m_UserId, info));
        ++ m_Count;
        break;
    case fs::SaveDataType::Account:
        NN_RESULT_THROW_UNLESS(m_Count < UserMigrationSaveDataCountMax, ResultSaveDataCountExceedingLimit());
        m_Writer.Update(MakeDataInfoFromAccountSaveDataInfo(m_UserId, info));
        ++ m_Count;
        break;
    default:
        break;
    }
    NN_RESULT_SUCCESS;
}

bool SaveDataExistenceCheckOperator::Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT
{
    switch (info.saveDataType)
    {
    case fs::SaveDataType::System:
        {
            fs::ScopedAutoAbortDisabler antiAbort;
            return true
                && m_DataInfo.type == detail::DataInfo::Type_System
                && m_DataInfo.spaceId == info.saveDataSpaceId
                && m_DataInfo.rawSize == fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info)
                && static_cast<size_t>(m_DataInfo.codedSize) == (fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info) + fs::SaveDataExporter::InitialDataSize)
                && m_DataInfo.attribute.system.systemSaveDataId == info.systemSaveDataId
                && fs::ConvertAccountUidToFsUserId(m_DataInfo.attribute.system.user) == info.saveDataUserId;
        }
    case fs::SaveDataType::Account:
        {
            fs::ScopedAutoAbortDisabler antiAbort;
            return true
                && m_DataInfo.type == detail::DataInfo::Type_User
                && m_DataInfo.spaceId == info.saveDataSpaceId
                && m_DataInfo.rawSize == fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info)
                && static_cast<size_t>(m_DataInfo.codedSize) == (fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info) + fs::SaveDataExporter::InitialDataSize)
                && m_DataInfo.attribute.user.applicationId == info.applicationId
                && fs::ConvertAccountUidToFsUserId(m_DataInfo.attribute.user.user) == info.saveDataUserId;
        }
    default:
        return false;
    }
}
Result SaveDataExistenceCheckOperator::Process(const fs::SaveDataInfo& info) NN_NOEXCEPT
{
    detail::PrintFsSaveDataInfo(info);
    m_Found = true;
    NN_RESULT_SUCCESS;
}

}}} // ~namespace migration::user
