﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/user/migration_UserMigrationContext.h>

#include <algorithm>

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/migration/user/migration_UserMigrationConfig.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_UuidApi.h>

#include "../detail/migration_SaveDataAggregator.h"
#include "migration_UserSaveDataAggregationOperator.h"

namespace nn { namespace migration { namespace user {

namespace {

bool TryRead(size_t* pOutActualSize, void* buffer, size_t bufferSize, const detail::AbstractStorage& storage, const char* path) NN_NOEXCEPT
{
    auto r = storage.Read(pOutActualSize, buffer, bufferSize, path);
    if (!r.IsSuccess())
    {
        if (!fs::ResultPathNotFound::Includes(r))
        {
            NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
        }
        return false;
    }
    return true;
}

uint32_t GetFirmwareVersion() NN_NOEXCEPT
{
    settings::system::FirmwareVersion fwVer;
    settings::system::GetFirmwareVersion(&fwVer);
    return static_cast<uint32_t>(fwVer.GetComparableVersion());
}

static const char UserMigrationInfoPath[] = "/migrationInfo.bin";
static const char MigrationListPath[] = "/migrationList.bin";
static const char TemporaryMigrationListPath[] = "/migrationList._bin";
static const char MigrationProgressPath[] = "/migrationProgress.bin";
static const char RelatedAppListPath[] = "/relatedAppList.bin";

Result CreateMigrationListImpl(
    detail::MigrationListHeader* pOut,
    const detail::AbstractStorage& storage, const char* path, const account::Uid& uid,
    void* buffer, size_t bufferSize, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    {
        auto lock = storage.AcquireWriteLock();
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Create(path, sizeof(detail::MigrationListHeader)));
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Commit());
    }

    detail::MigrationListHeader listHeader = {};
    listHeader.version = 0;
    {
        SaveDataAggregationOperator op(
            uid,
            storage, path, detail::MigrationList::GetDataInfoOffsetInBytes(listHeader, 0),
            buffer, bufferSize);
        NN_RESULT_DO(detail::AggregateSaveData(op, pCancellable));
        auto count = op.Complete();
        NN_ABORT_UNLESS_LESS_EQUAL(count, std::numeric_limits<decltype(listHeader.dataCount)>::max());
        listHeader.dataCount = static_cast<uint32_t>(count);
    }
    NN_SDK_ASSERT(detail::MigrationList::IsAccaptable(listHeader));

    size_t size;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.GetSize(&size, path));
    NN_SDK_ASSERT_EQUAL(size, detail::MigrationList::GetMigrationListSize(listHeader));
    NN_ABORT_UNLESS_LESS_EQUAL(size, UserMigrationListBytesMax);

    NN_MIGRATION_DETAIL_TRACE("[CreateMigrationListImpl] List: version=%d, count=%u, size=%zu\n", listHeader.version, listHeader.dataCount, size);

    auto lock = storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Append(path, 0, &listHeader, sizeof(listHeader)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Commit());

    *pOut = listHeader;
    NN_RESULT_SUCCESS;
}

size_t ReadMigrationListImpl(detail::DataInfo* pOut, size_t count, size_t offset, const detail::MigrationListHeader& header, const detail::AbstractStorage& storage) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(offset, header.dataCount);
    auto offsetInBytes = detail::MigrationList::GetDataInfoOffsetInBytes(header, offset);
    auto readSizeInBytes = sizeof(*pOut) * count;

    size_t readSize;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(storage.Read(&readSize, pOut, readSizeInBytes, MigrationListPath, offsetInBytes));
    NN_SDK_ASSERT(readSize % sizeof(*pOut) == 0);
    return readSize / sizeof(*pOut);
}

} // ~namespace nn::migration::user::<anonymous>

void ServerContext::CreateMigrationInfo(const account::Uid& uid, const UserMigrationServerProfile& serverProfile) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_pInfo);
    ServerInfo info = {
        GetFirmwareVersion(),
        util::GenerateUuid(),
        uid,
        serverProfile,
    };
    m_pInfo = info;
}
bool ServerContext::TryResume() NN_NOEXCEPT
{
    size_t actualSize;

    // ユーザー移行情報
    ServerInfo info;
    if (!TryRead(&actualSize, &info, sizeof(info), m_Storage, UserMigrationInfoPath))
    {
        NN_MIGRATION_DETAIL_ERROR("[ServerContext] Failed to load ServerInfo\n");
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(info));
    NN_SDK_ASSERT_NOT_EQUAL(info.sessionId, util::InvalidUuid);
    NN_SDK_ASSERT(info.user);

    if (!(info.fwVersion == GetFirmwareVersion()))
    {
        NN_MIGRATION_DETAIL_ERROR("[ServerContext] Firmware version mismatch: %08lx (expected: %08lx)\n", info.fwVersion, GetFirmwareVersion());
        return false;
    }

    // 移行リスト
    detail::MigrationListHeader listHeader;
    if (!TryRead(&actualSize, &listHeader, sizeof(listHeader), m_Storage, MigrationListPath))
    {
        NN_MIGRATION_DETAIL_ERROR("[ServerContext] Failed to load MigrationList\n");
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(listHeader));
    NN_SDK_ASSERT(detail::MigrationList::IsAccaptable(listHeader));
    NN_SDK_ASSERT_LESS(listHeader.dataCount, UserMigrationSaveDataCountMax);

    m_pInfo = info;
    m_pListHeader = listHeader;
    return true;
}

account::Uid ServerContext::GetUid() const NN_NOEXCEPT
{
    return m_pInfo->user;
}
void ServerContext::GetServerProfile(UserMigrationServerProfile* pOut) const NN_NOEXCEPT
{
    *pOut = m_pInfo->profile;
}
void ServerContext::GetServerInfoForTransfer(ServerInfoForTransfer* pOut) const NN_NOEXCEPT
{
    ServerInfoForTransfer info = {
        m_pInfo->fwVersion,
        m_pInfo->sessionId,
        m_pInfo->user,
        m_pInfo->profile,
    };
    *pOut = info;
}

Result ServerContext::ImportClientInfo(const ClientInfoForTransfer& clientInfo) NN_NOEXCEPT
{
    auto info = *m_pInfo;
    NN_SDK_ASSERT_NOT_EQUAL(info.sessionId, util::InvalidUuid);
    NN_SDK_ASSERT(info.user);
    info.clientInfo.profile = clientInfo.profile;
    m_pInfo = info;

    // 保存
    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Create(UserMigrationInfoPath, sizeof(info)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Write(UserMigrationInfoPath, &info, sizeof(info)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
bool ServerContext::MatchClientInfo(const ClientInfoForTransfer& clientInfo) const NN_NOEXCEPT
{
    NN_UNUSED(clientInfo);
    return true;
}
void ServerContext::GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT
{
    *pOut = m_pInfo->clientInfo.profile;
}

Result ServerContext::CreateMigrationList(void* buffer, size_t bufferSize, const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // 一時リストの作成
    detail::MigrationListHeader listHeader;
    NN_RESULT_DO(CreateMigrationListImpl(
        &listHeader, m_Storage, TemporaryMigrationListPath, m_pInfo->user,
        buffer, bufferSize, pCancellable));

    // リストの確定
    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Move(MigrationListPath, TemporaryMigrationListPath));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    m_pListHeader = listHeader;
    NN_RESULT_SUCCESS;
}
size_t ServerContext::GetMigrationListSize() const NN_NOEXCEPT
{
    return detail::MigrationList::GetMigrationListSize(*m_pListHeader);
}
size_t ServerContext::ReadMigrationList(detail::DataInfo* pOut, size_t count, size_t offset) const NN_NOEXCEPT
{
    return ReadMigrationListImpl(pOut, count, offset, *m_pListHeader, m_Storage);
}
size_t ServerContext::ReadMigrationListForExport(void* buffer, size_t bufferSize, size_t offset) const NN_NOEXCEPT
{
    size_t readSize;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Read(&readSize, buffer, bufferSize, MigrationListPath, offset));
    return readSize;
}

// ---------------------------------------------------------------------------------------------

TransferInfo ClientContext::CalculateTotalTransferInfo(void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<detail::DataInfo>::value, 0u);

    auto* list = reinterpret_cast<detail::DataInfo*>(buffer);
    const auto Capacity = bufferSize / sizeof(*list);
    NN_SDK_ASSERT(Capacity > 0);

    TransferInfo transferInfo = {};
    for (size_t offset = 0; offset < m_pListHeader->dataCount;)
    {
        auto readCount = ReadMigrationList(list, Capacity, offset);
        NN_SDK_ASSERT(readCount > 0);
        for (size_t i = 0; i < readCount; ++ i)
        {
            detail::PrintDataInfo(list[i]);

            ++ transferInfo.count;
            transferInfo.sizeInBytes += list[i].codedSize;
        }
        offset += readCount;
    }
    NN_MIGRATION_DETAIL_TRACE("[ClientContext] Total: count=%u, size=%zu\n", transferInfo.count, transferInfo.sizeInBytes);
    return transferInfo;
}
Result ClientContext::ResetProgressInfo(const detail::MigrationListHeader& listHeader, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    detail::SimpleMigrationProgress progress = {};
    progress.total = CalculateTotalTransferInfo(buffer, bufferSize);
    NN_RESULT_THROW_UNLESS(progress.total.count == listHeader.dataCount, detail::ResultInconsequentMigrationList());
    m_pProgress = progress;

    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Create(MigrationProgressPath, sizeof(progress)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Write(MigrationProgressPath, &progress, sizeof(progress)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}

void ClientContext::Initialize(void* relatedAppBuffer, size_t relatedAppBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(relatedAppBuffer);
    NN_SDK_ASSERT_GREATER_EQUAL(relatedAppBufferSize, sizeof(ncm::ApplicationId[UserMigrationRelatedApplicationCountMax]));
    m_RelatedApps = reinterpret_cast<ncm::ApplicationId*>(relatedAppBuffer);
    m_RelatedAppCapacity = std::min(relatedAppBufferSize / sizeof(ncm::ApplicationId), UserMigrationSaveDataCountMax);
    m_RelatedAppCount = 0u;
}

void ClientContext::CreateMigrationInfo(const UserMigrationClientProfile& clientProfile) NN_NOEXCEPT
{
    ClientInfo info = {{}, clientProfile};
    m_pInfo = info;
}
void ClientContext::SetNetworkServiceAccountId(const account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT
{
    auto info = *m_pInfo;
    NN_SDK_ASSERT_EQUAL(info.nsaId.id, 0x00ull);
    info.nsaId = nsaId;
    m_pInfo = info;
}
Result ClientContext::ImportServerInfo(const ServerInfoForTransfer& serverInfo) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(serverInfo.fwVersion == GetFirmwareVersion(), detail::ResultUnacceptableSystemVersion());
    NN_RESULT_THROW_UNLESS(serverInfo.sessionId != util::InvalidUuid, detail::ResultInvalidMigrationSessionId());
    NN_RESULT_THROW_UNLESS(serverInfo.user, detail::ResultInvalidImmigrantUid());

    char str[32];
    serverInfo.sessionId.ToString(str, sizeof(str));
    NN_MIGRATION_DETAIL_TRACE("[ClientContext] ServerInfo: sessionId=%s, user=%016llx_%016llx\n", str, serverInfo.user._data[0], serverInfo.user._data[1]);

    auto info = *m_pInfo;
    NN_SDK_ASSERT_NOT_EQUAL(info.nsaId.id, 0x00ull);
    NN_SDK_ASSERT_EQUAL(info.serverInfo.sessionId, util::InvalidUuid);
    NN_SDK_ASSERT(!info.serverInfo.user);
    info.serverInfo.fwVersion = serverInfo.fwVersion;
    info.serverInfo.sessionId = serverInfo.sessionId;
    info.serverInfo.user = serverInfo.user;
    info.serverInfo.profile = serverInfo.profile;
    m_pInfo = info;

    // 保存
    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Create(UserMigrationInfoPath, sizeof(info)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Write(UserMigrationInfoPath, &info, sizeof(info)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}
bool ClientContext::TryResume() NN_NOEXCEPT
{
    size_t actualSize;

    // ユーザー移行情報
    ClientInfo info;
    if (!TryRead(&actualSize, &info, sizeof(info), m_Storage, UserMigrationInfoPath))
    {
        NN_MIGRATION_DETAIL_ERROR("[ClientContext] Failed to load ClientInfo\n");
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(info));
    NN_SDK_ASSERT_NOT_EQUAL(info.nsaId.id, 0x00ull);
    NN_SDK_ASSERT_NOT_EQUAL(info.serverInfo.sessionId, util::InvalidUuid);
    NN_SDK_ASSERT(info.serverInfo.user);

    if (!(info.serverInfo.fwVersion == GetFirmwareVersion()))
    {
        NN_MIGRATION_DETAIL_ERROR("[ClientContext] Firmware version mismatch: %08lx (expected: %08lx)\n", info.serverInfo.fwVersion, GetFirmwareVersion());
        return false;
    }

    // 移行リスト
    detail::MigrationListHeader listHeader;
    if (!TryRead(&actualSize, &listHeader, sizeof(listHeader), m_Storage, MigrationListPath))
    {
        NN_MIGRATION_DETAIL_ERROR("[ClientContext] Failed to load MigrationList\n");
        return false;
    }
    // NOTE 移行リストのバージョンを上げる場合はここを変更する
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(listHeader));
    NN_SDK_ASSERT(detail::MigrationList::IsAccaptable(listHeader));
    NN_SDK_ASSERT_LESS(listHeader.dataCount, UserMigrationSaveDataCountMax);

    // 移行進捗
    detail::SimpleMigrationProgress progress;
    if (!TryRead(&actualSize, &progress, sizeof(progress), m_Storage, MigrationProgressPath))
    {
        NN_MIGRATION_DETAIL_ERROR("[ClientContext] Failed to load MigrationProgress\n");
        return false;
    }
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(progress));

    // Related Apps
    if (!TryRead(&actualSize, m_RelatedApps, m_RelatedAppCapacity * sizeof(ncm::ApplicationId), m_Storage, RelatedAppListPath))
    {
        actualSize = 0;
    }
    m_RelatedAppCount = actualSize / sizeof(ncm::ApplicationId);

    char str[32];
    info.serverInfo.sessionId.ToString(str, sizeof(str));
    NN_MIGRATION_DETAIL_TRACE("[ClientContext] Resumed\n");
    NN_MIGRATION_DETAIL_TRACE("[ClientContext]   - NSA ID: %016llx\n", info.nsaId.id);
    NN_MIGRATION_DETAIL_TRACE("[ClientContext]   - Session ID: %s\n", str);
    NN_MIGRATION_DETAIL_TRACE("[ClientContext]   - User: %016llx_%016llx\n", info.serverInfo.user._data[0], info.serverInfo.user._data[1]);
    NN_MIGRATION_DETAIL_TRACE("[ClientContext]   - List: version=%d, count=%u, size=%zu\n", listHeader.version, listHeader.dataCount, detail::MigrationList::GetMigrationListSize(listHeader));
    NN_MIGRATION_DETAIL_TRACE("[ClientContext]   - Total: count=%u, size=%zu\n", progress.total.count, progress.total.sizeInBytes);

    m_pInfo = info;
    m_pListHeader = listHeader;
    m_pProgress = progress;
    return true;
}
bool ClientContext::MatchServerInfo(const ServerInfoForTransfer& serverInfo) const NN_NOEXCEPT
{
    return m_pInfo->serverInfo.fwVersion == serverInfo.fwVersion
        && m_pInfo->serverInfo.sessionId == serverInfo.sessionId
        && m_pInfo->serverInfo.user == serverInfo.user;
}

account::NetworkServiceAccountId ClientContext::GetNetworkServiceAccountId() const NN_NOEXCEPT
{
    return m_pInfo->nsaId;
}
void ClientContext::GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT
{
    *pOut = m_pInfo->profile;
}
void ClientContext::GetClientInfoForTransfer(ClientInfoForTransfer* pOut) const NN_NOEXCEPT
{
    ClientInfoForTransfer info = {m_pInfo->profile};
    *pOut = info;
}
account::Uid ClientContext::GetUid() const NN_NOEXCEPT
{
    return m_pInfo->serverInfo.user;
}

Result ClientContext::BeginImportMigrationList(const detail::MigrationListHeader& listHeader) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(detail::MigrationList::IsAccaptable(listHeader), detail::ResultInvalidMigrationListHeader());
    NN_RESULT_THROW_UNLESS(listHeader.dataCount < UserMigrationSaveDataCountMax, detail::ResultInvalidMigrationListContentCount());

    NN_MIGRATION_DETAIL_TRACE("[ClientContext] List: version=%d, count=%u, size=%zu\n", listHeader.version, listHeader.dataCount, detail::MigrationList::GetMigrationListSize(listHeader));

    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Create(TemporaryMigrationListPath, detail::MigrationList::GetMigrationListSize(listHeader)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Append(TemporaryMigrationListPath, 0, &listHeader, sizeof(listHeader)));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());

    m_TemporaryMigrationListSize = detail::MigrationList::GetMigrationListSize(listHeader);
    m_TemporaryMigrationListOffset = sizeof(listHeader);
    NN_RESULT_SUCCESS;
}
Result ClientContext::UpdateImportMigrationList(const void* part, size_t partSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_TemporaryMigrationListOffset + partSize <= m_TemporaryMigrationListSize, detail::ResultInvalidMigrationListSize());

    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Append(TemporaryMigrationListPath, m_TemporaryMigrationListOffset, part, partSize));
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());

    m_TemporaryMigrationListOffset += partSize;
    NN_RESULT_SUCCESS;
}
Result ClientContext::EndImportMigrationList() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_TemporaryMigrationListOffset == m_TemporaryMigrationListSize, detail::ResultInvalidMigrationListSize());

    // リスト情報の取得
    size_t actualSize;
    detail::MigrationListHeader listHeader;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Read(&actualSize, &listHeader, sizeof(listHeader), TemporaryMigrationListPath));
    NN_ABORT_UNLESS_EQUAL(actualSize, sizeof(listHeader));

    // リストの確定
    {
        auto lock = m_Storage.AcquireWriteLock();
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Move(MigrationListPath, TemporaryMigrationListPath));
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    }
    m_pListHeader = listHeader;
    NN_RESULT_SUCCESS;
}
Result ClientContext::ResetProgressInfo(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return ResetProgressInfo(*m_pListHeader, buffer, bufferSize);
}

size_t ClientContext::ReadMigrationList(detail::DataInfo* pOut, size_t count, size_t offset) const NN_NOEXCEPT
{
    return ReadMigrationListImpl(pOut, count, offset, *m_pListHeader, m_Storage);
}

size_t ClientContext::CalculateCurrentShortfallOfUserSpace(void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(reinterpret_cast<uintptr_t>(buffer) % std::alignment_of<detail::DataInfo>::value, 0u);

    auto* list = reinterpret_cast<detail::DataInfo*>(buffer);
    const auto Capacity = bufferSize / sizeof(*list);
    NN_SDK_ASSERT(Capacity > 0);

    int64_t requiredSize = 0;
    for (size_t offset = m_pProgress->current.count; offset < m_pListHeader->dataCount;)
    {
        auto readCount = ReadMigrationList(list, Capacity, offset);
        NN_SDK_ASSERT(readCount > 0);
        for (size_t i = 0; i < readCount; ++ i)
        {
            const auto& data = list[i];
            if (data.spaceId == fs::SaveDataSpaceId::User)
            {
                requiredSize += data.rawSize;
            }
        }
        offset += readCount;
    }
    auto capacity = fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(fs::SaveDataSpaceId::User);

    NN_MIGRATION_DETAIL_TRACE("[Client] Storage: required=%lld, capacity=%lld\n", requiredSize, capacity);
    if (capacity >= requiredSize)
    {
        return 0u;
    }
    return static_cast<size_t>(requiredSize - capacity);
}

size_t ClientContext::GetCurrentRelatedApplications(ncm::ApplicationId* pOut, size_t count) const NN_NOEXCEPT
{
    auto copyCount = std::min(m_RelatedAppCount, count);
    if (copyCount > 0)
    {
        std::memcpy(pOut, m_RelatedApps, sizeof(ncm::ApplicationId) * copyCount);
    }
    return copyCount;
}

TransferInfo ClientContext::GetTotalTransferInfo() const NN_NOEXCEPT
{
    return m_pProgress->total;
}
TransferInfo ClientContext::GetCurrentTransferInfo() const NN_NOEXCEPT
{
    return m_pProgress->current;
}
void ClientContext::UpdateProgress(const detail::DataInfo& info) NN_NOEXCEPT
{
    // 進捗の更新
    auto progress = *m_pProgress;
    ++progress.current.count;
    progress.current.sizeInBytes += static_cast<uint64_t>(info.codedSize);
    m_pProgress = progress;

    // 起動必須バージョンと RelatedApps 更新
    bool relatedAppsUpdated = false;
    if (info.type == detail::DataInfo::Type_User)
    {
        auto appId = info.attribute.user.applicationId;
        auto appVer = info.attribute.user.applicationVersion;
        ns::UpgradeLaunchRequiredVersion(appId, appVer);

        if (m_RelatedAppCount < m_RelatedAppCapacity)
        {
            bool found = false;
            for (size_t i = 0; !found && i < m_RelatedAppCount; ++i)
            {
                found = (m_RelatedApps[i] == appId);
            }
            if (!found)
            {
                m_RelatedApps[m_RelatedAppCount ++] = appId;
                relatedAppsUpdated = true;
            }
        }
    }

    auto lock = m_Storage.AcquireWriteLock();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Write(MigrationProgressPath, &progress, sizeof(progress)));
    if (relatedAppsUpdated)
    {
        auto r = m_Storage.Create(RelatedAppListPath, UserMigrationSaveDataCountMax * sizeof(ncm::ApplicationId));
        if (!(r.IsSuccess() || fs::ResultPathAlreadyExists::Includes(r)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        }
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Write(RelatedAppListPath, m_RelatedApps, m_RelatedAppCount * sizeof(ncm::ApplicationId)));
    }
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
}

Result ClientContext::AdjustCurrentTransferInfo() NN_NOEXCEPT
{
    if (!(m_pProgress->current.count < m_pProgress->total.count))
    {
        NN_RESULT_SUCCESS;
    }

    detail::DataInfo info;
    auto readCount = ReadMigrationList(&info, 1, m_pProgress->current.count);
    NN_SDK_ASSERT_EQUAL(readCount, 1u);
    NN_UNUSED(readCount);

    SaveDataExistenceCheckOperator op(info);
    NN_RESULT_DO(detail::AggregateSaveData(op, nullptr));
    if (op.Exists())
    {
        // セーブデータがあれば進捗を進める
        NN_MIGRATION_DETAIL_TRACE("[ClientContext] CurrentTransferInfo is adjusted\n");
        UpdateProgress(info);
    }
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::user
