﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/nn_Abort.h>

#if defined(NN_BUILD_TARGET_PLATFORM_WIN)
#include <nn/crypto/crypto_Csrng.h>
#else
#include <nn/spl/spl_Api.h>
#endif

// #define ENABLE_TRANSFER_SPEED_MONITOR_LOG

namespace nn { namespace migration { namespace idc { namespace detail {

// TORIAEZU: memset ではコンパイラの最適化で消えるので独自に定義
//           memset_s が欲しい。
void SecureMemoryZero(Bit8* addr, size_t size) NN_NOEXCEPT
{
    auto ptr = static_cast<volatile Bit8*>(addr);
    for( size_t i = 0; i < size; ++i )
    {
        ptr[i] = 0;
    }
}

void GenerateRandomBytes(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
#if defined(NN_BUILD_TARGET_PLATFORM_WIN)
    crypto::GenerateCryptographicallyRandomBytes(buffer, bufferSize);
#else
    NN_ABORT_UNLESS_RESULT_SUCCESS(spl::GenerateRandomBytes(buffer, bufferSize));
#endif
}

#if defined(NN_SDK_BUILD_DEVELOP) || defined(NN_SDK_BUILD_DEBUG)
const char* GetCommandKindString(CommandKind command) NN_NOEXCEPT
{
    switch( command )
    {
    case Initiate0: return "Initiate0";
    case Initiate1: return "Initiate1";
    case Resume0:   return "Resume0";
    case Resume1:   return "Resume1";
    case Terminate: return "Terminate";
    case User:      return "User";
    case Error:     return "Error";
    default:        return "(unknown)";
    }
}
#endif

TransferSpeedMonitor::TransferSpeedMonitor() NN_NOEXCEPT
    : m_IsSet(false)
    , m_IsPaused(true)
    , m_BytePerSecLimit(0)
    , m_TrackDuration(0)
    , m_BelowLimitDuration(0)
    , m_LastUpdated(0)
    , m_CachedDuration(0)
    , m_CachedTransferredByteCount(0)
{
}

TransferSpeedMonitor::TransferSpeedMonitor(size_t bytesPerSec, TimeSpan timeSpan) NN_NOEXCEPT
    : m_IsSet(true)
    , m_IsPaused(true)
    , m_BytePerSecLimit(bytesPerSec)
    , m_TrackDuration(timeSpan)
    , m_BelowLimitDuration(0)
    , m_LastUpdated(0)
    , m_CachedDuration(0)
    , m_CachedTransferredByteCount(0)
{
    NN_SDK_REQUIRES_GREATER(bytesPerSec, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeSpan, TimeSpan::FromSeconds(1));
}

void TransferSpeedMonitor::SetLimit(size_t bytesPerSec, TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(bytesPerSec, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeSpan, TimeSpan::FromSeconds(1));
    NN_SDK_ASSERT(!m_IsSet);
    m_BytePerSecLimit = bytesPerSec;
    m_TrackDuration = timeSpan;
    m_IsPaused = true;
}

void TransferSpeedMonitor::Reset() NN_NOEXCEPT
{
    m_IsSet = false;
    m_BelowLimitDuration = nn::TimeSpan(0);
    m_CachedDuration = nn::TimeSpan(0);
    m_CachedTransferredByteCount = 0u;
    m_LastUpdated = os::GetSystemTick();
}

void TransferSpeedMonitor::Pause() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsSet);
    NN_SDK_ASSERT(!m_IsPaused);
    m_CachedDuration += (os::GetSystemTick() - m_LastUpdated).ToTimeSpan();
    m_IsPaused = true;
}

void TransferSpeedMonitor::Resume() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsSet);
    NN_SDK_ASSERT(m_IsPaused);
    m_IsPaused = false;
    m_LastUpdated = os::GetSystemTick();
}

void TransferSpeedMonitor::Update(size_t transferredByteCount) NN_NOEXCEPT
{
    if( !(m_IsSet && !m_IsPaused) )
    {
        return;
    }

    auto elapsed = (os::GetSystemTick() - m_LastUpdated).ToTimeSpan();
    m_LastUpdated = os::GetSystemTick();
    m_CachedDuration += elapsed;
    m_CachedTransferredByteCount += transferredByteCount;

    if( !(m_CachedDuration.GetSeconds() > 0) )
    {
        return;
    }

    auto bytesPerSec = static_cast<double>(m_CachedTransferredByteCount) * 1000 / m_CachedDuration.GetMilliSeconds();
    if( bytesPerSec < m_BytePerSecLimit )
    {
#if defined(ENABLE_TRANSFER_SPEED_MONITOR_LOG)
        NN_DETAIL_MIGRATION_TRACE_V1("[TransferSpeedMonitor::Update] %.2f bytes/sec (< %zu). BelowLimitDuration is now %lld ms (+%lld ms)\n",
           bytesPerSec, m_BytePerSecLimit, (m_BelowLimitDuration + m_CachedDuration).GetMilliSeconds(), m_CachedDuration.GetMilliSeconds());
#endif
        m_BelowLimitDuration += m_CachedDuration;
    }
    else
    {
#if defined(ENABLE_TRANSFER_SPEED_MONITOR_LOG)
        NN_DETAIL_MIGRATION_TRACE_V1("[TransferSpeedMonitor::Update] %.2f bytes/sec (>= %zu). Reset BelowLimitDuration\n", bytesPerSec, m_BytePerSecLimit);
#endif
        m_BelowLimitDuration = nn::TimeSpan();
    }
    m_CachedTransferredByteCount = 0u;
    m_CachedDuration = nn::TimeSpan(0);
}

bool TransferSpeedMonitor::IsLowLimitDetected() const NN_NOEXCEPT
{
    return m_BelowLimitDuration > m_TrackDuration;
}

}}}}
