﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Settings.h>

#include <nn/nn_Abort.h>
#include <nn/migration/detail/migration_Mutex.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace migration { namespace detail {

namespace {
const char SettingName[] = "migration";
bool g_Initialized = false;
Mutex g_Lock = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);

#define DECLARE_LABEL(name) \
    static const char name[]
#define DEFINE_LABEL(name, str) \
    const char Settings::Label::name[] = str
#define DECLARE_VALUE(name) \
    static int g_ ## name
#define DEFINE_VALUE(name, defaultValue) \
    int Settings::g_ ## name = defaultValue

#define LABEL_FULLNAME(name) \
    Settings::Label::name
#define VALUE_FULLNAME(name) \
    Settings::g_ ## name

struct Settings
{
    struct Label
    {
        DECLARE_LABEL(LdnConnectionTimeoutSeconds);
        DECLARE_LABEL(TransferLowSpeedTimeoutSeconds);
        DECLARE_LABEL(TransferMessagingTimeoutSeconds);
        DECLARE_LABEL(TransferSmallDataTimeoutSeconds);
        DECLARE_LABEL(TransferLargeDataTimeoutSeconds);
        DECLARE_LABEL(UiAcceptanceWaitTimeoutSeconds);
    };

    DECLARE_VALUE(LdnConnectionTimeoutSeconds);
    DECLARE_VALUE(TransferLowSpeedTimeoutSeconds);
    DECLARE_VALUE(TransferMessagingTimeoutSeconds);
    DECLARE_VALUE(TransferSmallDataTimeoutSeconds);
    DECLARE_VALUE(TransferLargeDataTimeoutSeconds);
    DECLARE_VALUE(UiAcceptanceWaitTimeoutSeconds);
};

DEFINE_LABEL(LdnConnectionTimeoutSeconds,       "ldn_connection_timeout_seconds");
DEFINE_LABEL(TransferLowSpeedTimeoutSeconds,    "transfer_low_speed_timeout_seconds");
DEFINE_LABEL(TransferMessagingTimeoutSeconds,   "transfer_messaging_timeout_seconds");
DEFINE_LABEL(TransferSmallDataTimeoutSeconds,   "transfer_small_data_timeout_seconds");
DEFINE_LABEL(TransferLargeDataTimeoutSeconds,   "transfer_large_data_timeout_seconds");
DEFINE_LABEL(UiAcceptanceWaitTimeoutSeconds,    "ui_acceptance_wait_timeout_seconds");

DEFINE_VALUE(LdnConnectionTimeoutSeconds, 1800);
DEFINE_VALUE(TransferLowSpeedTimeoutSeconds, 300);
DEFINE_VALUE(TransferMessagingTimeoutSeconds, 100);
DEFINE_VALUE(TransferSmallDataTimeoutSeconds, 300);
DEFINE_VALUE(TransferLargeDataTimeoutSeconds, 3600);
DEFINE_VALUE(UiAcceptanceWaitTimeoutSeconds, 900);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
inline int LoadSettingValue(const char* key) NN_NOEXCEPT
{
    int32_t value;
    auto s = settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), SettingName, key);
    NN_ABORT_UNLESS(s == sizeof(value));
    return static_cast<int>(value);
}

#define LOAD_VALUE(name) \
    VALUE_FULLNAME(name) = LoadSettingValue(LABEL_FULLNAME(name))

#elif defined(NN_BUILD_CONFIG_OS_WIN)
#define LOAD_VALUE(...)

#else
#error "Unsupported OS specified"
#endif

void Refresh() NN_NOEXCEPT
{
    LockGuard lock(g_Lock);
    if (g_Initialized)
    {
        return;
    }

    LOAD_VALUE(LdnConnectionTimeoutSeconds);
    LOAD_VALUE(TransferLowSpeedTimeoutSeconds);
    LOAD_VALUE(TransferMessagingTimeoutSeconds);
    LOAD_VALUE(TransferSmallDataTimeoutSeconds);
    LOAD_VALUE(TransferLargeDataTimeoutSeconds);
    LOAD_VALUE(UiAcceptanceWaitTimeoutSeconds);

    g_Initialized = true;
}
} // ~namespace nn::migration::detail::<anonymous>

#define DEFINE_VALUE_GETTER(name) \
    int Get ## name() NN_NOEXCEPT \
    { \
        Refresh(); \
        return VALUE_FULLNAME(name); \
    }

DEFINE_VALUE_GETTER(LdnConnectionTimeoutSeconds)
DEFINE_VALUE_GETTER(TransferLowSpeedTimeoutSeconds)
DEFINE_VALUE_GETTER(TransferMessagingTimeoutSeconds)
DEFINE_VALUE_GETTER(TransferSmallDataTimeoutSeconds)
DEFINE_VALUE_GETTER(TransferLargeDataTimeoutSeconds)
DEFINE_VALUE_GETTER(UiAcceptanceWaitTimeoutSeconds)

}}} // ~namesapce nn::migration::detail
