﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_SaveDataTransfer.h>

#include <algorithm>
#include <nn/fs/fs_Context.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace detail {

// ---------------------------------------------------------------------------------------------
// SaveDataExporter

SaveDataExporter::SaveDataExporter(IdcServerBase& server, ResourceStorage& workBuffer) NN_NOEXCEPT
    : m_Server(server)
    , m_pResource(reinterpret_cast<Resource*>(&workBuffer))
{
}

Result SaveDataExporter::Setup(const DataInfo& dataInfo) NN_NOEXCEPT
{
    fs::ScopedAutoAbortDisabler antiAbort;
    NN_RESULT_DO(m_Server.OpenSaveDataExporter(&m_pImpl, dataInfo));
    m_DataInfo = dataInfo;
    m_Cursor = 0u;

    NN_RESULT_DO(m_pImpl->PullInitialData(m_pResource->initialData, sizeof(m_pResource->initialData)));
    NN_RESULT_SUCCESS;
}

size_t SaveDataExporter::Export(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pImpl);
    if (m_Cursor < fs::SaveDataExporter::InitialDataSize)
    {
        auto copySize = std::min(bufferSize, static_cast<size_t>(fs::SaveDataExporter::InitialDataSize - m_Cursor));
        std::memcpy(buffer, m_pResource->initialData + m_Cursor, copySize);
        m_Cursor += copySize;

        NN_MIGRATION_DETAIL_TRACE("[SaveDataExporter] Sending: %llu of %zu\n", m_Cursor, m_DataInfo.codedSize);
        return copySize;
    }

    fs::ScopedAutoAbortDisabler antiAbort;
    size_t copySize;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(m_pImpl->Pull(&copySize, buffer, bufferSize));
    NN_RESULT_THROW_UNLESS(copySize > 0, ResultDataUnexpectedSize());
    m_Cursor += copySize;
    NN_SDK_ASSERT_EQUAL(m_DataInfo.codedSize - m_Cursor, static_cast<uint64_t>(m_pImpl->GetRestSize()));

    NN_MIGRATION_DETAIL_TRACE("[SaveDataExporter] Sending: %llu of %zu\n", m_Cursor, m_DataInfo.codedSize);
    return copySize;
}

size_t SaveDataExporter::GetRemainingSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pImpl);
    return static_cast<size_t>(static_cast<uint64_t>(m_DataInfo.codedSize) - m_Cursor);
}


// ---------------------------------------------------------------------------------------------
// SaveDataImporter

SaveDataImporter::SaveDataImporter(const DataInfo& dataInfo, ProgressMonitor& progressMonitor) NN_NOEXCEPT
    : m_Uid(account::InvalidUid)
    , m_DataInfo(dataInfo)
    , m_pClient(nullptr)
    , m_pResource(nullptr)
    , m_ProgressMonitor(progressMonitor)
    , m_Cursor(0u)
{
    m_ProgressMonitor.Start(dataInfo);
}

SaveDataImporter::SaveDataImporter(const account::Uid& uid, const DataInfo& dataInfo, ProgressMonitor& progressMonitor) NN_NOEXCEPT
    : m_Uid(uid)
    , m_DataInfo(dataInfo)
    , m_pClient(nullptr)
    , m_pResource(nullptr)
    , m_ProgressMonitor(progressMonitor)
    , m_Cursor(0u)
{
    m_ProgressMonitor.Start(dataInfo);
}

void SaveDataImporter::Initialize(IdcClientBase& client, ResourceStorage& workBuffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_pClient);
    NN_SDK_ASSERT(!m_pResource);

    m_pClient = &client;
    m_pResource = reinterpret_cast<Resource*>(&workBuffer);
}

fs::SaveDataId SaveDataImporter::GetSaveDataId() const NN_NOEXCEPT
{
    return m_DataInfo.id;
}

size_t SaveDataImporter::GetRemainingSize() const NN_NOEXCEPT
{
    return static_cast<size_t>(static_cast<uint64_t>(m_DataInfo.codedSize) - m_Cursor);
}

Result SaveDataImporter::Update(const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pClient);
    NN_MIGRATION_DETAIL_TRACE("[SaveDataImporter] Receiving: %zu of %zu\n", m_Cursor + dataSize, m_DataInfo.codedSize);

    if (m_Cursor < fs::SaveDataExporter::InitialDataSize)
    {
        auto copySize = std::min(dataSize, static_cast<size_t>(fs::SaveDataExporter::InitialDataSize - m_Cursor));
        std::memcpy(m_pResource->initialData, data, copySize);
        m_Cursor += copySize;
        m_ProgressMonitor.Update(copySize);

        if (m_Cursor >= fs::SaveDataExporter::InitialDataSize)
        {
            NN_SDK_ASSERT(m_Cursor == fs::SaveDataExporter::InitialDataSize);
            NN_RESULT_DO(m_pClient->OpenSaveDataImporter(&m_pImpl, m_Uid, m_DataInfo, m_pResource->initialData, sizeof(m_pResource->initialData)));
        }

        auto dataSize1 = dataSize - copySize;
        if (dataSize1 > 0u)
        {
            auto data1 = reinterpret_cast<const void*>(reinterpret_cast<uintptr_t>(data) + copySize);
            NN_RESULT_DO(m_pImpl->Push(data1, dataSize1));
            m_Cursor += dataSize1;
            NN_SDK_ASSERT_EQUAL(m_DataInfo.codedSize - m_Cursor, m_pImpl->GetRestSize());
            m_ProgressMonitor.Update(dataSize1);
        }
        NN_RESULT_SUCCESS;
    }

    NN_SDK_ASSERT(m_pImpl);
    NN_RESULT_DO(m_pImpl->Push(data, dataSize));
    m_Cursor += dataSize;
    NN_SDK_ASSERT_EQUAL(m_DataInfo.codedSize - m_Cursor, m_pImpl->GetRestSize());
    m_ProgressMonitor.Update(dataSize);
    NN_RESULT_SUCCESS;
}

Result SaveDataImporter::Complete() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pImpl);
    NN_RESULT_DO(m_pImpl->Finalize());
    // NOTE: ここで電源断されると「セーブデータがあり、進捗がない」状態となる。
    // -> 再開時にセーブデータがあるかどうかを検査し、あれば進捗を進める (UserMigrationClient::EnsureLastMigration など)
    m_ProgressMonitor.End();
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::detail
