﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_MigrationList.h>

namespace nn { namespace migration { namespace detail {

#if defined(NN_SDK_BUILD_RELEASE)
inline void PrintFsSaveDataInfo(const fs::SaveDataInfo&) NN_NOEXCEPT
{
}
inline void PrintDataInfo(const DataInfo&) NN_NOEXCEPT
{
}

#else // defined(NN_SDK_BUILD_RELEASE)
inline void PrintFsSaveDataInfo(const fs::SaveDataInfo& info) NN_NOEXCEPT
{
    switch (info.saveDataType)
    {
    case fs::SaveDataType::System:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] System\n");
        NN_MIGRATION_DETAIL_TRACE("  - System save data Id: %016llx\n", info.systemSaveDataId);
        NN_MIGRATION_DETAIL_TRACE("  - UserId: %016llx_%016llx\n", info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
        NN_MIGRATION_DETAIL_TRACE("  - Id: %016llx\n", info.saveDataId);
        NN_MIGRATION_DETAIL_TRACE("  - Space: %02x\n", info.saveDataSpaceId);
        NN_MIGRATION_DETAIL_TRACE("  - Size: %lld bytes\n", info.saveDataSize);
        break;
    case fs::SaveDataType::Device:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] Device\n");
        NN_MIGRATION_DETAIL_TRACE("  - Application Id: %016llx\n", info.applicationId.value);
        NN_MIGRATION_DETAIL_TRACE("  - Id: %016llx\n", info.saveDataId);
        NN_MIGRATION_DETAIL_TRACE("  - Space: %02x\n", info.saveDataSpaceId);
        NN_MIGRATION_DETAIL_TRACE("  - Size: %lld bytes\n", info.saveDataSize);
        break;
    case fs::SaveDataType::Account:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] Account\n");
        NN_MIGRATION_DETAIL_TRACE("  - Application Id: %016llx\n", info.applicationId.value);
        NN_MIGRATION_DETAIL_TRACE("  - UserId: %016llx_%016llx\n", info.saveDataUserId._data[0], info.saveDataUserId._data[1]);
        NN_MIGRATION_DETAIL_TRACE("  - Id: %016llx\n", info.saveDataId);
        NN_MIGRATION_DETAIL_TRACE("  - Space: %02x\n", info.saveDataSpaceId);
        NN_MIGRATION_DETAIL_TRACE("  - Size: %lld bytes\n", info.saveDataSize);
        break;
    case fs::SaveDataType::Bcat:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] Bcat\n");
        break;
    case fs::SaveDataType::Cache:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] Cache\n");
        break;
    case fs::SaveDataType::Temporary:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] Temporary\n");
        break;
    default:
        NN_MIGRATION_DETAIL_TRACE("[fs::SaveDataInfo] (Unknown)\n");
        break;
    }
}

inline void PrintDataInfo(const DataInfo& dataInfo) NN_NOEXCEPT
{
    switch (dataInfo.type)
    {
    case DataInfo::Type_System:
        NN_MIGRATION_DETAIL_TRACE("[DataInfo] System\n");
        NN_MIGRATION_DETAIL_TRACE("  - System save data Id: %016llx\n", dataInfo.attribute.system.systemSaveDataId);
        NN_MIGRATION_DETAIL_TRACE("  - UserId: %016llx_%016llx\n", dataInfo.attribute.system.user._data[0], dataInfo.attribute.system.user._data[1]);
        NN_MIGRATION_DETAIL_TRACE("  - Id: %016llx\n", dataInfo.id);
        NN_MIGRATION_DETAIL_TRACE("  - Space: %02x\n", dataInfo.spaceId);
        NN_MIGRATION_DETAIL_TRACE("  - Size: %lld bytes (coded: %lld bytes)\n", dataInfo.rawSize, dataInfo.codedSize);
        break;
    case DataInfo::Type_User:
        NN_MIGRATION_DETAIL_TRACE("[DataInfo] User\n");
        NN_MIGRATION_DETAIL_TRACE("  - Application: %016llx (versionL %08x)\n", dataInfo.attribute.user.applicationId, dataInfo.attribute.user.applicationVersion);
        NN_MIGRATION_DETAIL_TRACE("  - UserId: %016llx_%016llx\n", dataInfo.attribute.user.user._data[0], dataInfo.attribute.user.user._data[1]);
        NN_MIGRATION_DETAIL_TRACE("  - Id: %016llx\n", dataInfo.id);
        NN_MIGRATION_DETAIL_TRACE("  - Space: %02x\n", dataInfo.spaceId);
        NN_MIGRATION_DETAIL_TRACE("  - Size: %lld bytes (coded: %llu bytes)\n", dataInfo.rawSize, dataInfo.codedSize);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
#endif

inline bool Match(const DataInfo& dataInfo, const fs::SaveDataInfo& fsInfo) NN_NOEXCEPT
{
    PrintDataInfo(dataInfo);
    PrintFsSaveDataInfo(fsInfo);

    switch (dataInfo.type)
    {
    case DataInfo::Type_System:
        if (!(fsInfo.saveDataType == fs::SaveDataType::System))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] Type is not fs::SaveDataType::System\n");
        }
        if (!(fsInfo.saveDataSpaceId == dataInfo.spaceId))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] SpaceId mismatch %02x (expected: %02x)\n", fsInfo.saveDataSpaceId, dataInfo.spaceId);
        }
        if (!(fsInfo.systemSaveDataId == dataInfo.attribute.system.systemSaveDataId))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] SystemSaveDataId mismatch %016llx (expected: %016llx)\n", fsInfo.systemSaveDataId, dataInfo.attribute.system.systemSaveDataId);
        }
        return true
            && fsInfo.saveDataType == fs::SaveDataType::System
            && fsInfo.saveDataSpaceId == dataInfo.spaceId
            && fsInfo.systemSaveDataId == dataInfo.attribute.system.systemSaveDataId;
    case DataInfo::Type_User:
        if (!(fsInfo.saveDataType == fs::SaveDataType::Account))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] Type is not fs::SaveDataType::Account\n");
        }
        if (!(fsInfo.saveDataSpaceId == dataInfo.spaceId))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] SpaceId mismatch %02x (expected: %02x)\n", fsInfo.saveDataSpaceId, dataInfo.spaceId);
        }
        if (!(fsInfo.applicationId == dataInfo.attribute.user.applicationId))
        {
            NN_MIGRATION_DETAIL_WARN("[SaveDataTransfer::match] ApplicationId mismatch %016llx (expected: %016llx)\n", fsInfo.applicationId.value, dataInfo.attribute.user.applicationId.value);
        }
        return true
            && fsInfo.saveDataType == fs::SaveDataType::Account
            && fsInfo.saveDataSpaceId == dataInfo.spaceId
            && fsInfo.applicationId == dataInfo.attribute.user.applicationId;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}} // ~namespace nn::migration::detail
